/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.blockmanagement;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.LinkedListMultimap;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockCollection;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfo;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManager;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeDescriptor;
import org.apache.hadoop.hdfs.server.namenode.FSClusterStats;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.Namesystem;
import org.apache.hadoop.net.NetworkTopology;
import org.apache.hadoop.net.Node;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;

public class TestBlockManager {
    private List<DatanodeDescriptor> nodes;
    private List<DatanodeDescriptor> rackA;
    private List<DatanodeDescriptor> rackB;
    private static final int NUM_TEST_ITERS = 30;
    private static final int BLOCK_SIZE = 65536;
    private Configuration conf;
    private FSNamesystem fsn;
    private BlockManager bm;

    @Before
    public void setupMockCluster() throws IOException {
        this.conf = new HdfsConfiguration();
        this.conf.set("net.topology.script.file.name", "need to set a dummy value here so it assumes a multi-rack cluster");
        this.fsn = (FSNamesystem)Mockito.mock(FSNamesystem.class);
        ((FSNamesystem)Mockito.doReturn((Object)true).when((Object)this.fsn)).hasWriteLock();
        this.bm = new BlockManager((Namesystem)this.fsn, (FSClusterStats)this.fsn, this.conf);
        this.nodes = ImmutableList.of((Object)DFSTestUtil.getDatanodeDescriptor("1.1.1.1", "/rackA"), (Object)DFSTestUtil.getDatanodeDescriptor("2.2.2.2", "/rackA"), (Object)DFSTestUtil.getDatanodeDescriptor("3.3.3.3", "/rackA"), (Object)DFSTestUtil.getDatanodeDescriptor("4.4.4.4", "/rackB"), (Object)DFSTestUtil.getDatanodeDescriptor("5.5.5.5", "/rackB"), (Object)DFSTestUtil.getDatanodeDescriptor("6.6.6.6", "/rackB"));
        this.rackA = this.nodes.subList(0, 3);
        this.rackB = this.nodes.subList(3, 6);
    }

    private void addNodes(Iterable<DatanodeDescriptor> nodesToAdd) {
        NetworkTopology cluster = this.bm.getDatanodeManager().getNetworkTopology();
        for (DatanodeDescriptor dn : nodesToAdd) {
            cluster.add((Node)dn);
            dn.updateHeartbeat((long)(2 * HdfsConstants.MIN_BLOCKS_FOR_WRITE * 65536), 0L, (long)(2 * HdfsConstants.MIN_BLOCKS_FOR_WRITE * 65536), 0L, 0, 0);
            this.bm.getDatanodeManager().checkIfClusterIsNowMultiRack(dn);
        }
    }

    private void removeNode(DatanodeDescriptor deadNode) {
        NetworkTopology cluster = this.bm.getDatanodeManager().getNetworkTopology();
        cluster.remove((Node)deadNode);
        this.bm.removeBlocksAssociatedTo(deadNode);
    }

    @Test
    public void testBasicReplication() throws Exception {
        this.addNodes(this.nodes);
        for (int i = 0; i < 30; ++i) {
            this.doBasicTest(i);
        }
    }

    private void doBasicTest(int testIndex) {
        List<DatanodeDescriptor> origNodes = this.getNodes(0, 1);
        BlockInfo blockInfo = this.addBlockOnNodes(testIndex, origNodes);
        DatanodeDescriptor[] pipeline = this.scheduleSingleReplication((Block)blockInfo);
        Assert.assertEquals((long)2L, (long)pipeline.length);
        Assert.assertTrue((String)("Source of replication should be one of the nodes the block was on. Was: " + pipeline[0]), (boolean)origNodes.contains(pipeline[0]));
        Assert.assertTrue((String)("Destination of replication should be on the other rack. Was: " + pipeline[1]), (boolean)this.rackB.contains(pipeline[1]));
    }

    @Test
    public void testTwoOfThreeNodesDecommissioned() throws Exception {
        this.addNodes(this.nodes);
        for (int i = 0; i < 30; ++i) {
            this.doTestTwoOfThreeNodesDecommissioned(i);
        }
    }

    private void doTestTwoOfThreeNodesDecommissioned(int testIndex) throws Exception {
        List<DatanodeDescriptor> origNodes = this.getNodes(0, 1, 3);
        BlockInfo blockInfo = this.addBlockOnNodes(testIndex, origNodes);
        List<DatanodeDescriptor> decomNodes = this.startDecommission(0, 1);
        Object[] pipeline = this.scheduleSingleReplication((Block)blockInfo);
        Assert.assertTrue((String)("Source of replication should be one of the nodes the block was on. Was: " + pipeline[0]), (boolean)origNodes.contains(pipeline[0]));
        Assert.assertEquals((String)"Should have three targets", (long)3L, (long)pipeline.length);
        boolean foundOneOnRackA = false;
        for (int i = 1; i < pipeline.length; ++i) {
            DatanodeDescriptor target = pipeline[i];
            if (this.rackA.contains(target)) {
                foundOneOnRackA = true;
            }
            Assert.assertFalse((boolean)decomNodes.contains(target));
            Assert.assertFalse((boolean)origNodes.contains(target));
        }
        Assert.assertTrue((String)("Should have at least one target on rack A. Pipeline: " + Joiner.on((String)",").join(pipeline)), (boolean)foundOneOnRackA);
    }

    @Test
    public void testAllNodesHoldingReplicasDecommissioned() throws Exception {
        this.addNodes(this.nodes);
        for (int i = 0; i < 30; ++i) {
            this.doTestAllNodesHoldingReplicasDecommissioned(i);
        }
    }

    private void doTestAllNodesHoldingReplicasDecommissioned(int testIndex) throws Exception {
        List<DatanodeDescriptor> origNodes = this.getNodes(0, 1, 3);
        BlockInfo blockInfo = this.addBlockOnNodes(testIndex, origNodes);
        List<DatanodeDescriptor> decomNodes = this.startDecommission(0, 1, 3);
        Object[] pipeline = this.scheduleSingleReplication((Block)blockInfo);
        Assert.assertTrue((String)("Source of replication should be one of the nodes the block was on. Was: " + pipeline[0]), (boolean)origNodes.contains(pipeline[0]));
        Assert.assertEquals((String)"Should have three targets", (long)4L, (long)pipeline.length);
        boolean foundOneOnRackA = false;
        boolean foundOneOnRackB = false;
        for (int i = 1; i < pipeline.length; ++i) {
            DatanodeDescriptor target = pipeline[i];
            if (this.rackA.contains(target)) {
                foundOneOnRackA = true;
            } else if (this.rackB.contains(target)) {
                foundOneOnRackB = true;
            }
            Assert.assertFalse((boolean)decomNodes.contains(target));
            Assert.assertFalse((boolean)origNodes.contains(target));
        }
        Assert.assertTrue((String)("Should have at least one target on rack A. Pipeline: " + Joiner.on((String)",").join(pipeline)), (boolean)foundOneOnRackA);
        Assert.assertTrue((String)("Should have at least one target on rack B. Pipeline: " + Joiner.on((String)",").join(pipeline)), (boolean)foundOneOnRackB);
    }

    @Test
    public void testOneOfTwoRacksDecommissioned() throws Exception {
        this.addNodes(this.nodes);
        for (int i = 0; i < 30; ++i) {
            this.doTestOneOfTwoRacksDecommissioned(i);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void doTestOneOfTwoRacksDecommissioned(int testIndex) throws Exception {
        List<DatanodeDescriptor> origNodes = this.getNodes(0, 1, 3);
        BlockInfo blockInfo = this.addBlockOnNodes(testIndex, origNodes);
        List<DatanodeDescriptor> decomNodes = this.startDecommission(0, 1, 2);
        Object[] pipeline = this.scheduleSingleReplication((Block)blockInfo);
        Assert.assertTrue((String)("Source of replication should be one of the nodes the block was on. Was: " + pipeline[0]), (boolean)origNodes.contains(pipeline[0]));
        Assert.assertEquals((String)"Should have three targets", (long)3L, (long)pipeline.length);
        boolean foundOneOnRackB = false;
        for (int i = 1; i < pipeline.length; ++i) {
            DatanodeDescriptor target = pipeline[i];
            if (this.rackB.contains(target)) {
                foundOneOnRackB = true;
            }
            Assert.assertFalse((boolean)decomNodes.contains(target));
            Assert.assertFalse((boolean)origNodes.contains(target));
        }
        Assert.assertTrue((String)("Should have at least one target on rack B. Pipeline: " + Joiner.on((String)",").join(pipeline)), (boolean)foundOneOnRackB);
        this.fulfillPipeline(blockInfo, (DatanodeDescriptor[])pipeline);
        DatanodeDescriptor rackCNode = DFSTestUtil.getDatanodeDescriptor("7.7.7.7", "/rackC");
        this.addNodes((Iterable<DatanodeDescriptor>)ImmutableList.of((Object)rackCNode));
        try {
            DatanodeDescriptor[] pipeline2 = this.scheduleSingleReplication((Block)blockInfo);
            Assert.assertEquals((long)2L, (long)pipeline2.length);
            Assert.assertEquals((Object)rackCNode, (Object)pipeline2[1]);
        }
        finally {
            this.removeNode(rackCNode);
        }
    }

    @Test
    public void testSufficientlyReplBlocksUsesNewRack() throws Exception {
        this.addNodes(this.nodes);
        for (int i = 0; i < 30; ++i) {
            this.doTestSufficientlyReplBlocksUsesNewRack(i);
        }
    }

    private void doTestSufficientlyReplBlocksUsesNewRack(int testIndex) {
        List<DatanodeDescriptor> origNodes = this.rackA;
        BlockInfo blockInfo = this.addBlockOnNodes(testIndex, origNodes);
        DatanodeDescriptor[] pipeline = this.scheduleSingleReplication((Block)blockInfo);
        Assert.assertEquals((long)2L, (long)pipeline.length);
        Assert.assertTrue((String)("Source of replication should be one of the nodes the block was on. Was: " + pipeline[0]), (boolean)origNodes.contains(pipeline[0]));
        Assert.assertTrue((String)("Destination of replication should be on the other rack. Was: " + pipeline[1]), (boolean)this.rackB.contains(pipeline[1]));
    }

    @Test
    public void testBlocksAreNotUnderreplicatedInSingleRack() throws Exception {
        ImmutableList nodes = ImmutableList.of((Object)DFSTestUtil.getDatanodeDescriptor("1.1.1.1", "/rackA"), (Object)DFSTestUtil.getDatanodeDescriptor("2.2.2.2", "/rackA"), (Object)DFSTestUtil.getDatanodeDescriptor("3.3.3.3", "/rackA"), (Object)DFSTestUtil.getDatanodeDescriptor("4.4.4.4", "/rackA"), (Object)DFSTestUtil.getDatanodeDescriptor("5.5.5.5", "/rackA"), (Object)DFSTestUtil.getDatanodeDescriptor("6.6.6.6", "/rackA"));
        this.addNodes((Iterable<DatanodeDescriptor>)nodes);
        List<DatanodeDescriptor> origNodes = nodes.subList(0, 3);
        for (int i = 0; i < 30; ++i) {
            this.doTestSingleRackClusterIsSufficientlyReplicated(i, origNodes);
        }
    }

    private void doTestSingleRackClusterIsSufficientlyReplicated(int testIndex, List<DatanodeDescriptor> origNodes) throws Exception {
        Assert.assertEquals((long)0L, (long)this.bm.numOfUnderReplicatedBlocks());
        this.addBlockOnNodes(testIndex, origNodes);
        this.bm.processMisReplicatedBlocks();
        Assert.assertEquals((long)0L, (long)this.bm.numOfUnderReplicatedBlocks());
    }

    private void fulfillPipeline(BlockInfo blockInfo, DatanodeDescriptor[] pipeline) throws IOException {
        for (int i = 1; i < pipeline.length; ++i) {
            this.bm.addBlock(pipeline[i], (Block)blockInfo, null);
        }
    }

    private BlockInfo blockOnNodes(long blkId, List<DatanodeDescriptor> nodes) {
        Block block = new Block(blkId);
        BlockInfo blockInfo = new BlockInfo(block, 3);
        for (DatanodeDescriptor dn : nodes) {
            blockInfo.addNode(dn);
        }
        return blockInfo;
    }

    private List<DatanodeDescriptor> getNodes(int ... indexes) {
        ArrayList ret = Lists.newArrayList();
        for (int idx : indexes) {
            ret.add(this.nodes.get(idx));
        }
        return ret;
    }

    private List<DatanodeDescriptor> startDecommission(int ... indexes) {
        List<DatanodeDescriptor> nodes = this.getNodes(indexes);
        for (DatanodeDescriptor node : nodes) {
            node.startDecommission();
        }
        return nodes;
    }

    private BlockInfo addBlockOnNodes(long blockId, List<DatanodeDescriptor> nodes) {
        INodeFile iNode = (INodeFile)Mockito.mock(INodeFile.class);
        ((INodeFile)Mockito.doReturn((Object)3).when((Object)iNode)).getReplication();
        BlockInfo blockInfo = this.blockOnNodes(blockId, nodes);
        this.bm.blocksMap.addBlockCollection(blockInfo, (BlockCollection)iNode);
        return blockInfo;
    }

    private DatanodeDescriptor[] scheduleSingleReplication(Block block) {
        ArrayList<Block> list_p1 = new ArrayList<Block>();
        list_p1.add(block);
        ArrayList list_all = new ArrayList();
        list_all.add(new ArrayList());
        list_all.add(list_p1);
        Assert.assertEquals((String)"Block not initially pending replication", (long)0L, (long)this.bm.pendingReplications.getNumReplicas(block));
        Assert.assertEquals((String)"computeReplicationWork should indicate replication is needed", (long)1L, (long)this.bm.computeReplicationWorkForBlocks(list_all));
        Assert.assertTrue((String)"replication is pending after work is computed", (this.bm.pendingReplications.getNumReplicas(block) > 0 ? 1 : 0) != 0);
        LinkedListMultimap<DatanodeDescriptor, DatanodeDescriptor.BlockTargetPair> repls = this.getAllPendingReplications();
        Assert.assertEquals((long)1L, (long)repls.size());
        Map.Entry repl = (Map.Entry)repls.entries().iterator().next();
        DatanodeDescriptor[] targets = ((DatanodeDescriptor.BlockTargetPair)repl.getValue()).targets;
        DatanodeDescriptor[] pipeline = new DatanodeDescriptor[1 + targets.length];
        pipeline[0] = (DatanodeDescriptor)repl.getKey();
        System.arraycopy(targets, 0, pipeline, 1, targets.length);
        return pipeline;
    }

    private LinkedListMultimap<DatanodeDescriptor, DatanodeDescriptor.BlockTargetPair> getAllPendingReplications() {
        LinkedListMultimap repls = LinkedListMultimap.create();
        for (DatanodeDescriptor dn : this.nodes) {
            List thisRepls = dn.getReplicationCommand(10);
            if (thisRepls == null) continue;
            repls.putAll((Object)dn, (Iterable)thisRepls);
        }
        return repls;
    }
}

