/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.io.Closeable;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Options;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.security.token.delegation.DelegationTokenIdentifier;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.common.Storage;
import org.apache.hadoop.hdfs.server.namenode.BackupJournalManager;
import org.apache.hadoop.hdfs.server.namenode.EditLogInputStream;
import org.apache.hadoop.hdfs.server.namenode.EditLogOutputStream;
import org.apache.hadoop.hdfs.server.namenode.FSEditLogOp;
import org.apache.hadoop.hdfs.server.namenode.FSEditLogOpCodes;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.FileJournalManager;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.INodeFileUnderConstruction;
import org.apache.hadoop.hdfs.server.namenode.INodeSymlink;
import org.apache.hadoop.hdfs.server.namenode.JournalManager;
import org.apache.hadoop.hdfs.server.namenode.JournalSet;
import org.apache.hadoop.hdfs.server.namenode.LogsPurgeable;
import org.apache.hadoop.hdfs.server.namenode.MetaRecoveryContext;
import org.apache.hadoop.hdfs.server.namenode.NNStorage;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.metrics.NameNodeMetrics;
import org.apache.hadoop.hdfs.server.protocol.NamenodeRegistration;
import org.apache.hadoop.hdfs.server.protocol.NamespaceInfo;
import org.apache.hadoop.hdfs.server.protocol.RemoteEditLogManifest;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.security.token.delegation.DelegationKey;
import org.apache.hadoop.util.ExitUtil;
import org.apache.hadoop.util.Time;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class FSEditLog
implements LogsPurgeable {
    static final Log LOG = LogFactory.getLog(FSEditLog.class);
    private State state = State.UNINITIALIZED;
    private JournalSet journalSet = null;
    private EditLogOutputStream editLogStream = null;
    private long txid = 0L;
    private long synctxid = 0L;
    private long curSegmentTxId = -12345L;
    private long lastPrintTime;
    private volatile boolean isSyncRunning;
    private volatile boolean isAutoSyncScheduled = false;
    private long numTransactions;
    private long numTransactionsBatchedInSync;
    private long totalTimeTransactions;
    private NameNodeMetrics metrics;
    private NNStorage storage;
    private Configuration conf;
    private List<URI> editsDirs;
    private ThreadLocal<FSEditLogOp.OpInstanceCache> cache = new ThreadLocal<FSEditLogOp.OpInstanceCache>(){

        @Override
        protected FSEditLogOp.OpInstanceCache initialValue() {
            return new FSEditLogOp.OpInstanceCache();
        }
    };
    private List<URI> sharedEditsDirs;
    private static final ThreadLocal<TransactionId> myTransactionId = new ThreadLocal<TransactionId>(){

        @Override
        protected synchronized TransactionId initialValue() {
            return new TransactionId(Long.MAX_VALUE);
        }
    };

    FSEditLog(Configuration conf, NNStorage storage, List<URI> editsDirs) {
        this.init(conf, storage, editsDirs);
    }

    private void init(Configuration conf, NNStorage storage, List<URI> editsDirs) {
        this.isSyncRunning = false;
        this.conf = conf;
        this.storage = storage;
        this.metrics = NameNode.getNameNodeMetrics();
        this.lastPrintTime = Time.now();
        this.editsDirs = Lists.newArrayList(editsDirs);
        this.sharedEditsDirs = FSNamesystem.getSharedEditsDirs(conf);
    }

    public synchronized void initJournalsForWrite() {
        Preconditions.checkState((this.state == State.UNINITIALIZED || this.state == State.CLOSED ? 1 : 0) != 0, (String)"Unexpected state: %s", (Object[])new Object[]{this.state});
        this.initJournals(this.editsDirs);
        this.state = State.BETWEEN_LOG_SEGMENTS;
    }

    public synchronized void initSharedJournalsForRead() {
        if (this.state == State.OPEN_FOR_READING) {
            LOG.warn((Object)"Initializing shared journals for READ, already open for READ", (Throwable)new Exception());
            return;
        }
        Preconditions.checkState((this.state == State.UNINITIALIZED || this.state == State.CLOSED ? 1 : 0) != 0);
        this.initJournals(this.sharedEditsDirs);
        this.state = State.OPEN_FOR_READING;
    }

    private synchronized void initJournals(List<URI> dirs) {
        int minimumRedundantJournals = this.conf.getInt("dfs.namenode.edits.dir.minimum", 1);
        this.journalSet = new JournalSet(minimumRedundantJournals);
        for (URI u : dirs) {
            boolean required = FSNamesystem.getRequiredNamespaceEditsDirs(this.conf).contains(u);
            if (u.getScheme().equals("file")) {
                Storage.StorageDirectory sd = this.storage.getStorageDirectory(u);
                if (sd == null) continue;
                this.journalSet.add(new FileJournalManager(sd, this.storage), required);
                continue;
            }
            this.journalSet.add(this.createJournal(u), required);
        }
        if (this.journalSet.isEmpty()) {
            LOG.error((Object)"No edits directories configured!");
        }
    }

    Collection<URI> getEditURIs() {
        return this.editsDirs;
    }

    synchronized void openForWrite() throws IOException {
        Preconditions.checkState((this.state == State.BETWEEN_LOG_SEGMENTS ? 1 : 0) != 0, (String)"Bad state: %s", (Object[])new Object[]{this.state});
        long segmentTxId = this.getLastWrittenTxId() + 1L;
        ArrayList<EditLogInputStream> streams = new ArrayList<EditLogInputStream>();
        this.journalSet.selectInputStreams(streams, segmentTxId, true);
        if (!streams.isEmpty()) {
            String error = String.format("Cannot start writing at txid %s when there is a stream available for read: %s", segmentTxId, streams.get(0));
            IOUtils.cleanup((Log)LOG, (Closeable[])streams.toArray(new EditLogInputStream[0]));
            throw new IllegalStateException(error);
        }
        this.startLogSegment(segmentTxId, true);
        assert (this.state == State.IN_SEGMENT) : "Bad state: " + (Object)((Object)this.state);
    }

    synchronized boolean isOpenForWrite() {
        return this.state == State.IN_SEGMENT || this.state == State.BETWEEN_LOG_SEGMENTS;
    }

    synchronized boolean isSegmentOpen() {
        return this.state == State.IN_SEGMENT;
    }

    public synchronized boolean isOpenForRead() {
        return this.state == State.OPEN_FOR_READING;
    }

    synchronized void close() {
        if (this.state == State.CLOSED) {
            LOG.debug((Object)"Closing log when already closed");
            return;
        }
        if (this.state == State.IN_SEGMENT) {
            assert (this.editLogStream != null);
            this.waitForSyncToFinish();
            this.endCurrentLogSegment(true);
        }
        if (this.journalSet != null && !this.journalSet.isEmpty()) {
            try {
                this.journalSet.close();
            }
            catch (IOException ioe) {
                LOG.warn((Object)"Error closing journalSet", (Throwable)ioe);
            }
        }
        this.state = State.CLOSED;
    }

    synchronized void formatNonFileJournals(NamespaceInfo nsInfo) throws IOException {
        Preconditions.checkState((this.state == State.BETWEEN_LOG_SEGMENTS ? 1 : 0) != 0, (String)"Bad state: %s", (Object[])new Object[]{this.state});
        for (JournalManager jm : this.journalSet.getJournalManagers()) {
            if (jm instanceof FileJournalManager) continue;
            jm.format(nsInfo);
        }
    }

    synchronized List<Storage.FormatConfirmable> getFormatConfirmables() {
        Preconditions.checkState((this.state == State.BETWEEN_LOG_SEGMENTS ? 1 : 0) != 0, (String)"Bad state: %s", (Object[])new Object[]{this.state});
        ArrayList ret = Lists.newArrayList();
        for (JournalManager jm : this.journalSet.getJournalManagers()) {
            if (jm instanceof FileJournalManager) continue;
            ret.add(jm);
        }
        return ret;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void logEdit(FSEditLogOp op) {
        FSEditLog fSEditLog = this;
        synchronized (fSEditLog) {
            assert (this.isOpenForWrite()) : "bad state: " + (Object)((Object)this.state);
            this.waitIfAutoSyncScheduled();
            long start = this.beginTransaction();
            op.setTransactionId(this.txid);
            try {
                this.editLogStream.write(op);
            }
            catch (IOException ex) {
                // empty catch block
            }
            this.endTransaction(start);
            if (!this.shouldForceSync()) {
                return;
            }
            this.isAutoSyncScheduled = true;
        }
        this.logSync();
    }

    synchronized void waitIfAutoSyncScheduled() {
        try {
            while (this.isAutoSyncScheduled) {
                this.wait(1000L);
            }
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    synchronized void doneWithAutoSyncScheduling() {
        if (this.isAutoSyncScheduled) {
            this.isAutoSyncScheduled = false;
            this.notifyAll();
        }
    }

    private boolean shouldForceSync() {
        return this.editLogStream.shouldForceSync();
    }

    private long beginTransaction() {
        assert (Thread.holdsLock(this));
        ++this.txid;
        TransactionId id = myTransactionId.get();
        id.txid = this.txid;
        return Time.now();
    }

    private void endTransaction(long start) {
        assert (Thread.holdsLock(this));
        long end = Time.now();
        ++this.numTransactions;
        this.totalTimeTransactions += end - start;
        if (this.metrics != null) {
            this.metrics.addTransaction(end - start);
        }
    }

    public synchronized long getLastWrittenTxId() {
        return this.txid;
    }

    synchronized long getCurSegmentTxId() {
        Preconditions.checkState((boolean)this.isSegmentOpen(), (String)"Bad state: %s", (Object[])new Object[]{this.state});
        return this.curSegmentTxId;
    }

    synchronized void setNextTxId(long nextTxId) {
        Preconditions.checkArgument((this.synctxid <= this.txid && nextTxId >= this.txid ? 1 : 0) != 0, (String)"May not decrease txid. synctxid=%s txid=%s nextTxId=%s", (Object[])new Object[]{this.synctxid, this.txid, nextTxId});
        this.txid = nextTxId - 1L;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void logSyncAll() {
        FSEditLog fSEditLog = this;
        synchronized (fSEditLog) {
            TransactionId id = myTransactionId.get();
            id.txid = this.txid;
        }
        this.logSync();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public void logSync() {
        long syncStart = 0L;
        long mytxid = FSEditLog.myTransactionId.get().txid;
        boolean sync = false;
        try {
            EditLogOutputStream logStream = null;
            FSEditLog fSEditLog = this;
            synchronized (fSEditLog) {
                try {
                    this.printStatistics(false);
                    while (mytxid > this.synctxid && this.isSyncRunning) {
                        try {
                            this.wait(1000L);
                        }
                        catch (InterruptedException ie) {}
                    }
                    if (mytxid <= this.synctxid) {
                        ++this.numTransactionsBatchedInSync;
                        if (this.metrics == null) return;
                        this.metrics.incrTransactionsBatchedInSync();
                        return;
                    }
                    syncStart = this.txid;
                    this.isSyncRunning = true;
                    sync = true;
                    try {
                        if (this.journalSet.isEmpty()) {
                            throw new IOException("No journals available to flush");
                        }
                        this.editLogStream.setReadyToFlush();
                    }
                    catch (IOException e) {
                        String msg = "Could not sync enough journals to persistent storage due to " + e.getMessage() + ". " + "Unsynced transactions: " + (this.txid - this.synctxid);
                        LOG.fatal((Object)msg, (Throwable)new Exception());
                        ExitUtil.terminate((int)1, (String)msg);
                    }
                }
                finally {
                    this.doneWithAutoSyncScheduling();
                }
                logStream = this.editLogStream;
            }
            long start = Time.now();
            try {
                if (logStream != null) {
                    logStream.flush();
                }
            }
            catch (IOException ex) {
                FSEditLog throwable = this;
                synchronized (throwable) {
                    String msg = "Could not sync enough journals to persistent storage. Unsynced transactions: " + (this.txid - this.synctxid);
                    LOG.fatal((Object)msg, (Throwable)new Exception());
                    ExitUtil.terminate((int)1, (String)msg);
                }
            }
            long elapsed = Time.now() - start;
            if (this.metrics == null) return;
            this.metrics.addSync(elapsed);
            return;
        }
        finally {
            FSEditLog ie = this;
            synchronized (ie) {
                if (sync) {
                    this.synctxid = syncStart;
                    this.isSyncRunning = false;
                }
                this.notifyAll();
            }
        }
    }

    private void printStatistics(boolean force) {
        long now = Time.now();
        if (this.lastPrintTime + 60000L > now && !force) {
            return;
        }
        this.lastPrintTime = now;
        StringBuilder buf = new StringBuilder();
        buf.append("Number of transactions: ");
        buf.append(this.numTransactions);
        buf.append(" Total time for transactions(ms): ");
        buf.append(this.totalTimeTransactions);
        buf.append("Number of transactions batched in Syncs: ");
        buf.append(this.numTransactionsBatchedInSync);
        buf.append(" Number of syncs: ");
        buf.append(this.editLogStream.getNumSync());
        buf.append(" SyncTimes(ms): ");
        buf.append(this.journalSet.getSyncTimes());
        LOG.info((Object)buf);
    }

    public void logOpenFile(String path, INodeFileUnderConstruction newNode) {
        FSEditLogOp.AddOp op = (FSEditLogOp.AddOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)FSEditLogOp.AddOp.getInstance(this.cache.get()).setPath(path)).setReplication(newNode.getReplication())).setModificationTime(newNode.getModificationTime())).setAccessTime(newNode.getAccessTime())).setBlockSize(newNode.getPreferredBlockSize())).setBlocks(newNode.getBlocks())).setPermissionStatus(newNode.getPermissionStatus())).setClientName(newNode.getClientName())).setClientMachine(newNode.getClientMachine());
        this.logEdit(op);
    }

    public void logCloseFile(String path, INodeFile newNode) {
        FSEditLogOp.CloseOp op = (FSEditLogOp.CloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)FSEditLogOp.CloseOp.getInstance(this.cache.get()).setPath(path)).setReplication(newNode.getReplication())).setModificationTime(newNode.getModificationTime())).setAccessTime(newNode.getAccessTime())).setBlockSize(newNode.getPreferredBlockSize())).setBlocks(newNode.getBlocks())).setPermissionStatus(newNode.getPermissionStatus());
        this.logEdit(op);
    }

    public void logUpdateBlocks(String path, INodeFileUnderConstruction file) {
        FSEditLogOp.UpdateBlocksOp op = FSEditLogOp.UpdateBlocksOp.getInstance(this.cache.get()).setPath(path).setBlocks(file.getBlocks());
        this.logEdit(op);
    }

    public void logMkDir(String path, INode newNode) {
        FSEditLogOp.MkdirOp op = FSEditLogOp.MkdirOp.getInstance(this.cache.get()).setPath(path).setTimestamp(newNode.getModificationTime()).setPermissionStatus(newNode.getPermissionStatus());
        this.logEdit(op);
    }

    void logRename(String src, String dst, long timestamp) {
        FSEditLogOp.RenameOldOp op = FSEditLogOp.RenameOldOp.getInstance(this.cache.get()).setSource(src).setDestination(dst).setTimestamp(timestamp);
        this.logEdit(op);
    }

    void logRename(String src, String dst, long timestamp, Options.Rename ... options) {
        FSEditLogOp.RenameOp op = FSEditLogOp.RenameOp.getInstance(this.cache.get()).setSource(src).setDestination(dst).setTimestamp(timestamp).setOptions(options);
        this.logEdit(op);
    }

    void logSetReplication(String src, short replication) {
        FSEditLogOp.SetReplicationOp op = FSEditLogOp.SetReplicationOp.getInstance(this.cache.get()).setPath(src).setReplication(replication);
        this.logEdit(op);
    }

    void logSetQuota(String src, long nsQuota, long dsQuota) {
        FSEditLogOp.SetQuotaOp op = FSEditLogOp.SetQuotaOp.getInstance(this.cache.get()).setSource(src).setNSQuota(nsQuota).setDSQuota(dsQuota);
        this.logEdit(op);
    }

    void logSetPermissions(String src, FsPermission permissions) {
        FSEditLogOp.SetPermissionsOp op = FSEditLogOp.SetPermissionsOp.getInstance(this.cache.get()).setSource(src).setPermissions(permissions);
        this.logEdit(op);
    }

    void logSetOwner(String src, String username, String groupname) {
        FSEditLogOp.SetOwnerOp op = FSEditLogOp.SetOwnerOp.getInstance(this.cache.get()).setSource(src).setUser(username).setGroup(groupname);
        this.logEdit(op);
    }

    void logConcat(String trg, String[] srcs, long timestamp) {
        FSEditLogOp.ConcatDeleteOp op = FSEditLogOp.ConcatDeleteOp.getInstance(this.cache.get()).setTarget(trg).setSources(srcs).setTimestamp(timestamp);
        this.logEdit(op);
    }

    void logDelete(String src, long timestamp) {
        FSEditLogOp.DeleteOp op = FSEditLogOp.DeleteOp.getInstance(this.cache.get()).setPath(src).setTimestamp(timestamp);
        this.logEdit(op);
    }

    void logGenerationStamp(long genstamp) {
        FSEditLogOp.SetGenstampOp op = FSEditLogOp.SetGenstampOp.getInstance(this.cache.get()).setGenerationStamp(genstamp);
        this.logEdit(op);
    }

    void logTimes(String src, long mtime, long atime) {
        FSEditLogOp.TimesOp op = FSEditLogOp.TimesOp.getInstance(this.cache.get()).setPath(src).setModificationTime(mtime).setAccessTime(atime);
        this.logEdit(op);
    }

    void logSymlink(String path, String value, long mtime, long atime, INodeSymlink node) {
        FSEditLogOp.SymlinkOp op = FSEditLogOp.SymlinkOp.getInstance(this.cache.get()).setPath(path).setValue(value).setModificationTime(mtime).setAccessTime(atime).setPermissionStatus(node.getPermissionStatus());
        this.logEdit(op);
    }

    void logGetDelegationToken(DelegationTokenIdentifier id, long expiryTime) {
        FSEditLogOp.GetDelegationTokenOp op = FSEditLogOp.GetDelegationTokenOp.getInstance(this.cache.get()).setDelegationTokenIdentifier(id).setExpiryTime(expiryTime);
        this.logEdit(op);
    }

    void logRenewDelegationToken(DelegationTokenIdentifier id, long expiryTime) {
        FSEditLogOp.RenewDelegationTokenOp op = FSEditLogOp.RenewDelegationTokenOp.getInstance(this.cache.get()).setDelegationTokenIdentifier(id).setExpiryTime(expiryTime);
        this.logEdit(op);
    }

    void logCancelDelegationToken(DelegationTokenIdentifier id) {
        FSEditLogOp.CancelDelegationTokenOp op = FSEditLogOp.CancelDelegationTokenOp.getInstance(this.cache.get()).setDelegationTokenIdentifier(id);
        this.logEdit(op);
    }

    void logUpdateMasterKey(DelegationKey key) {
        FSEditLogOp.UpdateMasterKeyOp op = FSEditLogOp.UpdateMasterKeyOp.getInstance(this.cache.get()).setDelegationKey(key);
        this.logEdit(op);
    }

    void logReassignLease(String leaseHolder, String src, String newHolder) {
        FSEditLogOp.ReassignLeaseOp op = FSEditLogOp.ReassignLeaseOp.getInstance(this.cache.get()).setLeaseHolder(leaseHolder).setPath(src).setNewHolder(newHolder);
        this.logEdit(op);
    }

    synchronized List<JournalSet.JournalAndStream> getJournals() {
        return this.journalSet.getAllJournalStreams();
    }

    @VisibleForTesting
    public synchronized JournalSet getJournalSet() {
        return this.journalSet;
    }

    @VisibleForTesting
    void setMetricsForTests(NameNodeMetrics metrics) {
        this.metrics = metrics;
    }

    public synchronized RemoteEditLogManifest getEditLogManifest(long fromTxId) throws IOException {
        return this.journalSet.getEditLogManifest(fromTxId);
    }

    synchronized long rollEditLog() throws IOException {
        LOG.info((Object)"Rolling edit logs.");
        this.endCurrentLogSegment(true);
        long nextTxId = this.getLastWrittenTxId() + 1L;
        this.startLogSegment(nextTxId, true);
        assert (this.curSegmentTxId == nextTxId);
        return nextTxId;
    }

    synchronized void startLogSegment(long segmentTxId, boolean writeHeaderTxn) throws IOException {
        LOG.info((Object)("Starting log segment at " + segmentTxId));
        Preconditions.checkArgument((segmentTxId > 0L ? 1 : 0) != 0, (String)"Bad txid: %s", (Object[])new Object[]{segmentTxId});
        Preconditions.checkState((this.state == State.BETWEEN_LOG_SEGMENTS ? 1 : 0) != 0, (String)"Bad state: %s", (Object[])new Object[]{this.state});
        Preconditions.checkState((segmentTxId > this.curSegmentTxId ? 1 : 0) != 0, (Object)("Cannot start writing to log segment " + segmentTxId + " when previous log segment started at " + this.curSegmentTxId));
        Preconditions.checkArgument((segmentTxId == this.txid + 1L ? 1 : 0) != 0, (String)"Cannot start log segment at txid %s when next expected txid is %s", (Object[])new Object[]{segmentTxId, this.txid + 1L});
        this.numTransactionsBatchedInSync = 0L;
        this.totalTimeTransactions = 0L;
        this.numTransactions = 0L;
        this.storage.attemptRestoreRemovedStorage();
        try {
            this.editLogStream = this.journalSet.startLogSegment(segmentTxId);
        }
        catch (IOException ex) {
            throw new IOException("Unable to start log segment " + segmentTxId + ": too few journals successfully started.", ex);
        }
        this.curSegmentTxId = segmentTxId;
        this.state = State.IN_SEGMENT;
        if (writeHeaderTxn) {
            this.logEdit(FSEditLogOp.LogSegmentOp.getInstance(this.cache.get(), FSEditLogOpCodes.OP_START_LOG_SEGMENT));
            this.logSync();
        }
    }

    synchronized void endCurrentLogSegment(boolean writeEndTxn) {
        LOG.info((Object)("Ending log segment " + this.curSegmentTxId));
        Preconditions.checkState((boolean)this.isSegmentOpen(), (String)"Bad state: %s", (Object[])new Object[]{this.state});
        if (writeEndTxn) {
            this.logEdit(FSEditLogOp.LogSegmentOp.getInstance(this.cache.get(), FSEditLogOpCodes.OP_END_LOG_SEGMENT));
            this.logSync();
        }
        this.printStatistics(true);
        long lastTxId = this.getLastWrittenTxId();
        try {
            this.journalSet.finalizeLogSegment(this.curSegmentTxId, lastTxId);
            this.editLogStream = null;
        }
        catch (IOException e) {
            // empty catch block
        }
        this.state = State.BETWEEN_LOG_SEGMENTS;
    }

    synchronized void abortCurrentLogSegment() {
        try {
            if (this.editLogStream != null) {
                this.editLogStream.abort();
                this.editLogStream = null;
                this.state = State.BETWEEN_LOG_SEGMENTS;
            }
        }
        catch (IOException e) {
            LOG.warn((Object)"All journals failed to abort", (Throwable)e);
        }
    }

    @Override
    public synchronized void purgeLogsOlderThan(long minTxIdToKeep) {
        assert (this.curSegmentTxId == -12345L || minTxIdToKeep <= this.curSegmentTxId) : "cannot purge logs older than txid " + minTxIdToKeep + " when current segment starts at " + this.curSegmentTxId;
        if (minTxIdToKeep == 0L) {
            return;
        }
        try {
            this.journalSet.purgeLogsOlderThan(minTxIdToKeep);
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    synchronized void waitForSyncToFinish() {
        while (this.isSyncRunning) {
            try {
                this.wait(1000L);
            }
            catch (InterruptedException interruptedException) {}
        }
    }

    synchronized long getSyncTxId() {
        return this.synctxid;
    }

    synchronized void setOutputBufferCapacity(int size) {
        this.journalSet.setOutputBufferCapacity(size);
    }

    synchronized void registerBackupNode(NamenodeRegistration bnReg, NamenodeRegistration nnReg) throws IOException {
        if (bnReg.isRole(HdfsServerConstants.NamenodeRole.CHECKPOINT)) {
            return;
        }
        BackupJournalManager jas = this.findBackupJournal(bnReg);
        if (jas != null) {
            LOG.info((Object)("Backup node " + bnReg + " re-registers"));
            return;
        }
        LOG.info((Object)("Registering new backup node: " + bnReg));
        BackupJournalManager bjm = new BackupJournalManager(bnReg, nnReg);
        this.journalSet.add(bjm, true);
    }

    synchronized void releaseBackupStream(NamenodeRegistration registration) throws IOException {
        BackupJournalManager bjm = this.findBackupJournal(registration);
        if (bjm != null) {
            LOG.info((Object)("Removing backup journal " + bjm));
            this.journalSet.remove(bjm);
        }
    }

    private synchronized BackupJournalManager findBackupJournal(NamenodeRegistration bnReg) {
        for (JournalManager bjm : this.journalSet.getJournalManagers()) {
            if (!(bjm instanceof BackupJournalManager) || !((BackupJournalManager)bjm).matchesRegistration(bnReg)) continue;
            return (BackupJournalManager)bjm;
        }
        return null;
    }

    synchronized void logEdit(int length, byte[] data) {
        long start = this.beginTransaction();
        try {
            this.editLogStream.writeRaw(data, 0, length);
        }
        catch (IOException ex) {
            // empty catch block
        }
        this.endTransaction(start);
    }

    synchronized void recoverUnclosedStreams() {
        Preconditions.checkState((this.state == State.BETWEEN_LOG_SEGMENTS ? 1 : 0) != 0, (String)"May not recover segments - wrong state: %s", (Object[])new Object[]{this.state});
        try {
            this.journalSet.recoverUnfinalizedSegments();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public Collection<EditLogInputStream> selectInputStreams(long fromTxId, long toAtLeastTxId) throws IOException {
        return this.selectInputStreams(fromTxId, toAtLeastTxId, null, true);
    }

    public synchronized Collection<EditLogInputStream> selectInputStreams(long fromTxId, long toAtLeastTxId, MetaRecoveryContext recovery, boolean inProgressOk) throws IOException {
        ArrayList<EditLogInputStream> streams = new ArrayList<EditLogInputStream>();
        this.journalSet.selectInputStreams(streams, fromTxId, inProgressOk);
        try {
            this.checkForGaps(streams, fromTxId, toAtLeastTxId, inProgressOk);
        }
        catch (IOException e) {
            if (recovery != null) {
                LOG.error((Object)e);
            }
            FSEditLog.closeAllStreams(streams);
            throw e;
        }
        return streams;
    }

    private void checkForGaps(List<EditLogInputStream> streams, long fromTxId, long toAtLeastTxId, boolean inProgressOk) throws IOException {
        Iterator<EditLogInputStream> iter = streams.iterator();
        long txId = fromTxId;
        while (true) {
            EditLogInputStream elis;
            if (txId > toAtLeastTxId) {
                return;
            }
            if (!iter.hasNext() || (elis = iter.next()).getFirstTxId() > txId) break;
            long next = elis.getLastTxId();
            if (next == -12345L) {
                if (!inProgressOk) {
                    throw new RuntimeException("inProgressOk = false, but selectInputStreams returned an in-progress edit log input stream (" + elis + ")");
                }
                return;
            }
            txId = next + 1L;
        }
        throw new IOException(String.format("Gap in transactions. Expected to be able to read up until at least txid %d but unable to find any edit logs containing txid %d", toAtLeastTxId, txId));
    }

    static void closeAllStreams(Iterable<EditLogInputStream> streams) {
        for (EditLogInputStream s : streams) {
            IOUtils.closeStream((Closeable)s);
        }
    }

    static Class<? extends JournalManager> getJournalClass(Configuration conf, String uriScheme) {
        String key = "dfs.namenode.edits.journal-plugin." + uriScheme;
        Class clazz = null;
        try {
            clazz = conf.getClass(key, null, JournalManager.class);
        }
        catch (RuntimeException re) {
            throw new IllegalArgumentException("Invalid class specified for " + uriScheme, re);
        }
        if (clazz == null) {
            LOG.warn((Object)("No class configured for " + uriScheme + ", " + key + " is empty"));
            throw new IllegalArgumentException("No class configured for " + uriScheme);
        }
        return clazz;
    }

    private JournalManager createJournal(URI uri) {
        Class<? extends JournalManager> clazz = FSEditLog.getJournalClass(this.conf, uri.getScheme());
        try {
            Constructor<? extends JournalManager> cons = clazz.getConstructor(Configuration.class, URI.class, NamespaceInfo.class);
            return cons.newInstance(this.conf, uri, this.storage.getNamespaceInfo());
        }
        catch (Exception e) {
            throw new IllegalArgumentException("Unable to construct journal, " + uri, e);
        }
    }

    private static class TransactionId {
        public long txid;

        TransactionId(long value) {
            this.txid = value;
        }
    }

    private static enum State {
        UNINITIALIZED,
        BETWEEN_LOG_SEGMENTS,
        IN_SEGMENT,
        OPEN_FOR_READING,
        CLOSED;

    }
}

