/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.thrift;

import java.util.Arrays;
import java.util.Comparator;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.KeyspaceNotDefinedException;
import org.apache.cassandra.db.marshal.AbstractType;
import org.apache.cassandra.db.marshal.MarshalException;
import org.apache.cassandra.dht.IPartitioner;
import org.apache.cassandra.dht.RandomPartitioner;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.thrift.Column;
import org.apache.cassandra.thrift.ColumnOrSuperColumn;
import org.apache.cassandra.thrift.ColumnParent;
import org.apache.cassandra.thrift.ColumnPath;
import org.apache.cassandra.thrift.Deletion;
import org.apache.cassandra.thrift.InvalidRequestException;
import org.apache.cassandra.thrift.KeyRange;
import org.apache.cassandra.thrift.Mutation;
import org.apache.cassandra.thrift.SlicePredicate;
import org.apache.cassandra.thrift.SliceRange;
import org.apache.cassandra.thrift.ThriftGlue;
import org.apache.cassandra.utils.FBUtilities;

public class ThriftValidation {
    static void validateKey(String key) throws InvalidRequestException {
        if (key.isEmpty()) {
            throw new InvalidRequestException("Key may not be empty");
        }
        int utflen = FBUtilities.encodedUTF8Length(key);
        if (utflen > 65535) {
            throw new InvalidRequestException("Encoded key length of " + utflen + " is longer than maximum of 65535");
        }
    }

    private static void validateTable(String tablename) throws KeyspaceNotDefinedException {
        if (!DatabaseDescriptor.getTables().contains(tablename)) {
            throw new KeyspaceNotDefinedException("Keyspace " + tablename + " does not exist in this schema.");
        }
    }

    public static String validateColumnFamily(String tablename, String cfName) throws InvalidRequestException {
        if (cfName.isEmpty()) {
            throw new InvalidRequestException("non-empty columnfamily is required");
        }
        String cfType = DatabaseDescriptor.getColumnType(tablename, cfName);
        if (cfType == null) {
            throw new InvalidRequestException("unconfigured columnfamily " + cfName);
        }
        return cfType;
    }

    static void validateColumnPath(String tablename, ColumnPath column_path) throws InvalidRequestException {
        ThriftValidation.validateTable(tablename);
        String cfType = ThriftValidation.validateColumnFamily(tablename, column_path.column_family);
        if (cfType.equals("Standard")) {
            if (column_path.super_column != null) {
                throw new InvalidRequestException("supercolumn parameter is invalid for standard CF " + column_path.column_family);
            }
            if (column_path.column == null) {
                throw new InvalidRequestException("column parameter is not optional for standard CF " + column_path.column_family);
            }
        } else if (column_path.super_column == null) {
            throw new InvalidRequestException("supercolumn parameter is not optional for super CF " + column_path.column_family);
        }
        if (column_path.column != null) {
            ThriftValidation.validateColumns(tablename, column_path.column_family, column_path.super_column, Arrays.asList(new byte[][]{column_path.column}));
        }
        if (column_path.super_column != null) {
            ThriftValidation.validateColumns(tablename, column_path.column_family, null, Arrays.asList(new byte[][]{column_path.super_column}));
        }
    }

    static void validateColumnParent(String tablename, ColumnParent column_parent) throws InvalidRequestException {
        ThriftValidation.validateTable(tablename);
        String cfType = ThriftValidation.validateColumnFamily(tablename, column_parent.column_family);
        if (cfType.equals("Standard") && column_parent.super_column != null) {
            throw new InvalidRequestException("columnfamily alone is required for standard CF " + column_parent.column_family);
        }
        if (column_parent.super_column != null) {
            ThriftValidation.validateColumns(tablename, column_parent.column_family, null, Arrays.asList(new byte[][]{column_parent.super_column}));
        }
    }

    static void validateColumnPathOrParent(String tablename, ColumnPath column_path_or_parent) throws InvalidRequestException {
        ThriftValidation.validateTable(tablename);
        String cfType = ThriftValidation.validateColumnFamily(tablename, column_path_or_parent.column_family);
        if (cfType.equals("Standard") && column_path_or_parent.super_column != null) {
            throw new InvalidRequestException("supercolumn may not be specified for standard CF " + column_path_or_parent.column_family);
        }
        if (column_path_or_parent.column != null) {
            ThriftValidation.validateColumns(tablename, column_path_or_parent.column_family, column_path_or_parent.super_column, Arrays.asList(new byte[][]{column_path_or_parent.column}));
        }
        if (column_path_or_parent.super_column != null) {
            ThriftValidation.validateColumns(tablename, column_path_or_parent.column_family, null, Arrays.asList(new byte[][]{column_path_or_parent.super_column}));
        }
    }

    private static void validateColumns(String keyspace, String columnFamilyName, byte[] superColumnName, Iterable<byte[]> column_names) throws InvalidRequestException {
        if (superColumnName != null) {
            if (superColumnName.length > 65535) {
                throw new InvalidRequestException("supercolumn name length must not be greater than 65535");
            }
            if (superColumnName.length == 0) {
                throw new InvalidRequestException("supercolumn name must not be empty");
            }
            if (!DatabaseDescriptor.getColumnFamilyType(keyspace, columnFamilyName).equals("Super")) {
                throw new InvalidRequestException("supercolumn specified to ColumnFamily " + columnFamilyName + " containing normal columns");
            }
        }
        AbstractType comparator = ColumnFamily.getComparatorFor(keyspace, columnFamilyName, superColumnName);
        for (byte[] name : column_names) {
            if (name.length > 65535) {
                throw new InvalidRequestException("column name length must not be greater than 65535");
            }
            if (name.length == 0) {
                throw new InvalidRequestException("column name must not be empty");
            }
            try {
                comparator.validate(name);
            }
            catch (MarshalException e) {
                throw new InvalidRequestException(e.getMessage());
            }
        }
    }

    public static void validateColumns(String keyspace, ColumnParent column_parent, Iterable<byte[]> column_names) throws InvalidRequestException {
        ThriftValidation.validateColumns(keyspace, column_parent.column_family, column_parent.super_column, column_names);
    }

    public static void validateRange(String keyspace, ColumnParent column_parent, SliceRange range) throws InvalidRequestException {
        Comparator<byte[]> orderedComparator;
        AbstractType comparator = ColumnFamily.getComparatorFor(keyspace, column_parent.column_family, column_parent.super_column);
        try {
            comparator.validate(range.start);
            comparator.validate(range.finish);
        }
        catch (MarshalException e) {
            throw new InvalidRequestException(e.getMessage());
        }
        if (range.count < 0) {
            throw new InvalidRequestException("get_slice requires non-negative count");
        }
        Comparator<byte[]> comparator2 = orderedComparator = range.isReversed() ? comparator.getReverseComparator() : comparator;
        if (range.start.length > 0 && range.finish.length > 0 && orderedComparator.compare(range.start, range.finish) > 0) {
            throw new InvalidRequestException("range finish must come after start in the order of traversal");
        }
    }

    public static void validateColumnOrSuperColumn(String keyspace, String cfName, ColumnOrSuperColumn cosc) throws InvalidRequestException {
        if (cosc.column != null) {
            ThriftValidation.validateColumnPath(keyspace, ThriftGlue.createColumnPath(cfName, null, cosc.column.name));
        }
        if (cosc.super_column != null) {
            for (Column c : cosc.super_column.columns) {
                ThriftValidation.validateColumnPath(keyspace, ThriftGlue.createColumnPath(cfName, cosc.super_column.name, c.name));
            }
        }
        if (cosc.column == null && cosc.super_column == null) {
            throw new InvalidRequestException("ColumnOrSuperColumn must have one or both of Column or SuperColumn");
        }
    }

    public static void validateMutation(String keyspace, String cfName, Mutation mut) throws InvalidRequestException {
        ColumnOrSuperColumn cosc = mut.column_or_supercolumn;
        Deletion del = mut.deletion;
        if (cosc != null && del != null) {
            throw new InvalidRequestException("Mutation may have either a ColumnOrSuperColumn or a Deletion, but not both");
        }
        if (cosc != null) {
            ThriftValidation.validateColumnOrSuperColumn(keyspace, cfName, cosc);
        } else if (del != null) {
            ThriftValidation.validateDeletion(keyspace, cfName, del);
        } else {
            throw new InvalidRequestException("Mutation must have one ColumnOrSuperColumn or one Deletion");
        }
    }

    public static void validateDeletion(String keyspace, String cfName, Deletion del) throws InvalidRequestException {
        if (del.super_column == null && del.predicate == null) {
            throw new InvalidRequestException("A Deletion must have a SuperColumn, a SlicePredicate or both.");
        }
        if (del.predicate != null) {
            ThriftValidation.validateSlicePredicate(keyspace, cfName, del.super_column, del.predicate);
            if (del.predicate.slice_range != null) {
                throw new InvalidRequestException("Deletion does not yet support SliceRange predicates.");
            }
        }
    }

    public static void validateSlicePredicate(String keyspace, String cfName, byte[] scName, SlicePredicate predicate) throws InvalidRequestException {
        if (predicate.column_names == null && predicate.slice_range == null) {
            throw new InvalidRequestException("A SlicePredicate must be given a list of Columns, a SliceRange, or both");
        }
        if (predicate.slice_range != null) {
            ThriftValidation.validateRange(keyspace, ThriftGlue.createColumnParent(cfName, scName), predicate.slice_range);
        }
        if (predicate.column_names != null) {
            ThriftValidation.validateColumns(keyspace, cfName, scName, predicate.column_names);
        }
    }

    public static void validatePredicate(String keyspace, ColumnParent column_parent, SlicePredicate predicate) throws InvalidRequestException {
        if (predicate.column_names == null && predicate.slice_range == null) {
            throw new InvalidRequestException("predicate column_names and slice_range may not both be null");
        }
        if (predicate.column_names != null && predicate.slice_range != null) {
            throw new InvalidRequestException("predicate column_names and slice_range may not both be present");
        }
        if (predicate.getSlice_range() != null) {
            ThriftValidation.validateRange(keyspace, column_parent, predicate.slice_range);
        } else {
            ThriftValidation.validateColumns(keyspace, column_parent, predicate.column_names);
        }
    }

    public static void validateKeyRange(KeyRange range) throws InvalidRequestException {
        if (range.start_key == null != (range.end_key == null)) {
            throw new InvalidRequestException("start key and end key must either both be non-null, or both be null");
        }
        if (range.start_token == null != (range.end_token == null)) {
            throw new InvalidRequestException("start token and end token must either both be non-null, or both be null");
        }
        if (range.start_key == null == (range.start_token == null)) {
            throw new InvalidRequestException("exactly one of {start key, end key} or {start token, end token} must be specified");
        }
        if (range.start_key != null) {
            IPartitioner p = StorageService.getPartitioner();
            Object startToken = p.decorateKey((String)range.start_key).token;
            Object endToken = p.decorateKey((String)range.end_key).token;
            if (((Token)startToken).compareTo(endToken) > 0 && !((Token)endToken).equals(p.getMinimumToken())) {
                if (p instanceof RandomPartitioner) {
                    throw new InvalidRequestException("start key's md5 sorts after end key's md5.  this is not allowed; you probably should not specify end key at all, under RandomPartitioner");
                }
                throw new InvalidRequestException("start key must sort before (or equal to) finish key in your partitioner!");
            }
        }
        if (range.count <= 0) {
            throw new InvalidRequestException("maxRows must be positive");
        }
    }
}

