/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import java.io.File;
import java.io.IOError;
import java.io.IOException;
import java.util.zip.CRC32;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.RowMutation;
import org.apache.cassandra.db.commitlog.CommitLogHeader;
import org.apache.cassandra.io.util.BufferedRandomAccessFile;
import org.apache.cassandra.io.util.DataOutputBuffer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLogSegment {
    private static final Logger logger = LoggerFactory.getLogger(CommitLogSegment.class);
    private final BufferedRandomAccessFile logWriter;
    private final CommitLogHeader header = new CommitLogHeader();

    public CommitLogSegment() {
        String logFile = DatabaseDescriptor.getCommitLogLocation() + File.separator + "CommitLog-" + System.currentTimeMillis() + ".log";
        logger.info("Creating new commitlog segment " + logFile);
        try {
            this.logWriter = CommitLogSegment.createWriter(logFile);
            this.writeHeader();
        }
        catch (IOException e) {
            throw new IOError(e);
        }
    }

    public static boolean possibleCommitLogFile(String filename) {
        return filename.matches("CommitLog-\\d+.log");
    }

    public void writeHeader() throws IOException {
        CommitLogHeader.writeCommitLogHeader(this.header, this.getHeaderPath());
    }

    private static BufferedRandomAccessFile createWriter(String file) throws IOException {
        return new BufferedRandomAccessFile(file, "rw", 131072);
    }

    public CommitLogContext write(RowMutation rowMutation, Object serializedRow) throws IOException {
        long currentPosition = -1L;
        try {
            byte[] bytes;
            currentPosition = this.logWriter.getFilePointer();
            CommitLogContext cLogCtx = new CommitLogContext(currentPosition);
            for (ColumnFamily columnFamily : rowMutation.getColumnFamilies()) {
                CFMetaData cfm = DatabaseDescriptor.getCFMetaData(columnFamily.id());
                if (cfm == null) {
                    logger.error("Attempted to write commit log entry for unrecognized column family: " + columnFamily.id());
                    continue;
                }
                Integer id = cfm.cfId;
                if (this.header.isDirty(id)) continue;
                this.header.turnOn(id, this.logWriter.getFilePointer());
                this.writeHeader();
            }
            CRC32 checksum = new CRC32();
            if (serializedRow instanceof DataOutputBuffer) {
                bytes = ((DataOutputBuffer)serializedRow).getData();
            } else {
                assert (serializedRow instanceof byte[]);
                bytes = (byte[])serializedRow;
            }
            checksum.update(bytes.length);
            this.logWriter.writeInt(bytes.length);
            this.logWriter.writeLong(checksum.getValue());
            this.logWriter.write(bytes);
            checksum.update(bytes, 0, bytes.length);
            this.logWriter.writeLong(checksum.getValue());
            return cLogCtx;
        }
        catch (IOException e) {
            if (currentPosition != -1L) {
                this.logWriter.seek(currentPosition);
            }
            throw e;
        }
    }

    public void sync() throws IOException {
        this.logWriter.sync();
    }

    public void flush() throws IOException {
        this.logWriter.flush();
    }

    public CommitLogContext getContext() {
        return new CommitLogContext(this.logWriter.getFilePointer());
    }

    public CommitLogHeader getHeader() {
        return this.header;
    }

    public String getPath() {
        return this.logWriter.getPath();
    }

    public String getHeaderPath() {
        return CommitLogHeader.getHeaderPathFromSegment(this);
    }

    public long length() {
        try {
            return this.logWriter.length();
        }
        catch (IOException e) {
            throw new IOError(e);
        }
    }

    public void close() {
        try {
            this.logWriter.close();
        }
        catch (IOException e) {
            throw new IOError(e);
        }
    }

    public String toString() {
        return "CommitLogSegment(" + this.logWriter.getPath() + ')';
    }

    public class CommitLogContext {
        public final long position;

        public CommitLogContext(long position) {
            assert (position >= 0L);
            this.position = position;
        }

        public CommitLogSegment getSegment() {
            return CommitLogSegment.this;
        }

        public String toString() {
            return "CommitLogContext(file='" + CommitLogSegment.this.logWriter.getPath() + '\'' + ", position=" + this.position + ')';
        }
    }
}

