/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.sstable;

import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOError;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.Table;
import org.apache.cassandra.dht.IPartitioner;
import org.apache.cassandra.io.AbstractCompactedRow;
import org.apache.cassandra.io.ICompactionInfo;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.IndexHelper;
import org.apache.cassandra.io.sstable.IndexSummary;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.io.util.BufferedRandomAccessFile;
import org.apache.cassandra.io.util.FileMark;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.io.util.SegmentedFile;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.BloomFilter;
import org.apache.cassandra.utils.EstimatedHistogram;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SSTableWriter
extends SSTable {
    private static Logger logger = LoggerFactory.getLogger(SSTableWriter.class);
    private IndexWriter iwriter;
    private SegmentedFile.Builder dbuilder;
    private final BufferedRandomAccessFile dataFile;
    private DecoratedKey lastWrittenKey;
    private FileMark dataMark;

    public SSTableWriter(String filename, long keyCount) throws IOException {
        this(filename, keyCount, DatabaseDescriptor.getCFMetaData(Descriptor.fromFilename(filename)), StorageService.getPartitioner());
    }

    public SSTableWriter(String filename, long keyCount, CFMetaData metadata, IPartitioner partitioner) throws IOException {
        super(Descriptor.fromFilename(filename), new HashSet<Component>(Arrays.asList(Component.DATA, Component.FILTER, Component.PRIMARY_INDEX, Component.STATS)), metadata, partitioner, SSTable.defaultRowHistogram(), SSTable.defaultColumnHistogram());
        this.iwriter = new IndexWriter(this.descriptor, partitioner, keyCount);
        this.dbuilder = SegmentedFile.getBuilder(DatabaseDescriptor.getDiskAccessMode());
        this.dataFile = new BufferedRandomAccessFile(this.getFilename(), "rw", DatabaseDescriptor.getInMemoryCompactionLimit());
    }

    public void mark() {
        this.dataMark = this.dataFile.mark();
        this.iwriter.mark();
    }

    public void reset() {
        try {
            this.dataFile.reset(this.dataMark);
            this.iwriter.reset();
        }
        catch (IOException e) {
            throw new IOError(e);
        }
    }

    private long beforeAppend(DecoratedKey decoratedKey) throws IOException {
        if (decoratedKey == null) {
            throw new IOException("Keys must not be null.");
        }
        if (this.lastWrittenKey != null && this.lastWrittenKey.compareTo(decoratedKey) > 0) {
            logger.info("Last written key : " + this.lastWrittenKey);
            logger.info("Current key : " + decoratedKey);
            logger.info("Writing into file " + this.getFilename());
            throw new IOException("Keys must be written in ascending order.");
        }
        return this.lastWrittenKey == null ? 0L : this.dataFile.getFilePointer();
    }

    private void afterAppend(DecoratedKey decoratedKey, long dataPosition) throws IOException {
        this.lastWrittenKey = decoratedKey;
        if (logger.isTraceEnabled()) {
            logger.trace("wrote " + decoratedKey + " at " + dataPosition);
        }
        this.iwriter.afterAppend(decoratedKey, dataPosition);
        this.dbuilder.addPotentialBoundary(dataPosition);
    }

    public void append(AbstractCompactedRow row) throws IOException {
        long currentPosition = this.beforeAppend(row.key);
        FBUtilities.writeShortByteArray(row.key.key, this.dataFile);
        row.write(this.dataFile);
        this.estimatedRowSize.add(this.dataFile.getFilePointer() - currentPosition);
        this.estimatedColumnCount.add(row.columnCount());
        this.afterAppend(row.key, currentPosition);
    }

    public void append(DecoratedKey decoratedKey, ColumnFamily cf) throws IOException {
        long startPosition = this.beforeAppend(decoratedKey);
        FBUtilities.writeShortByteArray(decoratedKey.key, this.dataFile);
        long sizePosition = this.dataFile.getFilePointer();
        this.dataFile.writeLong(-1L);
        int columnCount = ColumnFamily.serializer().serializeWithIndexes(cf, this.dataFile);
        long endPosition = this.dataFile.getFilePointer();
        this.dataFile.seek(sizePosition);
        this.dataFile.writeLong(endPosition - (sizePosition + 8L));
        this.dataFile.seek(endPosition);
        this.afterAppend(decoratedKey, startPosition);
        this.estimatedRowSize.add(endPosition - startPosition);
        this.estimatedColumnCount.add(columnCount);
    }

    public void append(DecoratedKey decoratedKey, ByteBuffer value) throws IOException {
        long currentPosition = this.beforeAppend(decoratedKey);
        FBUtilities.writeShortByteArray(decoratedKey.key, this.dataFile);
        assert (value.remaining() > 0);
        this.dataFile.writeLong(value.remaining());
        this.dataFile.write(value.array(), value.position() + value.arrayOffset(), value.remaining());
        this.afterAppend(decoratedKey, currentPosition);
    }

    public SSTableReader closeAndOpenReader() throws IOException {
        return this.closeAndOpenReader(System.currentTimeMillis());
    }

    public SSTableReader closeAndOpenReader(long maxDataAge) throws IOException {
        this.iwriter.close();
        long position = this.dataFile.getFilePointer();
        this.dataFile.close();
        FileUtils.truncate(this.dataFile.getPath(), position);
        SSTableWriter.writeStatistics(this.descriptor, this.estimatedRowSize, this.estimatedColumnCount);
        Descriptor newdesc = SSTableWriter.rename(this.descriptor, this.components);
        SegmentedFile ifile = this.iwriter.builder.complete(newdesc.filenameFor(SSTable.COMPONENT_INDEX));
        SegmentedFile dfile = this.dbuilder.complete(newdesc.filenameFor(SSTable.COMPONENT_DATA));
        SSTableReader sstable = SSTableReader.internalOpen(newdesc, this.components, this.metadata, this.partitioner, ifile, dfile, this.iwriter.summary, this.iwriter.bf, maxDataAge, this.estimatedRowSize, this.estimatedColumnCount);
        this.iwriter = null;
        this.dbuilder = null;
        return sstable;
    }

    private static void writeStatistics(Descriptor desc, EstimatedHistogram rowSizes, EstimatedHistogram columnnCounts) throws IOException {
        DataOutputStream out = new DataOutputStream(new FileOutputStream(desc.filenameFor(SSTable.COMPONENT_STATS)));
        EstimatedHistogram.serializer.serialize(rowSizes, out);
        EstimatedHistogram.serializer.serialize(rowSizes, out);
        out.close();
    }

    static Descriptor rename(Descriptor tmpdesc, Set<Component> components) {
        Descriptor newdesc = tmpdesc.asTemporary(false);
        try {
            for (Component component : components) {
                FBUtilities.renameWithConfirm(tmpdesc.filenameFor(component), newdesc.filenameFor(component));
            }
        }
        catch (IOException e) {
            throw new IOError(e);
        }
        return newdesc;
    }

    public long getFilePointer() {
        return this.dataFile.getFilePointer();
    }

    public static Builder createBuilder(Descriptor desc) {
        if (!desc.isLatestVersion) {
            throw new RuntimeException(String.format("Cannot recover SSTable with version %s (current version %s).", desc.version, "e"));
        }
        return new Builder(desc);
    }

    static class IndexWriter {
        private final BufferedRandomAccessFile indexFile;
        public final Descriptor desc;
        public final IPartitioner partitioner;
        public final SegmentedFile.Builder builder;
        public final IndexSummary summary;
        public final BloomFilter bf;
        private FileMark mark;

        IndexWriter(Descriptor desc, IPartitioner part, long keyCount) throws IOException {
            this.desc = desc;
            this.partitioner = part;
            this.indexFile = new BufferedRandomAccessFile(desc.filenameFor(SSTable.COMPONENT_INDEX), "rw", 0x800000);
            this.builder = SegmentedFile.getBuilder(DatabaseDescriptor.getIndexAccessMode());
            this.summary = new IndexSummary(keyCount);
            this.bf = BloomFilter.getFilter(keyCount, 15);
        }

        public void afterAppend(DecoratedKey key, long dataPosition) throws IOException {
            this.bf.add(key.key);
            long indexPosition = this.indexFile.getFilePointer();
            FBUtilities.writeShortByteArray(key.key, this.indexFile);
            this.indexFile.writeLong(dataPosition);
            if (logger.isTraceEnabled()) {
                logger.trace("wrote index of " + key + " at " + indexPosition);
            }
            this.summary.maybeAddEntry(key, indexPosition);
            this.builder.addPotentialBoundary(indexPosition);
        }

        public void close() throws IOException {
            FileOutputStream fos = new FileOutputStream(this.desc.filenameFor(SSTable.COMPONENT_FILTER));
            DataOutputStream stream = new DataOutputStream(fos);
            BloomFilter.serializer().serialize(this.bf, stream);
            stream.flush();
            fos.getFD().sync();
            stream.close();
            long position = this.indexFile.getFilePointer();
            this.indexFile.close();
            FileUtils.truncate(this.indexFile.getPath(), position);
            this.summary.complete();
        }

        public void mark() {
            this.mark = this.indexFile.mark();
        }

        public void reset() throws IOException {
            this.indexFile.reset(this.mark);
        }
    }

    public static class Builder
    implements ICompactionInfo {
        private final Descriptor desc;
        public final ColumnFamilyStore cfs;
        private BufferedRandomAccessFile dfile;

        public Builder(Descriptor desc) {
            this.desc = desc;
            this.cfs = Table.open(desc.ksname).getColumnFamilyStore(desc.cfname);
            try {
                this.dfile = new BufferedRandomAccessFile(desc.filenameFor(SSTable.COMPONENT_DATA), "r", 0x800000);
            }
            catch (IOException e) {
                throw new IOError(e);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public SSTableReader build() throws IOException {
            IndexWriter iwriter;
            long estimatedRows;
            if (this.cfs.isInvalid()) {
                return null;
            }
            File ifile = new File(this.desc.filenameFor(SSTable.COMPONENT_INDEX));
            File ffile = new File(this.desc.filenameFor(SSTable.COMPONENT_FILTER));
            assert (!ifile.exists());
            assert (!ffile.exists());
            EstimatedHistogram rowSizes = SSTable.defaultRowHistogram();
            EstimatedHistogram columnCounts = SSTable.defaultColumnHistogram();
            try {
                estimatedRows = SSTable.estimateRowsFromData(this.desc, this.dfile);
                iwriter = new IndexWriter(this.desc, StorageService.getPartitioner(), estimatedRows);
            }
            catch (IOException e) {
                this.dfile.close();
                throw e;
            }
            long rows = 0L;
            try {
                long rowPosition = 0L;
                while (rowPosition < this.dfile.length()) {
                    DecoratedKey key = SSTableReader.decodeKey(StorageService.getPartitioner(), this.desc, FBUtilities.readShortByteArray(this.dfile));
                    iwriter.afterAppend(key, rowPosition);
                    long dataSize = SSTableReader.readRowSize(this.dfile, this.desc);
                    rowPosition = this.dfile.getFilePointer() + dataSize;
                    IndexHelper.skipBloomFilter(this.dfile);
                    IndexHelper.skipIndex(this.dfile);
                    ColumnFamily.serializer().deserializeFromSSTableNoColumns(ColumnFamily.create(this.cfs.metadata), this.dfile);
                    rowSizes.add(dataSize);
                    columnCounts.add(this.dfile.readInt());
                    this.dfile.seek(rowPosition);
                    ++rows;
                }
                SSTableWriter.writeStatistics(this.desc, rowSizes, columnCounts);
            }
            finally {
                try {
                    this.dfile.close();
                    iwriter.close();
                }
                catch (IOException e) {
                    throw new IOError(e);
                }
            }
            logger.debug("estimated row count was %s of real count", (Object)((double)estimatedRows / (double)rows));
            return SSTableReader.open(SSTableWriter.rename(this.desc, SSTable.componentsFor(this.desc)));
        }

        @Override
        public long getTotalBytes() {
            try {
                return this.dfile.length();
            }
            catch (IOException e) {
                throw new IOError(e);
            }
        }

        @Override
        public long getBytesRead() {
            return this.dfile.getFilePointer();
        }

        @Override
        public String getTaskType() {
            return "SSTable rebuild";
        }
    }
}

