/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.thrift;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Comparator;
import java.util.SortedSet;
import org.apache.cassandra.config.ConfigurationException;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyType;
import org.apache.cassandra.db.KeyspaceNotDefinedException;
import org.apache.cassandra.db.Table;
import org.apache.cassandra.db.marshal.AbstractType;
import org.apache.cassandra.db.marshal.MarshalException;
import org.apache.cassandra.dht.IPartitioner;
import org.apache.cassandra.dht.RandomPartitioner;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.thrift.CfDef;
import org.apache.cassandra.thrift.Column;
import org.apache.cassandra.thrift.ColumnDef;
import org.apache.cassandra.thrift.ColumnOrSuperColumn;
import org.apache.cassandra.thrift.ColumnParent;
import org.apache.cassandra.thrift.ColumnPath;
import org.apache.cassandra.thrift.Deletion;
import org.apache.cassandra.thrift.IndexClause;
import org.apache.cassandra.thrift.IndexExpression;
import org.apache.cassandra.thrift.IndexOperator;
import org.apache.cassandra.thrift.InvalidRequestException;
import org.apache.cassandra.thrift.KeyRange;
import org.apache.cassandra.thrift.Mutation;
import org.apache.cassandra.thrift.SlicePredicate;
import org.apache.cassandra.thrift.SliceRange;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ThriftValidation {
    private static final Logger logger = LoggerFactory.getLogger(ThriftValidation.class);

    static void validateKey(ByteBuffer key) throws InvalidRequestException {
        if (key == null || key.remaining() == 0) {
            throw new InvalidRequestException("Key may not be empty");
        }
        if (key.remaining() > 65535) {
            throw new InvalidRequestException("Key length of " + key.remaining() + " is longer than maximum of " + 65535);
        }
    }

    public static void validateTable(String tablename) throws KeyspaceNotDefinedException {
        if (!DatabaseDescriptor.getTables().contains(tablename)) {
            throw new KeyspaceNotDefinedException("Keyspace " + tablename + " does not exist");
        }
    }

    public static ColumnFamilyType validateColumnFamily(String tablename, String cfName) throws InvalidRequestException {
        if (cfName.isEmpty()) {
            throw new InvalidRequestException("non-empty columnfamily is required");
        }
        ColumnFamilyType cfType = DatabaseDescriptor.getColumnFamilyType(tablename, cfName);
        if (cfType == null) {
            throw new InvalidRequestException("unconfigured columnfamily " + cfName);
        }
        return cfType;
    }

    static void validateColumnPath(String tablename, ColumnPath column_path) throws InvalidRequestException {
        ThriftValidation.validateTable(tablename);
        ColumnFamilyType cfType = ThriftValidation.validateColumnFamily(tablename, column_path.column_family);
        if (cfType == ColumnFamilyType.Standard) {
            if (column_path.super_column != null) {
                throw new InvalidRequestException("supercolumn parameter is invalid for standard CF " + column_path.column_family);
            }
            if (column_path.column == null) {
                throw new InvalidRequestException("column parameter is not optional for standard CF " + column_path.column_family);
            }
        } else if (column_path.super_column == null) {
            throw new InvalidRequestException("supercolumn parameter is not optional for super CF " + column_path.column_family);
        }
        if (column_path.column != null) {
            ThriftValidation.validateColumns(tablename, column_path.column_family, column_path.super_column, Arrays.asList(column_path.column));
        }
        if (column_path.super_column != null) {
            ThriftValidation.validateColumns(tablename, column_path.column_family, null, Arrays.asList(column_path.super_column));
        }
    }

    static void validateColumnParent(String tablename, ColumnParent column_parent) throws InvalidRequestException {
        ThriftValidation.validateTable(tablename);
        ColumnFamilyType cfType = ThriftValidation.validateColumnFamily(tablename, column_parent.column_family);
        if (cfType == ColumnFamilyType.Standard && column_parent.super_column != null) {
            throw new InvalidRequestException("columnfamily alone is required for standard CF " + column_parent.column_family);
        }
        if (column_parent.super_column != null) {
            ThriftValidation.validateColumns(tablename, column_parent.column_family, null, Arrays.asList(column_parent.super_column));
        }
    }

    static void validateColumnPathOrParent(String tablename, ColumnPath column_path_or_parent) throws InvalidRequestException {
        ThriftValidation.validateTable(tablename);
        ColumnFamilyType cfType = ThriftValidation.validateColumnFamily(tablename, column_path_or_parent.column_family);
        if (cfType == ColumnFamilyType.Standard && column_path_or_parent.super_column != null) {
            throw new InvalidRequestException("supercolumn may not be specified for standard CF " + column_path_or_parent.column_family);
        }
        if (column_path_or_parent.column != null) {
            ThriftValidation.validateColumns(tablename, column_path_or_parent.column_family, column_path_or_parent.super_column, Arrays.asList(column_path_or_parent.column));
        }
        if (column_path_or_parent.super_column != null) {
            ThriftValidation.validateColumns(tablename, column_path_or_parent.column_family, null, Arrays.asList(column_path_or_parent.super_column));
        }
    }

    private static void validateColumns(String keyspace, String columnFamilyName, ByteBuffer superColumnName, Iterable<ByteBuffer> column_names) throws InvalidRequestException {
        if (superColumnName != null) {
            if (superColumnName.remaining() > 65535) {
                throw new InvalidRequestException("supercolumn name length must not be greater than 65535");
            }
            if (superColumnName.remaining() == 0) {
                throw new InvalidRequestException("supercolumn name must not be empty");
            }
            if (DatabaseDescriptor.getColumnFamilyType(keyspace, columnFamilyName) == ColumnFamilyType.Standard) {
                throw new InvalidRequestException("supercolumn specified to ColumnFamily " + columnFamilyName + " containing normal columns");
            }
        }
        AbstractType comparator = ColumnFamily.getComparatorFor(keyspace, columnFamilyName, superColumnName);
        for (ByteBuffer name : column_names) {
            if (name.remaining() > 65535) {
                throw new InvalidRequestException("column name length must not be greater than 65535");
            }
            if (name.remaining() == 0) {
                throw new InvalidRequestException("column name must not be empty");
            }
            try {
                comparator.validate(name);
            }
            catch (MarshalException e) {
                throw new InvalidRequestException(e.getMessage());
            }
        }
    }

    public static void validateColumns(String keyspace, ColumnParent column_parent, Iterable<ByteBuffer> column_names) throws InvalidRequestException {
        ThriftValidation.validateColumns(keyspace, column_parent.column_family, column_parent.super_column, column_names);
    }

    public static void validateRange(String keyspace, ColumnParent column_parent, SliceRange range) throws InvalidRequestException {
        Comparator<ByteBuffer> orderedComparator;
        AbstractType comparator = ColumnFamily.getComparatorFor(keyspace, column_parent.column_family, column_parent.super_column);
        try {
            comparator.validate(range.start);
            comparator.validate(range.finish);
        }
        catch (MarshalException e) {
            throw new InvalidRequestException(e.getMessage());
        }
        if (range.count < 0) {
            throw new InvalidRequestException("get_slice requires non-negative count");
        }
        Comparator<ByteBuffer> comparator2 = orderedComparator = range.isReversed() ? comparator.getReverseComparator() : comparator;
        if (range.start.remaining() > 0 && range.finish.remaining() > 0 && orderedComparator.compare(range.start, range.finish) > 0) {
            throw new InvalidRequestException("range finish must come after start in the order of traversal");
        }
    }

    public static void validateColumnOrSuperColumn(String keyspace, String cfName, ColumnOrSuperColumn cosc) throws InvalidRequestException {
        if (cosc.column != null) {
            ThriftValidation.validateTtl(cosc.column);
            ThriftValidation.validateColumnPath(keyspace, new ColumnPath(cfName).setSuper_column((ByteBuffer)null).setColumn(cosc.column.name));
        }
        if (cosc.super_column != null) {
            for (Column c : cosc.super_column.columns) {
                ThriftValidation.validateTtl(c);
                ThriftValidation.validateColumnPath(keyspace, new ColumnPath(cfName).setSuper_column(cosc.super_column.name).setColumn(c.name));
            }
        }
        if (cosc.column == null && cosc.super_column == null) {
            throw new InvalidRequestException("ColumnOrSuperColumn must have one or both of Column or SuperColumn");
        }
    }

    private static void validateTtl(Column column) throws InvalidRequestException {
        if (column.isSetTtl() && column.ttl <= 0) {
            throw new InvalidRequestException("ttl must be positive");
        }
        assert (column.isSetTtl() || column.ttl == 0);
    }

    public static void validateMutation(String keyspace, String cfName, Mutation mut) throws InvalidRequestException {
        ColumnOrSuperColumn cosc = mut.column_or_supercolumn;
        Deletion del = mut.deletion;
        if (cosc != null && del != null) {
            throw new InvalidRequestException("Mutation may have either a ColumnOrSuperColumn or a Deletion, but not both");
        }
        if (cosc != null) {
            ThriftValidation.validateColumnOrSuperColumn(keyspace, cfName, cosc);
        } else if (del != null) {
            ThriftValidation.validateDeletion(keyspace, cfName, del);
        } else {
            throw new InvalidRequestException("Mutation must have one ColumnOrSuperColumn or one Deletion");
        }
    }

    public static void validateDeletion(String keyspace, String cfName, Deletion del) throws InvalidRequestException {
        ThriftValidation.validateColumnFamily(keyspace, cfName);
        if (del.predicate != null) {
            ThriftValidation.validateSlicePredicate(keyspace, cfName, del.super_column, del.predicate);
            if (del.predicate.slice_range != null) {
                throw new InvalidRequestException("Deletion does not yet support SliceRange predicates.");
            }
        }
        if (ColumnFamilyType.Standard == DatabaseDescriptor.getColumnFamilyType(keyspace, cfName) && del.super_column != null) {
            String msg = String.format("deletion of super_column is not possible on a standard ColumnFamily (KeySpace=%s ColumnFamily=%s Deletion=%s)", keyspace, cfName, del);
            throw new InvalidRequestException(msg);
        }
    }

    public static void validateSlicePredicate(String keyspace, String cfName, ByteBuffer scName, SlicePredicate predicate) throws InvalidRequestException {
        if (predicate.column_names == null && predicate.slice_range == null) {
            throw new InvalidRequestException("A SlicePredicate must be given a list of Columns, a SliceRange, or both");
        }
        if (predicate.slice_range != null) {
            ThriftValidation.validateRange(keyspace, new ColumnParent(cfName).setSuper_column(scName), predicate.slice_range);
        }
        if (predicate.column_names != null) {
            ThriftValidation.validateColumns(keyspace, cfName, scName, predicate.column_names);
        }
    }

    public static void validateColumn(String keyspace, ColumnParent column_parent, Column column) throws InvalidRequestException {
        ThriftValidation.validateTtl(column);
        ThriftValidation.validateColumns(keyspace, column_parent, Arrays.asList(column.name));
        try {
            AbstractType validator = DatabaseDescriptor.getValueValidator(keyspace, column_parent.column_family, column.name);
            if (validator != null) {
                validator.validate(column.value);
            }
        }
        catch (MarshalException me) {
            throw new InvalidRequestException(String.format("[%s][%s][%s] = [%s] failed validation (%s)", keyspace, column_parent.getColumn_family(), FBUtilities.bytesToHex(column.name), FBUtilities.bytesToHex(column.value), me.getMessage()));
        }
    }

    public static void validatePredicate(String keyspace, ColumnParent column_parent, SlicePredicate predicate) throws InvalidRequestException {
        if (predicate.column_names == null && predicate.slice_range == null) {
            throw new InvalidRequestException("predicate column_names and slice_range may not both be null");
        }
        if (predicate.column_names != null && predicate.slice_range != null) {
            throw new InvalidRequestException("predicate column_names and slice_range may not both be present");
        }
        if (predicate.getSlice_range() != null) {
            ThriftValidation.validateRange(keyspace, column_parent, predicate.slice_range);
        } else {
            ThriftValidation.validateColumns(keyspace, column_parent, predicate.column_names);
        }
    }

    public static void validateKeyRange(KeyRange range) throws InvalidRequestException {
        Object endToken;
        IPartitioner p;
        Object startToken;
        if (range.start_key == null != (range.end_key == null)) {
            throw new InvalidRequestException("start key and end key must either both be non-null, or both be null");
        }
        if (range.start_token == null != (range.end_token == null)) {
            throw new InvalidRequestException("start token and end token must either both be non-null, or both be null");
        }
        if (range.start_key == null == (range.start_token == null)) {
            throw new InvalidRequestException("exactly one of {start key, end key} or {start token, end token} must be specified");
        }
        if (range.start_key != null && ((Token)(startToken = (p = StorageService.getPartitioner()).getToken(range.start_key))).compareTo(endToken = p.getToken(range.end_key)) > 0 && !((Token)endToken).equals(p.getMinimumToken())) {
            if (p instanceof RandomPartitioner) {
                throw new InvalidRequestException("start key's md5 sorts after end key's md5.  this is not allowed; you probably should not specify end key at all, under RandomPartitioner");
            }
            throw new InvalidRequestException("start key must sort before (or equal to) finish key in your partitioner!");
        }
        if (range.count <= 0) {
            throw new InvalidRequestException("maxRows must be positive");
        }
    }

    public static void validateIndexClauses(String keyspace, String columnFamily, IndexClause index_clause) throws InvalidRequestException {
        if (index_clause.expressions.isEmpty()) {
            throw new InvalidRequestException("index clause list may not be empty");
        }
        SortedSet<ByteBuffer> indexedColumns = Table.open(keyspace).getColumnFamilyStore(columnFamily).getIndexedColumns();
        for (IndexExpression expression : index_clause.expressions) {
            if (!expression.op.equals((Object)IndexOperator.EQ) || !indexedColumns.contains(expression.column_name)) continue;
            return;
        }
        throw new InvalidRequestException("No indexed columns present in index clause with operator EQ");
    }

    public static void validateCfDef(CfDef cf_def) throws InvalidRequestException {
        try {
            ColumnFamilyType cfType = ColumnFamilyType.create(cf_def.column_type);
            if (cfType == null) {
                throw new InvalidRequestException("invalid column type " + cf_def.column_type);
            }
            DatabaseDescriptor.getComparator(cf_def.comparator_type);
            DatabaseDescriptor.getComparator(cf_def.subcomparator_type);
            DatabaseDescriptor.getComparator(cf_def.default_validation_class);
            if (cfType != ColumnFamilyType.Super && cf_def.subcomparator_type != null) {
                throw new InvalidRequestException("subcomparator_type is invalid for standard columns");
            }
            if (cf_def.column_metadata == null) {
                return;
            }
            AbstractType comparator = cfType == ColumnFamilyType.Standard ? DatabaseDescriptor.getComparator(cf_def.comparator_type) : DatabaseDescriptor.getComparator(cf_def.subcomparator_type);
            for (ColumnDef c : cf_def.column_metadata) {
                DatabaseDescriptor.getComparator(c.validation_class);
                try {
                    comparator.validate(c.name);
                }
                catch (MarshalException e) {
                    throw new InvalidRequestException(String.format("Column name %s is not valid for comparator %s", FBUtilities.bytesToHex(c.name), cf_def.comparator_type));
                }
                if (c.index_name != null && c.index_type == null) {
                    throw new ConfigurationException("index_name cannot be set without index_type");
                }
                if (cfType != ColumnFamilyType.Super || c.index_type == null) continue;
                throw new InvalidRequestException("Secondary indexes are not supported on supercolumns");
            }
        }
        catch (ConfigurationException e) {
            throw new InvalidRequestException(e.getMessage());
        }
    }
}

