/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.util;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.util.Shell;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.yarn.util.Clock;
import org.apache.hadoop.yarn.util.CpuTimeTracker;
import org.apache.hadoop.yarn.util.ResourceCalculatorPlugin;
import org.apache.hadoop.yarn.util.ResourceCalculatorProcessTree;
import org.apache.hadoop.yarn.util.SystemClock;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public class ProcfsBasedProcessTree
extends ResourceCalculatorProcessTree {
    static final Log LOG = LogFactory.getLog(ProcfsBasedProcessTree.class);
    private static final String PROCFS = "/proc/";
    private static final Pattern PROCFS_STAT_FILE_FORMAT = Pattern.compile("^([\\d-]+)\\s\\(([^)]+)\\)\\s[^\\s]\\s([\\d-]+)\\s([\\d-]+)\\s([\\d-]+)\\s([\\d-]+\\s){7}(\\d+)\\s(\\d+)\\s([\\d-]+\\s){7}(\\d+)\\s(\\d+)(\\s[\\d-]+){15}");
    public static final String PROCFS_STAT_FILE = "stat";
    public static final String PROCFS_CMDLINE_FILE = "cmdline";
    public static final long PAGE_SIZE;
    public static final long JIFFY_LENGTH_IN_MILLIS;
    private final CpuTimeTracker cpuTimeTracker;
    private Clock clock;
    public static final String SMAPS = "smaps";
    public static final int KB_TO_BYTES = 1024;
    private static final String KB = "kB";
    private static final String READ_ONLY_WITH_SHARED_PERMISSION = "r--s";
    private static final String READ_EXECUTE_WITH_SHARED_PERMISSION = "r-xs";
    private static final Pattern ADDRESS_PATTERN;
    private static final Pattern MEM_INFO_PATTERN;
    private boolean smapsEnabled;
    protected Map<String, ProcessTreeSmapMemInfo> processSMAPTree = new HashMap<String, ProcessTreeSmapMemInfo>();
    private String procfsDir;
    private static String deadPid;
    private String pid = deadPid;
    private static Pattern numberPattern;
    private long cpuTime = -1L;
    protected Map<String, ProcessInfo> processTree = new HashMap<String, ProcessInfo>();
    private static final String PROCESSTREE_DUMP_FORMAT = "\t|- %s %s %d %d %s %d %d %d %d %s\n";

    public ProcfsBasedProcessTree(String pid) {
        this(pid, PROCFS, new SystemClock());
    }

    public void setConf(Configuration conf) {
        super.setConf(conf);
        if (conf != null) {
            this.smapsEnabled = conf.getBoolean("yarn.nodemanager.container-monitor.procfs-tree.smaps-based-rss.enabled", false);
        }
    }

    public ProcfsBasedProcessTree(String pid, String procfsDir) {
        this(pid, procfsDir, new SystemClock());
    }

    public ProcfsBasedProcessTree(String pid, String procfsDir, Clock clock) {
        super(pid);
        this.clock = clock;
        this.pid = ProcfsBasedProcessTree.getValidPID(pid);
        this.procfsDir = procfsDir;
        this.cpuTimeTracker = new CpuTimeTracker(JIFFY_LENGTH_IN_MILLIS);
    }

    public static boolean isAvailable() {
        try {
            if (!Shell.LINUX) {
                LOG.info((Object)"ProcfsBasedProcessTree currently is supported only on Linux.");
                return false;
            }
        }
        catch (SecurityException se) {
            LOG.warn((Object)("Failed to get Operating System name. " + se));
            return false;
        }
        return true;
    }

    @Override
    public void updateProcessTree() {
        if (!this.pid.equals(deadPid)) {
            List<String> processList = this.getProcessList();
            HashMap<String, ProcessInfo> allProcessInfo = new HashMap<String, ProcessInfo>();
            HashMap<String, ProcessInfo> oldProcs = new HashMap<String, ProcessInfo>(this.processTree);
            this.processTree.clear();
            ProcessInfo me = null;
            for (String string : processList) {
                ProcessInfo pInfo = new ProcessInfo(string);
                if (ProcfsBasedProcessTree.constructProcessInfo(pInfo, this.procfsDir) == null) continue;
                allProcessInfo.put(string, pInfo);
                if (!string.equals(this.pid)) continue;
                me = pInfo;
                this.processTree.put(string, pInfo);
            }
            if (me == null) {
                return;
            }
            for (Map.Entry entry : allProcessInfo.entrySet()) {
                ProcessInfo pInfo;
                ProcessInfo parentPInfo;
                String pID = (String)entry.getKey();
                if (pID.equals("1") || (parentPInfo = (ProcessInfo)allProcessInfo.get((pInfo = (ProcessInfo)entry.getValue()).getPpid())) == null) continue;
                parentPInfo.addChild(pInfo);
            }
            LinkedList<ProcessInfo> pInfoQueue = new LinkedList<ProcessInfo>();
            pInfoQueue.addAll(me.getChildren());
            while (!pInfoQueue.isEmpty()) {
                ProcessInfo processInfo = (ProcessInfo)pInfoQueue.remove();
                if (!this.processTree.containsKey(processInfo.getPid())) {
                    this.processTree.put(processInfo.getPid(), processInfo);
                }
                pInfoQueue.addAll(processInfo.getChildren());
            }
            for (Map.Entry<String, ProcessInfo> procs : this.processTree.entrySet()) {
                ProcessInfo oldInfo = (ProcessInfo)oldProcs.get(procs.getKey());
                if (procs.getValue() == null) continue;
                procs.getValue().updateJiffy(oldInfo);
                if (oldInfo == null) continue;
                procs.getValue().updateAge(oldInfo);
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)this.toString());
            }
            if (this.smapsEnabled) {
                this.processSMAPTree.clear();
                for (ProcessInfo p : this.processTree.values()) {
                    if (p == null) continue;
                    ProcessTreeSmapMemInfo memInfo = new ProcessTreeSmapMemInfo(p.getPid());
                    ProcfsBasedProcessTree.constructProcessSMAPInfo(memInfo, this.procfsDir);
                    this.processSMAPTree.put(p.getPid(), memInfo);
                }
            }
        }
    }

    @Override
    public boolean checkPidPgrpidForMatch() {
        return ProcfsBasedProcessTree.checkPidPgrpidForMatch(this.pid, PROCFS);
    }

    public static boolean checkPidPgrpidForMatch(String _pid, String procfs) {
        ProcessInfo pInfo = new ProcessInfo(_pid);
        if ((pInfo = ProcfsBasedProcessTree.constructProcessInfo(pInfo, procfs)) == null) {
            return true;
        }
        String pgrpId = pInfo.getPgrpId().toString();
        return pgrpId.equals(_pid);
    }

    public List<String> getCurrentProcessIDs() {
        ArrayList<String> currentPIDs = new ArrayList<String>();
        currentPIDs.addAll(this.processTree.keySet());
        return currentPIDs;
    }

    @Override
    public String getProcessTreeDump() {
        StringBuilder ret = new StringBuilder();
        ret.append(String.format("\t|- PID PPID PGRPID SESSID CMD_NAME USER_MODE_TIME(MILLIS) SYSTEM_TIME(MILLIS) VMEM_USAGE(BYTES) RSSMEM_USAGE(PAGES) FULL_CMD_LINE\n", new Object[0]));
        for (ProcessInfo p : this.processTree.values()) {
            if (p == null) continue;
            ret.append(String.format(PROCESSTREE_DUMP_FORMAT, p.getPid(), p.getPpid(), p.getPgrpId(), p.getSessionId(), p.getName(), p.getUtime(), p.getStime(), p.getVmem(), p.getRssmemPage(), p.getCmdLine(this.procfsDir)));
        }
        return ret.toString();
    }

    @Override
    public long getVirtualMemorySize(int olderThanAge) {
        long total = -1L;
        for (ProcessInfo p : this.processTree.values()) {
            if (p == null) continue;
            if (total == -1L) {
                total = 0L;
            }
            if (p.getAge() <= olderThanAge) continue;
            total += p.getVmem().longValue();
        }
        return total;
    }

    @Override
    public long getCumulativeVmem(int olderThanAge) {
        return this.getVirtualMemorySize(olderThanAge);
    }

    @Override
    public long getRssMemorySize(int olderThanAge) {
        if (PAGE_SIZE < 0L) {
            return -1L;
        }
        if (this.smapsEnabled) {
            return this.getSmapBasedRssMemorySize(olderThanAge);
        }
        boolean isAvailable = false;
        long totalPages = 0L;
        for (ProcessInfo p : this.processTree.values()) {
            if (p == null) continue;
            if (p.getAge() > olderThanAge) {
                totalPages += p.getRssmemPage().longValue();
            }
            isAvailable = true;
        }
        return isAvailable ? totalPages * PAGE_SIZE : -1L;
    }

    @Override
    public long getCumulativeRssmem(int olderThanAge) {
        return this.getRssMemorySize(olderThanAge);
    }

    private long getSmapBasedRssMemorySize(int olderThanAge) {
        long total = -1L;
        for (ProcessInfo p : this.processTree.values()) {
            if (p == null) continue;
            if (total == -1L) {
                total = 0L;
            }
            if (p.getAge() <= olderThanAge) continue;
            ProcessTreeSmapMemInfo procMemInfo = this.processSMAPTree.get(p.getPid());
            if (procMemInfo != null) {
                for (ProcessSmapMemoryInfo info : procMemInfo.getMemoryInfoList()) {
                    if (info.getPermission().trim().equalsIgnoreCase(READ_ONLY_WITH_SHARED_PERMISSION) || info.getPermission().trim().equalsIgnoreCase(READ_EXECUTE_WITH_SHARED_PERMISSION)) continue;
                    total += (long)(Math.min(info.sharedDirty, info.pss) + info.privateDirty + info.privateClean);
                    if (!LOG.isDebugEnabled()) continue;
                    LOG.debug((Object)(" total(" + olderThanAge + "): PID : " + p.getPid() + ", SharedDirty : " + info.sharedDirty + ", PSS : " + info.pss + ", Private_Dirty : " + info.privateDirty + ", Private_Clean : " + info.privateClean + ", total : " + total * 1024L));
                }
            }
            if (!LOG.isDebugEnabled()) continue;
            LOG.debug((Object)procMemInfo.toString());
        }
        if (total > 0L) {
            total *= 1024L;
        }
        LOG.info((Object)("SmapBasedCumulativeRssmem (bytes) : " + total));
        return total;
    }

    @Override
    public long getCumulativeCpuTime() {
        if (JIFFY_LENGTH_IN_MILLIS < 0L) {
            return -1L;
        }
        long incJiffies = 0L;
        boolean isAvailable = false;
        for (ProcessInfo p : this.processTree.values()) {
            if (p == null) continue;
            incJiffies += p.getDtime().longValue();
            isAvailable = true;
        }
        if (isAvailable) {
            if (this.cpuTime == -1L) {
                this.cpuTime = 0L;
            }
            this.cpuTime += incJiffies * JIFFY_LENGTH_IN_MILLIS;
        }
        return this.cpuTime;
    }

    private BigInteger getTotalProcessJiffies() {
        BigInteger totalStime = BigInteger.ZERO;
        long totalUtime = 0L;
        for (ProcessInfo p : this.processTree.values()) {
            if (p == null) continue;
            totalUtime += p.getUtime().longValue();
            totalStime = totalStime.add(p.getStime());
        }
        return totalStime.add(BigInteger.valueOf(totalUtime));
    }

    @Override
    public float getCpuUsagePercent() {
        BigInteger processTotalJiffies = this.getTotalProcessJiffies();
        this.cpuTimeTracker.updateElapsedJiffies(processTotalJiffies, this.clock.getTime());
        return this.cpuTimeTracker.getCpuTrackerUsagePercent();
    }

    private static String getValidPID(String pid) {
        if (pid == null) {
            return deadPid;
        }
        Matcher m = numberPattern.matcher(pid);
        if (m.matches()) {
            return pid;
        }
        return deadPid;
    }

    private List<String> getProcessList() {
        String[] processDirs = new File(this.procfsDir).list();
        ArrayList<String> processList = new ArrayList<String>();
        for (String dir : processDirs) {
            Matcher m = numberPattern.matcher(dir);
            if (!m.matches()) continue;
            try {
                if (!new File(this.procfsDir, dir).isDirectory()) continue;
                processList.add(dir);
            }
            catch (SecurityException s) {
                // empty catch block
            }
        }
        return processList;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static ProcessInfo constructProcessInfo(ProcessInfo pinfo, String procfsDir) {
        ProcessInfo ret = null;
        BufferedReader in = null;
        FileReader fReader = null;
        try {
            File pidDir = new File(procfsDir, pinfo.getPid());
            fReader = new FileReader(new File(pidDir, PROCFS_STAT_FILE));
            in = new BufferedReader(fReader);
        }
        catch (FileNotFoundException f) {
            return ret;
        }
        ret = pinfo;
        try {
            String str = in.readLine();
            Matcher m = PROCFS_STAT_FILE_FORMAT.matcher(str);
            boolean mat = m.find();
            if (mat) {
                String processName = "(" + m.group(2) + ")";
                pinfo.updateProcessInfo(processName, m.group(3), Integer.parseInt(m.group(4)), Integer.parseInt(m.group(5)), Long.parseLong(m.group(7)), new BigInteger(m.group(8)), Long.parseLong(m.group(10)), Long.parseLong(m.group(11)));
            } else {
                LOG.warn((Object)("Unexpected: procfs stat file is not in the expected format for process with pid " + pinfo.getPid()));
                ret = null;
            }
        }
        catch (IOException io) {
            LOG.warn((Object)("Error reading the stream " + io));
            ret = null;
        }
        finally {
            try {
                fReader.close();
                try {
                    in.close();
                }
                catch (IOException i) {
                    LOG.warn((Object)("Error closing the stream " + in));
                }
            }
            catch (IOException i) {
                LOG.warn((Object)("Error closing the stream " + fReader));
            }
        }
        return ret;
    }

    public String toString() {
        StringBuffer pTree = new StringBuffer("[ ");
        for (String p : this.processTree.keySet()) {
            pTree.append(p);
            pTree.append(" ");
        }
        return pTree.substring(0, pTree.length()) + "]";
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void constructProcessSMAPInfo(ProcessTreeSmapMemInfo pInfo, String procfsDir) {
        BufferedReader in = null;
        FileReader fReader = null;
        try {
            File pidDir = new File(procfsDir, pInfo.getPid());
            File file = new File(pidDir, SMAPS);
            if (!file.exists()) {
                return;
            }
            fReader = new FileReader(file);
            in = new BufferedReader(fReader);
            ProcessSmapMemoryInfo memoryMappingInfo = null;
            List<String> lines = IOUtils.readLines(in);
            for (String line : lines) {
                line = line.trim();
                try {
                    Matcher address = ADDRESS_PATTERN.matcher(line);
                    if (address.find()) {
                        memoryMappingInfo = new ProcessSmapMemoryInfo(line);
                        memoryMappingInfo.setPermission(address.group(4));
                        pInfo.getMemoryInfoList().add(memoryMappingInfo);
                        continue;
                    }
                    Matcher memInfo = MEM_INFO_PATTERN.matcher(line);
                    if (!memInfo.find()) continue;
                    String key = memInfo.group(1).trim();
                    String value = memInfo.group(2).replace(KB, "").trim();
                    if (LOG.isDebugEnabled()) {
                        LOG.debug((Object)("MemInfo : " + key + " : Value  : " + value));
                    }
                    memoryMappingInfo.setMemInfo(key, value);
                }
                catch (Throwable t) {
                    LOG.warn((Object)("Error parsing smaps line : " + line + "; " + t.getMessage()));
                }
            }
            IOUtils.closeQuietly(in);
        }
        catch (FileNotFoundException f) {
            LOG.error((Object)f.getMessage());
        }
        catch (IOException e) {
            LOG.error((Object)e.getMessage());
        }
        catch (Throwable t) {
            LOG.error((Object)t.getMessage());
        }
        finally {
            IOUtils.closeQuietly(in);
        }
    }

    public static void main(String[] args) {
        if (args.length != 1) {
            System.out.println("Provide <pid of process to monitor>");
            return;
        }
        int numprocessors = ResourceCalculatorPlugin.getResourceCalculatorPlugin(null, null).getNumProcessors();
        System.out.println("Number of processors " + numprocessors);
        System.out.println("Creating ProcfsBasedProcessTree for process " + args[0]);
        ProcfsBasedProcessTree procfsBasedProcessTree = new ProcfsBasedProcessTree(args[0]);
        procfsBasedProcessTree.updateProcessTree();
        System.out.println(procfsBasedProcessTree.getProcessTreeDump());
        System.out.println("Get cpu usage " + procfsBasedProcessTree.getCpuUsagePercent());
        try {
            Thread.sleep(500L);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        procfsBasedProcessTree.updateProcessTree();
        System.out.println(procfsBasedProcessTree.getProcessTreeDump());
        System.out.println("Cpu usage  " + procfsBasedProcessTree.getCpuUsagePercent());
        System.out.println("Vmem usage in bytes " + procfsBasedProcessTree.getVirtualMemorySize());
        System.out.println("Rss mem usage in bytes " + procfsBasedProcessTree.getRssMemorySize());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static {
        ADDRESS_PATTERN = Pattern.compile("([[a-f]|(0-9)]*)-([[a-f]|(0-9)]*)(\\s)*([rxwps\\-]*)");
        MEM_INFO_PATTERN = Pattern.compile("(^[A-Z].*):[\\s ]*(.*)");
        long jiffiesPerSecond = -1L;
        long pageSize = -1L;
        try {
            if (Shell.LINUX) {
                Shell.ShellCommandExecutor shellExecutorClk = new Shell.ShellCommandExecutor(new String[]{"getconf", "CLK_TCK"});
                shellExecutorClk.execute();
                jiffiesPerSecond = Long.parseLong(shellExecutorClk.getOutput().replace("\n", ""));
                Shell.ShellCommandExecutor shellExecutorPage = new Shell.ShellCommandExecutor(new String[]{"getconf", "PAGESIZE"});
                shellExecutorPage.execute();
                pageSize = Long.parseLong(shellExecutorPage.getOutput().replace("\n", ""));
            }
            JIFFY_LENGTH_IN_MILLIS = jiffiesPerSecond != -1L ? Math.round(1000.0 / (double)jiffiesPerSecond) : -1L;
        }
        catch (IOException e) {
            try {
                LOG.error((Object)StringUtils.stringifyException((Throwable)e));
                JIFFY_LENGTH_IN_MILLIS = jiffiesPerSecond != -1L ? Math.round(1000.0 / (double)jiffiesPerSecond) : -1L;
            }
            catch (Throwable throwable) {
                JIFFY_LENGTH_IN_MILLIS = jiffiesPerSecond != -1L ? Math.round(1000.0 / (double)jiffiesPerSecond) : -1L;
                PAGE_SIZE = pageSize;
                throw throwable;
            }
            PAGE_SIZE = pageSize;
        }
        PAGE_SIZE = pageSize;
        deadPid = "-1";
        numberPattern = Pattern.compile("[1-9][0-9]*");
    }

    static class ProcessSmapMemoryInfo {
        private int size;
        private int rss;
        private int pss;
        private int sharedClean;
        private int sharedDirty;
        private int privateClean;
        private int privateDirty;
        private int referenced;
        private String regionName;
        private String permission;

        public ProcessSmapMemoryInfo(String name) {
            this.regionName = name;
        }

        public String getName() {
            return this.regionName;
        }

        public void setPermission(String permission) {
            this.permission = permission;
        }

        public String getPermission() {
            return this.permission;
        }

        public int getSize() {
            return this.size;
        }

        public int getRss() {
            return this.rss;
        }

        public int getPss() {
            return this.pss;
        }

        public int getSharedClean() {
            return this.sharedClean;
        }

        public int getSharedDirty() {
            return this.sharedDirty;
        }

        public int getPrivateClean() {
            return this.privateClean;
        }

        public int getPrivateDirty() {
            return this.privateDirty;
        }

        public int getReferenced() {
            return this.referenced;
        }

        public void setMemInfo(String key, String value) {
            MemInfo info = MemInfo.getMemInfoByName(key);
            int val = 0;
            try {
                val = Integer.parseInt(value.trim());
            }
            catch (NumberFormatException ne) {
                LOG.error((Object)("Error in parsing : " + (Object)((Object)info) + " : value" + value.trim()));
                return;
            }
            if (info == null) {
                return;
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("setMemInfo : memInfo : " + (Object)((Object)info)));
            }
            switch (info) {
                case SIZE: {
                    this.size = val;
                    break;
                }
                case RSS: {
                    this.rss = val;
                    break;
                }
                case PSS: {
                    this.pss = val;
                    break;
                }
                case SHARED_CLEAN: {
                    this.sharedClean = val;
                    break;
                }
                case SHARED_DIRTY: {
                    this.sharedDirty = val;
                    break;
                }
                case PRIVATE_CLEAN: {
                    this.privateClean = val;
                    break;
                }
                case PRIVATE_DIRTY: {
                    this.privateDirty = val;
                    break;
                }
                case REFERENCED: {
                    this.referenced = val;
                    break;
                }
            }
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("\t").append(this.getName()).append("\n");
            sb.append("\t").append(MemInfo.SIZE.name + ":" + this.getSize()).append(" kB\n");
            sb.append("\t").append(MemInfo.PSS.name + ":" + this.getPss()).append(" kB\n");
            sb.append("\t").append(MemInfo.RSS.name + ":" + this.getRss()).append(" kB\n");
            sb.append("\t").append(MemInfo.SHARED_CLEAN.name + ":" + this.getSharedClean()).append(" kB\n");
            sb.append("\t").append(MemInfo.SHARED_DIRTY.name + ":" + this.getSharedDirty()).append(" kB\n");
            sb.append("\t").append(MemInfo.PRIVATE_CLEAN.name + ":" + this.getPrivateClean()).append(" kB\n");
            sb.append("\t").append(MemInfo.PRIVATE_DIRTY.name + ":" + this.getPrivateDirty()).append(" kB\n");
            sb.append("\t").append(MemInfo.REFERENCED.name + ":" + this.getReferenced()).append(" kB\n");
            sb.append("\t").append(MemInfo.PRIVATE_DIRTY.name + ":" + this.getPrivateDirty()).append(" kB\n");
            sb.append("\t").append(MemInfo.PRIVATE_DIRTY.name + ":" + this.getPrivateDirty()).append(" kB\n");
            return sb.toString();
        }
    }

    static class ProcessTreeSmapMemInfo {
        private String pid;
        private List<ProcessSmapMemoryInfo> memoryInfoList;

        public ProcessTreeSmapMemInfo(String pid) {
            this.pid = pid;
            this.memoryInfoList = new LinkedList<ProcessSmapMemoryInfo>();
        }

        public List<ProcessSmapMemoryInfo> getMemoryInfoList() {
            return this.memoryInfoList;
        }

        public String getPid() {
            return this.pid;
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            for (ProcessSmapMemoryInfo info : this.memoryInfoList) {
                sb.append("\n");
                sb.append(info.toString());
            }
            return sb.toString();
        }
    }

    private static class ProcessInfo {
        private String pid;
        private String name;
        private Integer pgrpId;
        private String ppid;
        private Integer sessionId;
        private Long vmem;
        private Long rssmemPage;
        private Long utime = 0L;
        private final BigInteger MAX_LONG = BigInteger.valueOf(Long.MAX_VALUE);
        private BigInteger stime = new BigInteger("0");
        private int age;
        private Long dtime = 0L;
        private List<ProcessInfo> children = new ArrayList<ProcessInfo>();

        public ProcessInfo(String pid) {
            this.pid = pid;
            this.age = 1;
        }

        public String getPid() {
            return this.pid;
        }

        public String getName() {
            return this.name;
        }

        public Integer getPgrpId() {
            return this.pgrpId;
        }

        public String getPpid() {
            return this.ppid;
        }

        public Integer getSessionId() {
            return this.sessionId;
        }

        public Long getVmem() {
            return this.vmem;
        }

        public Long getUtime() {
            return this.utime;
        }

        public BigInteger getStime() {
            return this.stime;
        }

        public Long getDtime() {
            return this.dtime;
        }

        public Long getRssmemPage() {
            return this.rssmemPage;
        }

        public int getAge() {
            return this.age;
        }

        public void updateProcessInfo(String name, String ppid, Integer pgrpId, Integer sessionId, Long utime, BigInteger stime, Long vmem, Long rssmem) {
            this.name = name;
            this.ppid = ppid;
            this.pgrpId = pgrpId;
            this.sessionId = sessionId;
            this.utime = utime;
            this.stime = stime;
            this.vmem = vmem;
            this.rssmemPage = rssmem;
        }

        public void updateJiffy(ProcessInfo oldInfo) {
            if (oldInfo == null) {
                BigInteger sum = this.stime.add(BigInteger.valueOf(this.utime));
                if (sum.compareTo(this.MAX_LONG) > 0) {
                    this.dtime = 0L;
                    LOG.warn((Object)("Sum of stime (" + this.stime + ") and utime (" + this.utime + ") is greater than " + Long.MAX_VALUE));
                } else {
                    this.dtime = sum.longValue();
                }
                return;
            }
            this.dtime = this.utime - oldInfo.utime + this.stime.subtract(oldInfo.stime).longValue();
        }

        public void updateAge(ProcessInfo oldInfo) {
            this.age = oldInfo.age + 1;
        }

        public boolean addChild(ProcessInfo p) {
            return this.children.add(p);
        }

        public List<ProcessInfo> getChildren() {
            return this.children;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public String getCmdLine(String procfsDir) {
            String ret = "N/A";
            if (this.pid == null) {
                return ret;
            }
            BufferedReader in = null;
            FileReader fReader = null;
            try {
                fReader = new FileReader(new File(new File(procfsDir, this.pid.toString()), ProcfsBasedProcessTree.PROCFS_CMDLINE_FILE));
            }
            catch (FileNotFoundException f) {
                return ret;
            }
            in = new BufferedReader(fReader);
            try {
                ret = in.readLine();
                if (ret == null) {
                    ret = "N/A";
                } else if ((ret = ret.replace('\u0000', ' ')).equals("")) {
                    ret = "N/A";
                }
            }
            catch (IOException io) {
                LOG.warn((Object)("Error reading the stream " + io));
                ret = "N/A";
            }
            finally {
                try {
                    fReader.close();
                    try {
                        in.close();
                    }
                    catch (IOException i) {
                        LOG.warn((Object)("Error closing the stream " + in));
                    }
                }
                catch (IOException i) {
                    LOG.warn((Object)("Error closing the stream " + fReader));
                }
            }
            return ret;
        }
    }

    static enum MemInfo {
        SIZE("Size"),
        RSS("Rss"),
        PSS("Pss"),
        SHARED_CLEAN("Shared_Clean"),
        SHARED_DIRTY("Shared_Dirty"),
        PRIVATE_CLEAN("Private_Clean"),
        PRIVATE_DIRTY("Private_Dirty"),
        REFERENCED("Referenced"),
        ANONYMOUS("Anonymous"),
        ANON_HUGE_PAGES("AnonHugePages"),
        SWAP("swap"),
        KERNEL_PAGE_SIZE("kernelPageSize"),
        MMU_PAGE_SIZE("mmuPageSize"),
        INVALID("invalid");

        private String name;

        private MemInfo(String name) {
            this.name = name;
        }

        public static MemInfo getMemInfoByName(String name) {
            for (MemInfo info : MemInfo.values()) {
                if (!info.name.trim().equalsIgnoreCase(name.trim())) continue;
                return info;
            }
            return INVALID;
        }
    }
}

