package eu.dnetlib.data.information.oai.publisher.conf;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.DocumentException;

import com.google.common.collect.Lists;

import eu.dnetlib.data.information.oai.publisher.OaiPublisherException;
import eu.dnetlib.data.information.oai.publisher.OaiPublisherRuntimeException;
import eu.dnetlib.data.information.oai.publisher.info.MDFInfo;
import eu.dnetlib.data.information.oai.publisher.info.SetInfo;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpDocumentNotFoundException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.tools.ServiceLocator;
import eu.dnetlib.miscutils.functional.IdentityFunction;
import eu.dnetlib.miscutils.functional.UnaryFunction;
import eu.dnetlib.miscutils.functional.xml.ApplyXslt;

/**
 * Helper class for common queries to the IS.
 * 
 * @author alessia
 * 
 */
public class ISLookUpClient {

	/**
	 * lookup service.
	 */
	@Resource(name = "lookupLocator")
	private ServiceLocator<ISLookUpService> lookupLocator;
	@Resource
	private PublisherConfigurationReader configuration;

	private static final Log log = LogFactory.getLog(ISLookUpClient.class); // NOPMD by marko on 11/24/08 5:02 PM

	public List<MDFInfo> listMetadataFormats(final boolean onlyEnabled) throws ISLookUpException, DocumentException, OaiPublisherException {
		return Lists.newArrayList(this.configuration.getMetadataFormatInfo(onlyEnabled));
	}

	public List<SetInfo> listRepositories() throws ISLookUpException {
		final String xQuery = "for $x in collection('/db/DRIVER/RepositoryServiceResources/RepositoryServiceResourceType') return concat($x//RESOURCE_IDENTIFIER/@value/string(),'|-$-|',$x//OFFICIAL_NAME/string(),'|-$-|',$x//ENGLISH_NAME/string())";

		final List<SetInfo> response = new ArrayList<SetInfo>();

		for (String item : lookupLocator.getService().quickSearchProfile(xQuery)) {
			String[] repoInfo = item.split("\\|-\\$-\\|");
			if (repoInfo.length != 3) {
				log.fatal("There are missing information in Repository Profile while getting the list of sets ");
			}
			String repoId = repoInfo[0];
			if (StringUtils.isBlank(repoId)) {
				log.fatal("Missing repository id: can't generate set");
			} else {
				String repoOfficialName = repoInfo[1];
				String repoEnglishName = repoInfo[2];
				boolean hasName = true;
				if (StringUtils.isBlank(repoOfficialName) && StringUtils.isBlank(repoEnglishName)) {
					hasName = false;
					log.fatal("Missing official or english name in repository profile " + repoId);
				}

				final SetInfo set = new SetInfo();
				set.setSetSpec(repoId.substring(0, repoId.indexOf("_")));
				if (hasName) {
					String repoName = StringUtils.join(Lists.newArrayList(repoOfficialName, repoEnglishName), " - ");
					set.setSetName(repoName);
					set.setSetDescription("This set contains metadata originally harvested from the repository " + repoName);
				}
				response.add(set);
			}
		}
		return response;
	}

	public UnaryFunction<String, String> getUnaryFunctionFromTDSRule(final String tdsRuleId) {
		final String queryCodeRule = "//RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value='" + tdsRuleId
				+ "']/BODY/CONFIGURATION/SCRIPT/CODE/*[local-name()='stylesheet']";
		String xsltCode = null;
		try {
			xsltCode = this.lookupLocator.getService().getResourceProfileByQuery(queryCodeRule);
		} catch (ISLookUpDocumentNotFoundException e) {
			log.fatal("Unexisting TDSRule profile with identifier " + tdsRuleId);
			throw new OaiPublisherRuntimeException(e);
		} catch (ISLookUpException e) {
			throw new OaiPublisherRuntimeException(e);
		}
		if (StringUtils.isBlank(xsltCode)) {
			log.warn("Unexpected blank stylesheet in TDSRule profile with id: " + tdsRuleId + ". Returning identity function.");
			return new IdentityFunction<String>();
		} else return new ApplyXslt(xsltCode);
	}

	public ServiceLocator<ISLookUpService> getLookupLocator() {
		return lookupLocator;
	}

	public void setLookupLocator(final ServiceLocator<ISLookUpService> lookupLocator) {
		this.lookupLocator = lookupLocator;
	}

	public PublisherConfigurationReader getConfiguration() {
		return configuration;
	}

	public void setConfiguration(final PublisherConfigurationReader configuration) {
		this.configuration = configuration;
	}

}
