package eu.dnetlib.data.information.oai.publisher.conf;

import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;

import com.google.common.collect.Lists;

import eu.dnetlib.data.information.oai.publisher.OaiPublisherRuntimeException;
import eu.dnetlib.data.information.oai.publisher.info.MDFInfo;
import eu.dnetlib.enabling.tools.AbstractSchedulable;

public class SchedulableOAIInitializer extends AbstractSchedulable {

	private static final Log log = LogFactory.getLog(SchedulableOAIInitializer.class); // NOPMD by marko on 11/24/08 5:02 PM
	/**
	 * OAI Publisher configuration.
	 */
	@Resource
	private PublisherConfigurationReader configuration;

	private OAIStoreInitializerFactory initializerFactory;

	private int maxInitThreads = 2;
	private boolean forceContentSynchronization;

	@Override
	protected void doExecute() {
		log.info("Initializing OAI Publisher stores from its configuration profile with initializer from factory: "
				+ initializerFactory.getClass().getCanonicalName());
		// starting new thread to avoid the "neverending" reload of the inspector page.
		new Thread() {

			@Override
			public void run() {
				checkStores();
			}

		}.start();
		log.info("OAI Publisher configuration updated.");
	}

	protected void checkStores() {
		final ExecutorService executor = Executors.newFixedThreadPool(maxInitThreads);
		List<MDFInfo> metadataFormats = this.configuration.getMetadataFormatInfo();
		List<Future<?>> futures = Lists.newArrayList();
		for (MDFInfo mdf : metadataFormats) {
			futures.add(executor.submit(initializerFactory.create(mdf, this.forceContentSynchronization)));
		}
		executor.shutdown();
		for (Future<?> future : futures) {
			try {
				future.get();
			} catch (InterruptedException e) {
				throw new OaiPublisherRuntimeException(e);
			} catch (ExecutionException e) {
				throw new OaiPublisherRuntimeException(e);
			}
		}
	}

	public OAIStoreInitializerFactory getInitializerFactory() {
		return initializerFactory;
	}

	@Required
	public void setInitializerFactory(final OAIStoreInitializerFactory initializerFactory) {
		this.initializerFactory = initializerFactory;
	}

	public int getMaxInitThreads() {
		return maxInitThreads;
	}

	public void setMaxInitThreads(final int maxInitThreads) {
		this.maxInitThreads = maxInitThreads;
	}

	public PublisherConfigurationReader getConfiguration() {
		return configuration;
	}

	public void setConfiguration(final PublisherConfigurationReader configuration) {
		this.configuration = configuration;
	}

	public boolean isForceContentSynchronization() {
		return forceContentSynchronization;
	}

	@Required
	public void setForceContentSynchronization(final boolean forceContentSynchronization) {
		this.forceContentSynchronization = forceContentSynchronization;
	}

}
