package eu.dnetlib.data.information.oai.publisher.core;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.DocumentException;

import com.google.common.collect.Lists;
import com.sun.xml.messaging.saaj.util.Base64;

import eu.dnetlib.data.information.oai.publisher.OaiPublisherException;
import eu.dnetlib.data.information.oai.publisher.OaiPublisherRuntimeException;
import eu.dnetlib.data.information.oai.publisher.conf.ISLookUpClient;
import eu.dnetlib.data.information.oai.publisher.conf.PublisherConfigurationReader;
import eu.dnetlib.data.information.oai.publisher.info.ListDocumentsInfo;
import eu.dnetlib.data.information.oai.publisher.info.MDFInfo;
import eu.dnetlib.data.information.oai.publisher.info.RecordInfo;
import eu.dnetlib.data.information.oai.publisher.info.SetInfo;
import eu.dnetlib.data.information.oai.publisher.sets.ConfigurationSetCollection;
import eu.dnetlib.data.information.oai.publisher.store.Cursor;
import eu.dnetlib.data.information.oai.publisher.store.PublisherStore;
import eu.dnetlib.data.information.oai.publisher.store.PublisherStoreDAO;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.miscutils.functional.UnaryFunction;

public class DNetOAICore extends AbstractOAICore {

	private static final Log log = LogFactory.getLog(DNetOAICore.class); // NOPMD by marko on 11/24/08 5:02 PM
	public static final String REPO_BASE64_SUFFIX = new String(Base64.encode("RepositoryServiceResources/RepositoryServiceResourceType".getBytes()));
	@Resource
	private ISLookUpClient lookupClient;

	/**
	 * set collection loaded from the publisher configuration file.
	 */
	@Resource
	private ConfigurationSetCollection configurationSetCollection;
	@Resource
	private PublisherStoreDAO<PublisherStore<Cursor>, Cursor> publisherStoreDAO;

	private String defaultDate = "2008-01-01T12:00:00Z";

	private int pageSize = 100;

	/**
	 * 
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.data.information.oai.publisher.core.AbstractOAICore#getRecordById(eu.dnetlib.data.information.oai.publisher.info.MDFInfo,
	 *      java.lang.String, java.lang.String)
	 */
	@Override
	protected RecordInfo getRecordById(final MDFInfo mdf, final String setSpec, final String id) throws OaiPublisherException {
		final String recordID = (StringUtils.isBlank(setSpec)) ? id : setSpec + "_" + REPO_BASE64_SUFFIX + "::" + id;
		PublisherStore<Cursor> store = this.publisherStoreDAO.getStoreFor(mdf.getPrefix());
		if (store == null)
			throw new OaiPublisherRuntimeException("Missing store for metadata prefix " + mdf.getPrefix() + ". Please check OAI publisher configuration.");
		RecordInfo record = null;
		if (StringUtils.isBlank(mdf.getTransformationRuleID())) {
			record = store.getRecord(recordID);
		} else {
			UnaryFunction<String, String> function = this.lookupClient.getUnaryFunctionFromTDSRule(mdf.getTransformationRuleID());
			record = store.getRecord(recordID, function);
		}
		if (record != null) {
			record.setPrefix(mdf.getPrefix());
		}
		return record;
	}

	/**
	 * 
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.data.information.oai.publisher.core.AbstractOAICore#listSets()
	 */
	@Override
	public List<SetInfo> listSets() throws OaiPublisherException {
		try {
			final List<SetInfo> sets = new ArrayList<SetInfo>();
			sets.addAll(lookupClient.listRepositories());
			sets.addAll(configurationSetCollection.getAllSets());
			return sets;
		} catch (ISLookUpException e) {
			throw new OaiPublisherException("Error executing listSets", e);
		}
	}

	/**
	 * List metadata formats.
	 * 
	 * @return
	 * 
	 * @return list of metadata formats
	 * @throws DocumentException
	 *             could happen
	 * @throws ISLookUpException
	 *             could happen
	 * @throws OaiPublisherException
	 *             could happen
	 * @throws IOException
	 *             could happen
	 */
	@Override
	public List<MDFInfo> listMetadataFormats() throws OaiPublisherException {
		try {
			return lookupClient.listMetadataFormats(true);
		} catch (ISLookUpException e) {
			throw new OaiPublisherException("Error invoking lookup", e);
		} catch (DocumentException e) {
			throw new OaiPublisherException("Error in metadata formats xml profile", e);
		}
	}

	/**
	 * 
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.data.information.oai.publisher.core.AbstractOAICore#getDocuments(boolean, java.lang.String, java.lang.String,
	 *      java.lang.String, java.lang.String)
	 */
	@Override
	protected ListDocumentsInfo getDocuments(final boolean onlyIdentifiers, final String set, final String metadataPrefix, final String from, final String until)
			throws OaiPublisherException {
		try {
			MDFInfo mdf = obtainMDFInfo(metadataPrefix);
			String query = this.generateQuery(mdf, set, from, until);
			Cursor results = this.getCursor(query, onlyIdentifiers, mdf);
			log.debug("Total number of records: " + results.count());
			ListDocumentsInfo res = this.prepareListDocumentsInfo(results, mdf, query, set, 0, results.count());
			log.debug("Delivering " + res.getDocs().size() + " in a page");
			return res;
		} catch (Exception e) {
			throw new OaiPublisherException(e);
		}
	}

	@Override
	protected ListDocumentsInfo getDocuments(final boolean onlyIdentifiers, final String resumptionToken) throws OaiPublisherException {
		ResumptionTokenImpl resToken = new ResumptionTokenImpl();
		resToken.deserialize(resumptionToken);

		log.info(resToken.toString());

		MDFInfo mdf = obtainMDFInfo(resToken.getMetadataPrefix());
		String lastID = resToken.getLastObjIdentifier();
		String query = resToken.getQuery();
		String newQuery = query + " AND _id > \"" + lastID + "\"";
		log.debug("NEW QUERY BECAUSE of resumptionToken: " + newQuery);

		Cursor results = this.getCursor(newQuery, onlyIdentifiers, mdf);
		int oldCount = resToken.getnMaxElements();
		int newCount = results.count() + resToken.getnRead();
		// if the number of records changed, then for sure we can invalidate the resumption token.
		if (oldCount != newCount) throw new OaiPublisherException("badResumptionToken");

		ListDocumentsInfo res = this.prepareListDocumentsInfo(results, mdf, query, resToken.getRequestedSet(), resToken.getnRead(), resToken.getnMaxElements());
		res.setCursor(resToken.getnRead());
		return res;
	}

	@Override
	protected MDFInfo obtainMDFInfo(final String metadataPrefix) throws OaiPublisherException {
		MDFInfo mdf = this.configurationSetCollection.getConfiguration().getMetadataFormatInfo(metadataPrefix);
		if (mdf == null) throw new OaiPublisherException("Invalid metadataPrefix " + metadataPrefix);
		return mdf;

	}

	protected ListDocumentsInfo prepareListDocumentsInfo(final Cursor results,
			final MDFInfo mdf,
			final String query,
			final String requestedSet,
			final int read,
			final int totalNumber) throws OaiPublisherException {
		ListDocumentsInfo documentList = new ListDocumentsInfo();
		documentList.setnMaxElements(totalNumber);
		documentList.setMetadataPrefix(mdf.getPrefix());
		documentList.setCursor(0);
		List<RecordInfo> theRecords = this.generateOAIRecords(mdf, requestedSet, results);
		if (documentList.getnMaxElements() > 0) {
			documentList.setDocs(theRecords);
		} else throw new OaiPublisherException("noRecordsMatch");

		if ((theRecords == null) || (theRecords.isEmpty())) throw new OaiPublisherException("noRecordsMatch: 'documents' is null or empty");

		if (documentList.getnMaxElements() > (read + theRecords.size())) {
			String lastID = theRecords.get(theRecords.size() - 1).getInternalId();
			ResumptionTokenImpl nextToken = new ResumptionTokenImpl();
			nextToken.setLastObjIdentifier(lastID);
			nextToken.setMetadataPrefix(mdf.getPrefix());
			nextToken.setnMaxElements(totalNumber);
			nextToken.setnRead(read + theRecords.size());
			nextToken.setQuery(query);
			nextToken.setRequestedSet(requestedSet);
			documentList.setResumptionToken(nextToken);
		}

		return documentList;
	}

	protected Cursor getCursor(final String query, final boolean onlyIdentifiers, final MDFInfo mdfInfo) {
		PublisherStore<Cursor> store = this.publisherStoreDAO.getStore(mdfInfo.getSourceFormatName(), mdfInfo.getSourceFormatInterpretation(),
				mdfInfo.getSourceFormatLayout());
		if (store == null)
			throw new OaiPublisherRuntimeException("Missing store for metadata prefix " + mdfInfo.getPrefix() + ". Please check OAI publisher configuration.");
		Cursor results = null;
		if (StringUtils.isBlank(mdfInfo.getTransformationRuleID())) {
			results = store.getRecords(query, !onlyIdentifiers);
		} else {
			UnaryFunction<String, String> function = this.lookupClient.getUnaryFunctionFromTDSRule(mdfInfo.getTransformationRuleID());
			results = store.getRecords(query, function, !onlyIdentifiers);
		}
		return results;
	}

	/**
	 * Generates the List of RecordInfo to be delivered.
	 * 
	 * @param onlyIdentifiers
	 *            boolean, true if the request is a ListIdentifiers. false if it is a ListRecords.
	 * @param mdf
	 *            MDFInfo, the requested metadata format information.
	 * @param requestedSet
	 *            set specified in the request. It is blank if no set was requested.
	 * @param cursor
	 *            Cursor instance to use to get the records.
	 * @return List of RecordInfo instances
	 */
	protected List<RecordInfo> generateOAIRecords(final MDFInfo mdf, final String requestedSet, final Cursor cursor) {
		final List<RecordInfo> documents = Lists.newArrayList();
		Iterator<RecordInfo> cursorIterator = cursor.iterator();
		int i = 0;
		while (cursorIterator.hasNext() && (i < this.pageSize)) {
			RecordInfo current = cursorIterator.next();
			current.addSetspec(requestedSet);
			current.setPrefix(mdf.getPrefix());
			documents.add(current);
			i++;
		}
		return documents;
	}

	protected String generateQuery(final MDFInfo mdf, final String set, final String from, final String until) {
		String datestampIndexName = PublisherConfigurationReader.DATESTAMP_FIELD;

		String query = mdf.getBaseQuery();
		if (set != null) {
			if (query.length() > 0) {
				query += " AND ";
			}
			query += obtainQuerySet(set);
		}
		if ((from != null) && (until != null)) {
			if (query.length() > 0) {
				query += " AND ";
			}
			query += datestampIndexName + " >= " + from + " AND " + datestampIndexName + " <= " + until;
		} else if (from != null) {
			if (query.length() > 0) {
				query += " AND ";
			}
			query += datestampIndexName + " >= " + from;
		} else if (until != null) {
			if (query.length() > 0) {
				query += " AND ";
			}
			query += datestampIndexName + " <= " + until;
		}
		log.info("QUERY GENERATED: \n" + query);
		return query;
	}

	/**
	 * Obtain a query based on a set name.
	 * 
	 * @param set
	 *            set name
	 * @return query to obtain documents in the given set
	 */
	private String obtainQuerySet(final String set) {
		String query = "(" + PublisherConfigurationReader.SET_FIELD + " = \"" + set + "_" + REPO_BASE64_SUFFIX + "\")";
		if (this.configurationSetCollection.containSet(set)) {
			query = this.configurationSetCollection.getSetQuery(set);
		}
		return query;
	}

	public void setLookupClient(final ISLookUpClient lookupClient) {
		this.lookupClient = lookupClient;
	}

	public String getDefaultDate() {
		return defaultDate;
	}

	public void setDefaultDate(final String defaultDate) {
		this.defaultDate = defaultDate;
	}

	public ConfigurationSetCollection getConfigurationSetCollection() {
		return configurationSetCollection;
	}

	public void setConfigurationSetCollection(final ConfigurationSetCollection configurationSetCollection) {
		this.configurationSetCollection = configurationSetCollection;
	}

	public PublisherStoreDAO<PublisherStore<Cursor>, Cursor> getPublisherStoreDAO() {
		return publisherStoreDAO;
	}

	public void setPublisherStoreDAO(final PublisherStoreDAO<PublisherStore<Cursor>, Cursor> publisherStoreDAO) {
		this.publisherStoreDAO = publisherStoreDAO;
	}

	public int getPageSize() {
		return pageSize;
	}

	public void setPageSize(final int pageSize) {
		this.pageSize = pageSize;
	}

}
