package eu.dnetlib.data.information.oai.publisher.store;

import java.io.StringReader;
import java.util.Iterator;

import org.dom4j.DocumentException;
import org.dom4j.io.SAXReader;

import com.mongodb.DBCursor;
import com.mongodb.DBObject;

import eu.dnetlib.data.information.oai.publisher.OaiPublisherRuntimeException;
import eu.dnetlib.data.information.oai.publisher.core.MetadataExtractor;
import eu.dnetlib.data.information.oai.publisher.info.RecordInfo;
import eu.dnetlib.miscutils.functional.UnaryFunction;

public class MongoCursor implements Cursor {

	/**
	 * Underlying mongo cursor.
	 */
	private DBCursor dbCursor;
	/**
	 * Function to apply to records before delivering.
	 */
	private UnaryFunction<String, String> function;

	/**
	 * true if the RecordInfo returned by this Cursor must include the record body, false otherwise.
	 */
	private boolean bodyIncluded;

	private RecordInfoGenerator recordInfoGenerator;
	private MetadataExtractor metadataExtractor;

	/**
	 * 
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.data.information.oai.publisher.store.Cursor#count()
	 */
	@Override
	public int count() {
		return this.dbCursor.count();
	}

	/**
	 * 
	 * {@inheritDoc}
	 * 
	 * @see java.lang.Iterable#iterator()
	 */
	@Override
	public Iterator<RecordInfo> iterator() {

		return new Iterator<RecordInfo>() {

			@Override
			public boolean hasNext() {
				return dbCursor.hasNext();
			}

			@Override
			public RecordInfo next() {
				try {
					DBObject res = dbCursor.next();
					RecordInfo info = recordInfoGenerator.transformDBObject(res, bodyIncluded);
					if ((function != null) && bodyIncluded) {
						String body = metadataExtractor.evaluate(new SAXReader().read(new StringReader(info.getMetadata())));
						info.setMetadata(function.evaluate(body));
					}
					return info;
				} catch (DocumentException e) {
					throw new OaiPublisherRuntimeException(e);
				}
			}

			@Override
			public void remove() {
				throw new UnsupportedOperationException();
			}

		};
	}

	public MongoCursor() {
		super();
	}

	public MongoCursor(final DBCursor dbCursor, final boolean bodyIncluded, final RecordInfoGenerator recordInfoGenerator,
			final MetadataExtractor metadataExtractor) {
		this(dbCursor, null, bodyIncluded, recordInfoGenerator, metadataExtractor);
	}

	public MongoCursor(final DBCursor dbCursor, final UnaryFunction<String, String> function, final boolean bodyIncluded,
			final RecordInfoGenerator recordInfoGenerator, final MetadataExtractor metadataExtractor) {
		super();
		this.dbCursor = dbCursor;
		this.function = function;
		this.bodyIncluded = bodyIncluded;
		this.recordInfoGenerator = recordInfoGenerator;
		this.metadataExtractor = metadataExtractor;
	}

	public UnaryFunction<String, String> getFunction() {
		return function;
	}

	public void setFunction(final UnaryFunction<String, String> function) {
		this.function = function;
	}

	public DBCursor getDbCursor() {
		return dbCursor;
	}

	public void setDbCursor(final DBCursor dbCursor) {
		this.dbCursor = dbCursor;
	}

	@Override
	public boolean isBodyIncluded() {
		return this.bodyIncluded;
	}

	@Override
	public void setBodyIncluded(final boolean bodyIncluded) {
		this.bodyIncluded = bodyIncluded;
	}

	public RecordInfoGenerator getRecordInfoGenerator() {
		return recordInfoGenerator;
	}

	public void setRecordInfoGenerator(final RecordInfoGenerator recordInfoGenerator) {
		this.recordInfoGenerator = recordInfoGenerator;
	}

	public MetadataExtractor getMetadataExtractor() {
		return metadataExtractor;
	}

	public void setMetadataExtractor(final MetadataExtractor metadataExtractor) {
		this.metadataExtractor = metadataExtractor;
	}

}
