package eu.dnetlib.data.information.oai.publisher.store;

import java.io.IOException;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.bson.types.ObjectId;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;
import org.z3950.zing.cql.CQLAndNode;
import org.z3950.zing.cql.CQLBooleanNode;
import org.z3950.zing.cql.CQLNode;
import org.z3950.zing.cql.CQLNotNode;
import org.z3950.zing.cql.CQLOrNode;
import org.z3950.zing.cql.CQLParseException;
import org.z3950.zing.cql.CQLParser;
import org.z3950.zing.cql.CQLTermNode;

import com.google.common.collect.Lists;
import com.mongodb.BasicDBObject;
import com.mongodb.BasicDBObjectBuilder;
import com.mongodb.DBObject;

import eu.dnetlib.data.information.oai.publisher.OaiPublisherRuntimeException;
import eu.dnetlib.data.information.oai.publisher.conf.PublisherConfigurationReader;
import eu.dnetlib.functionality.index.parse.Relation;
import eu.dnetlib.functionality.index.parse.Relations;

/**
 * Instances of this class parse query string into mongo DBObject query.
 * 
 * @author alessia
 * 
 */
public class MongoQueryParser {

	private static final Log log = LogFactory.getLog(MongoQueryParser.class); // NOPMD by marko on 11/24/08 5:02 PM

	/**
	 * Parses the given query string into a mongo DBObject.
	 * 
	 * @param query
	 *            String to parse
	 * @return DBObject corresponding to the query string
	 */
	public DBObject parse(final String query) {
		log.info("PARSING: " + query);
		DBObject parsed = toMongo(query);
		log.info(parsed);
		return parsed;
	}

	private DBObject toMongo(final String query) {
		CQLParser parser = new CQLParser();
		CQLNode root;
		try {
			root = parser.parse(query);
			return this.toMongo(root);
		} catch (CQLParseException e) {
			throw new OaiPublisherRuntimeException(e);
		} catch (IOException e) {
			throw new OaiPublisherRuntimeException(e);
		}
	}

	private DBObject toMongo(final CQLNode node) {
		if (node instanceof CQLTermNode) return doTranslate((CQLTermNode) node);
		if (node instanceof CQLBooleanNode) return doTranslate((CQLBooleanNode) node);

		throw new RuntimeException("error choice for CQLNode " + node.getClass());
	}

	private DBObject doTranslate(final CQLTermNode termNode) {
		if (termNode.getTerm().equals("*")) return new BasicDBObject();
		String relation = termNode.getRelation().getBase();
		Relation rel = Relations.get(relation);
		return this.handleRelationNode(rel, termNode);
	}

	private DBObject handleRelationNode(final Relation rel, final CQLTermNode termNode) {
		DBObject mongoQueryObject = new BasicDBObject();
		String term = termNode.getTerm();
		String indexName = termNode.getIndex();
		Object termObj = term;
		if (indexName.equals("_id")) {
			termObj = new ObjectId(term);
		} else {
			if (indexName.equals(PublisherConfigurationReader.DATESTAMP_FIELD) || indexName.equals(PublisherConfigurationReader.LAST_COLLECTION_DATE_FIELD)) {
				// termObj = this.parseDate(term);
				OAIDate termDate = this.parseDate(term);
				return handleDateRelationNode(indexName, rel, termDate);
			}
		}
		switch (rel) {
		case EQUAL:
		case EXACT:
			mongoQueryObject.put(indexName, termObj);
			break;
		case NOT:
			mongoQueryObject.put(indexName, new BasicDBObject("$ne", termObj));
			break;
		case GT:
			mongoQueryObject.put(indexName, new BasicDBObject("$gt", termObj));
			break;
		case GTE:
			mongoQueryObject.put(indexName, new BasicDBObject("$gte", termObj));
			break;
		case LT:
			mongoQueryObject.put(indexName, new BasicDBObject("$lt", termObj));
			break;
		case LTE:
			mongoQueryObject.put(indexName, new BasicDBObject("$lte", termObj));
			break;
		default:
			throw new OaiPublisherRuntimeException("Can't parse query: relation " + rel + " not supported!");
		}
		return mongoQueryObject;
	}

	/**
	 * The construction of the query changes based on the granularity of the date to handle.
	 * <p>
	 * If the date has yyyy-MM-ddThh:mm:ssZ granularity we have to create a range query because in mongo we have milliseconds, hence an
	 * exact match will return nothing.
	 * </p>
	 * <p>
	 * If the date has yyyy-MM-dd granularity then we have to trick the query. If we are interested in the date 2013-10-28, the date has
	 * been converted into 2013-10-28T00:00:00Z : if we ask for datestamp = 2013-10-28T00:00:00Z, we'll get nothing: we have to ask for
	 * records whose day is the one specified by the date.
	 * 
	 * </p>
	 * 
	 * @param indexName
	 * @param rel
	 * @param date
	 * @return
	 */
	private DBObject handleDateRelationNode(final String indexName, final Relation rel, final OAIDate date) {
		DBObject mongoQueryObject = new BasicDBObject();
		DateTime fromDate = date.date;
		switch (rel) {
		case EQUAL:
		case EXACT:
			if (date.onlyDate) {
				DateTime endDate = date.date.plusDays(1);
				mongoQueryObject.put(indexName, BasicDBObjectBuilder.start("$gte", fromDate.toDate()).append("$lt", endDate.toDate()).get());
			} else {
				DateTime endDate = date.date.plusSeconds(1);
				mongoQueryObject.put(indexName, BasicDBObjectBuilder.start("$gte", fromDate.toDate()).append("$lt", endDate.toDate()).get());
			}
			break;
		case NOT:
			mongoQueryObject.put(indexName, new BasicDBObject("$ne", fromDate.toDate()));
			break;
		case GT:
			mongoQueryObject.put(indexName, new BasicDBObject("$gt", fromDate.toDate()));
			break;
		case GTE:
			mongoQueryObject.put(indexName, new BasicDBObject("$gte", fromDate.toDate()));
			break;
		case LT:
			mongoQueryObject.put(indexName, new BasicDBObject("$lt", fromDate.toDate()));
			break;
		case LTE:
			mongoQueryObject.put(indexName, new BasicDBObject("$lte", fromDate.toDate()));
			break;
		default:
			throw new OaiPublisherRuntimeException("Can't parse query: relation " + rel + " not supported!");
		}
		return mongoQueryObject;
	}

	private DBObject doTranslate(final CQLBooleanNode node) {
		if (node instanceof CQLAndNode) return getBooleanQuery("$and", node);
		if (node instanceof CQLOrNode) return getBooleanQuery("$or", node);
		if (node instanceof CQLNotNode) return getNotQuery((CQLNotNode) node);
		throw new RuntimeException("error choice for CQLBooleanNode " + node.getClass());
	}

	private DBObject getBooleanQuery(final String mongoOperator, final CQLBooleanNode node) {
		DBObject left = this.toMongo(node.left);
		DBObject right = this.toMongo(node.right);
		BasicDBObject andQuery = new BasicDBObject();
		List<DBObject> termList = Lists.newArrayList(left, right);
		andQuery.put(mongoOperator, termList);
		return andQuery;
	}

	private DBObject getNotQuery(final CQLNotNode node) {
		DBObject left = this.toMongo(node.left);
		DBObject right = this.toMongo(node.right);
		DBObject notRight = new BasicDBObject("$not", right);
		BasicDBObject andQuery = new BasicDBObject();
		List<DBObject> termList = Lists.newArrayList(left, notRight);
		andQuery.put("$and", termList);
		return andQuery;
	}

	private OAIDate parseDate(final String date) {
		DateTimeFormatter dateNoTimeFormatter = DateTimeFormat.forPattern("yyyy-MM-dd").withZoneUTC();
		DateTimeFormatter iso8601NoMsTimeFormatter = DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ssZ").withZoneUTC();
		DateTimeFormatter iso8601Formatter = ISODateTimeFormat.dateTime().withZoneUTC();
		OAIDate res = null;
		try {
			log.debug("Using default " + iso8601Formatter.getClass());
			DateTime dt = iso8601Formatter.parseDateTime(date);
			res = new OAIDate(dt, false);
		} catch (Exception e) {
			try {
				log.debug("Switching to ISO with no millisecond date formatter: yyyy-MM-dd'T'HH:mm:ssZ");
				DateTime dt = iso8601NoMsTimeFormatter.parseDateTime(date);
				res = new OAIDate(dt, false);
			} catch (Exception ex) {
				log.debug("Switching to simple date formatter: yyyy-MM-dd");
				DateTime dt = dateNoTimeFormatter.parseDateTime(date);
				res = new OAIDate(dt, true);
			}
		}
		return res;
	}

	class OAIDate {

		DateTime date;
		boolean onlyDate;

		OAIDate(final DateTime date, final boolean onlyDate) {
			this.date = date;
			this.onlyDate = onlyDate;
		}

	}

}
