package eu.dnetlib.data.information.oai.publisher.conf.sync;

import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.collect.Lists;

import eu.dnetlib.data.information.oai.publisher.OaiPublisherRuntimeException;
import eu.dnetlib.data.information.oai.publisher.conf.OAIConfigurationReader;
import eu.dnetlib.data.information.oai.publisher.info.MDFInfo;
import eu.dnetlib.data.information.oai.publisher.info.SetInfo;
import eu.dnetlib.data.information.oai.publisher.sets.MongoSetCollection;
import eu.dnetlib.enabling.tools.AbstractSchedulable;

/**
 * When notified, it updates the OAI store counters based on the current configuration profiles. It updates counters for all sets, also
 * those inferred by the content (values in the field "set" of oai stores).
 * 
 * @author alessia
 * 
 */
public class SchedulableOAIStoreAllCounters extends AbstractSchedulable {

	private static final Log log = LogFactory.getLog(SchedulableOAIStoreAllCounters.class); // NOPMD by marko on 11/24/08 5:02 PM

	@Resource(name = "oaiConfigurationExistReader")
	private OAIConfigurationReader configurationReader;
	@Resource
	private MongoSetCollection mongoSetCollection;
	@Resource
	private OAISetCounterHelper oaiSetCounterHelper;
	/** Max number of threads to use for the application of the configuration. **/
	private int maxThreads = 2;

	private boolean alreadyStarted = false;

	@Override
	protected void doExecute() {
		// TODO: this is not very safe...RACE CONDITIONS CAN OCCUR.
		if (!alreadyStarted) {
			this.alreadyStarted = true;
			log.debug("*****Dropping and re-creating configuration sets******");
			this.mongoSetCollection.dropConfigurationSets();
			List<SetInfo> oaiConfigSets = this.configurationReader.getSets();
			for (SetInfo setInfo : oaiConfigSets) {
				this.mongoSetCollection.upsertSet(setInfo, true);
			}
			log.info("Starting threads to update OAI store counters");
			List<MDFInfo> metadataFormats = this.configurationReader.getMetadataFormatInfo();
			final ExecutorService executor = Executors.newFixedThreadPool(maxThreads);
			List<Future<?>> futures = Lists.newArrayList();
			for (MDFInfo mdf : metadataFormats) {
				futures.add(executor.submit(new OAIStoreAllCounterUpdater(mdf, oaiSetCounterHelper)));
			}
			executor.shutdown();

			// waiting for the store configuration to end
			for (Future<?> future : futures) {
				try {
					future.get();
				} catch (InterruptedException e) {
					throw new OaiPublisherRuntimeException(e);
				} catch (ExecutionException e) {
					throw new OaiPublisherRuntimeException(e);
				}
			}
			log.info("OAI store counters updated...");
			this.alreadyStarted = false;
		} else {
			log.info("The update of counters for all metadata prefix and sets is still ongoing...");
		}

	}

	public OAIConfigurationReader getConfigurationReader() {
		return configurationReader;
	}

	public void setConfigurationReader(final OAIConfigurationReader configurationReader) {
		this.configurationReader = configurationReader;
	}

	public int getMaxThreads() {
		return maxThreads;
	}

	public void setMaxThreads(final int maxThreads) {
		this.maxThreads = maxThreads;
	}

	public MongoSetCollection getMongoSetCollection() {
		return mongoSetCollection;
	}

	public void setMongoSetCollection(final MongoSetCollection mongoSetCollection) {
		this.mongoSetCollection = mongoSetCollection;
	}

	public OAISetCounterHelper getOaiSetCounterHelper() {
		return oaiSetCounterHelper;
	}

	public void setOaiSetCounterHelper(final OAISetCounterHelper oaiSetCounterHelper) {
		this.oaiSetCounterHelper = oaiSetCounterHelper;
	}

}
