package eu.dnetlib.data.information.oai.publisher.wf;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.googlecode.sarasvati.Engine;
import com.googlecode.sarasvati.NodeToken;

import eu.dnetlib.data.information.oai.publisher.store.MongoPublisherStore;
import eu.dnetlib.data.information.oai.publisher.store.MongoPublisherStoreDAO;
import eu.dnetlib.enabling.tools.SplittedQueryExecutor;
import eu.dnetlib.workflow.AbstractJobNode;

/**
 * Finds the publisher store based on the metadata format, layout and interpretation of the mdstore whose identifier is in the env.
 * <p>
 * The publisher store id is then set as an attribute of the env. The name of the target attribute can be configured via property.
 * </p>
 * 
 * @author alessia
 * 
 */
public class FindPublisherStoreJob extends AbstractJobNode {

	private static final Log log = LogFactory.getLog(FindPublisherStoreJob.class); // NOPMD by marko on 11/24/08 5:02 PM

	/**
	 * Name of the env attribute containing the mdstore, publisherstore and the repository identifiers.
	 */
	private String mdstoreIDAttributeName = "mdId";
	private String pubstoreIDAttributeName = "pubstoreId";
	private String repositoryIDAttributeName = "repositoryId";

	/**
	 * Bean for the execution of XQuery returning concatenated values.
	 */
	@Resource
	private SplittedQueryExecutor queryExecutor;

	/**
	 * Publisher store DAO based on mongodb.
	 */
	@Resource
	private MongoPublisherStoreDAO publisherStoreDAO;

	@Override
	public void execute(final Engine engine, final NodeToken token) {
		try {
			final String mdstoreID = token.getFullEnv().getAttribute(this.mdstoreIDAttributeName);
			final String repoID = token.getFullEnv().getAttribute(this.repositoryIDAttributeName);
			token.getEnv().setAttribute(this.repositoryIDAttributeName, repoID);

			final String xquery = "let $x := //RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value='" + mdstoreID + "']//CONFIGURATION "
					+ "return concat($x/METADATA_FORMAT/text(), ':=:', $x/METADATA_FORMAT_INTERPRETATION/text(), ':=:', $x/METADATA_FORMAT_LAYOUT/text())";
			String mdfInfo = this.queryExecutor.queryFirst(xquery);
			if (mdfInfo != null) {
				String[] infoSplitted = mdfInfo.split(":=:");
				MongoPublisherStore store = this.publisherStoreDAO.getStore(infoSplitted[0], infoSplitted[1], infoSplitted[2]);
				if (store == null) {
					log.fatal("Publisher store for " + infoSplitted[0] + " " + infoSplitted[1] + " " + infoSplitted[2] + " does not exist!");
					engine.complete(token, "notFound");
				} else {
					log.debug("Found Publisher store for " + infoSplitted[0] + " " + infoSplitted[1] + " " + infoSplitted[2]);
					token.getEnv().setAttribute(pubstoreIDAttributeName, store.getId());
					engine.complete(token, "found");
				}
			} else {
				token.getEnv().setAttribute("hasFailed", true);
				token.getEnv().setAttribute("errorMessage", "cannot find metadata format properties for mdstore with id " + mdstoreID);
				engine.complete(token, "failed");

			}
		} catch (final Exception e) {
			this.failed(engine, token, e);
		}

	}

	public String getMdstoreIDAttributeName() {
		return mdstoreIDAttributeName;
	}

	public void setMdstoreIDAttributeName(final String mdstoreIDAttributeName) {
		this.mdstoreIDAttributeName = mdstoreIDAttributeName;
	}

	public SplittedQueryExecutor getQueryExecutor() {
		return queryExecutor;
	}

	public void setQueryExecutor(final SplittedQueryExecutor queryExecutor) {
		this.queryExecutor = queryExecutor;
	}

	public MongoPublisherStoreDAO getPublisherStoreDAO() {
		return publisherStoreDAO;
	}

	public void setPublisherStoreDAO(final MongoPublisherStoreDAO publisherStoreDAO) {
		this.publisherStoreDAO = publisherStoreDAO;
	}

	public String getPubstoreIDAttributeName() {
		return pubstoreIDAttributeName;
	}

	public void setPubstoreIDAttributeName(final String pubstoreIDAttributeName) {
		this.pubstoreIDAttributeName = pubstoreIDAttributeName;
	}

	public String getRepositoryIDAttributeName() {
		return repositoryIDAttributeName;
	}

	public void setRepositoryIDAttributeName(final String repositoryIDAttributeName) {
		this.repositoryIDAttributeName = repositoryIDAttributeName;
	}
}
