package eu.dnetlib.data.information.oai.publisher.core;

import java.io.IOException;
import java.util.Iterator;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.DocumentException;

import com.google.common.collect.Lists;

import eu.dnetlib.data.information.oai.publisher.BadResumptionTokenException;
import eu.dnetlib.data.information.oai.publisher.CannotDisseminateFormatException;
import eu.dnetlib.data.information.oai.publisher.NoRecordsMatchException;
import eu.dnetlib.data.information.oai.publisher.OaiPublisherException;
import eu.dnetlib.data.information.oai.publisher.OaiPublisherRuntimeException;
import eu.dnetlib.data.information.oai.publisher.conf.ISLookUpClient;
import eu.dnetlib.data.information.oai.publisher.conf.OAIConfigurationReader;
import eu.dnetlib.data.information.oai.publisher.info.ListDocumentsInfo;
import eu.dnetlib.data.information.oai.publisher.info.MDFInfo;
import eu.dnetlib.data.information.oai.publisher.info.RecordInfo;
import eu.dnetlib.data.information.oai.publisher.info.ResumptionTokenImpl;
import eu.dnetlib.data.information.oai.publisher.info.SetInfo;
import eu.dnetlib.data.oai.store.Cursor;
import eu.dnetlib.data.oai.store.PublisherStore;
import eu.dnetlib.data.oai.store.PublisherStoreDAO;
import eu.dnetlib.data.oai.store.sets.MongoSetCollection;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.miscutils.functional.UnaryFunction;

public class DNetOAICore extends AbstractOAICore {

	private static final Log log = LogFactory.getLog(DNetOAICore.class); // NOPMD by marko on 11/24/08 5:02 PM
	@Resource
	private ISLookUpClient lookupClient;
	@Resource(name = "oaiConfigurationExistReader")
	private OAIConfigurationReader configurationReader;
	@Resource
	private MongoSetCollection setCollection;
	@Resource
	private PublisherStoreDAO<PublisherStore<Cursor>, Cursor> publisherStoreDAO;

	private String defaultDate = "2008-01-01T12:00:00Z";

	private int pageSize = 100;

	@Override
	protected RecordInfo getRecordById(final MDFInfo mdf, final String id) throws OaiPublisherException {
		PublisherStore<Cursor> store = this.publisherStoreDAO.getStoreFor(mdf.getPrefix());
		if (store == null)
			throw new OaiPublisherRuntimeException("Missing store for metadata prefix " + mdf.getPrefix() + ". Please check OAI publisher configuration.");
		RecordInfo record = null;
		if (StringUtils.isBlank(mdf.getTransformationRuleID())) {
			record = store.getRecord(id);
		} else {
			UnaryFunction<String, String> function = this.lookupClient.getUnaryFunctionFromTDSRule(mdf.getTransformationRuleID());
			record = store.getRecord(id, function);
		}
		if (record != null) {
			record.setPrefix(mdf.getPrefix());
		}
		return record;
	}

	/**
	 * 
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.data.information.oai.publisher.core.AbstractOAICore#listSets()
	 */
	@Override
	public List<SetInfo> listSets() {
		return this.setCollection.getAllSets();
	}

	/**
	 * List metadata formats.
	 * 
	 * @return
	 * 
	 * @return list of metadata formats
	 * @throws DocumentException
	 *             could happen
	 * @throws ISLookUpException
	 *             could happen
	 * @throws OaiPublisherException
	 *             could happen
	 * @throws IOException
	 *             could happen
	 */
	@Override
	public List<MDFInfo> listMetadataFormats() {
		return this.configurationReader.getMetadataFormatInfo(true);
	}

	/**
	 * 
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.data.information.oai.publisher.core.AbstractOAICore#getDocuments(boolean, java.lang.String, java.lang.String,
	 *      java.lang.String, java.lang.String)
	 */
	@Override
	protected ListDocumentsInfo getDocuments(final boolean onlyIdentifiers, final String set, final String metadataPrefix, final String from, final String until)
			throws OaiPublisherException {
		MDFInfo mdf = obtainMDFInfo(metadataPrefix);
		boolean hasDateRange = StringUtils.isNotBlank(from) || StringUtils.isNotBlank(until);
		String query = this.generateQuery(mdf, set, from, until, hasDateRange);
		int total = this.countTotal(hasDateRange, query, set, mdf);
		log.debug("Total number of records: " + total);
		Cursor results = this.getCursor(query, onlyIdentifiers, mdf);
		ListDocumentsInfo res = this.prepareListDocumentsInfo(results, mdf, query, set, 0, total, hasDateRange);
		log.debug("Delivering " + res.getDocs().size() + " in a page");
		return res;
	}

	@Override
	protected ListDocumentsInfo getDocuments(final boolean onlyIdentifiers, final String resumptionToken) throws OaiPublisherException {
		ResumptionTokenImpl resToken = new ResumptionTokenImpl();
		resToken.deserialize(resumptionToken);

		log.info(resToken.toString());

		MDFInfo mdf = obtainMDFInfo(resToken.getMetadataPrefix());
		String lastID = resToken.getLastObjIdentifier();
		String query = resToken.getQuery();
		String newQuery = query + " AND _id > \"" + lastID + "\"";
		log.debug("NEW QUERY BECAUSE of resumptionToken: " + newQuery);
		int total = this.countTotal(resToken.hasDateRange(), query, resToken.getRequestedSet(), mdf);
		Cursor results = this.getCursor(newQuery, onlyIdentifiers, mdf);
		int oldCount = resToken.getnMaxElements();
		// if the number of records changed, then for sure we can invalidate the resumption token.
		if (oldCount != total) throw new BadResumptionTokenException(resumptionToken);

		ListDocumentsInfo res = this.prepareListDocumentsInfo(results, mdf, query, resToken.getRequestedSet(), resToken.getnRead(), resToken.getnMaxElements(),
				resToken.hasDateRange());
		res.setCursor(resToken.getnRead());
		return res;
	}

	@Override
	protected MDFInfo obtainMDFInfo(final String metadataPrefix) throws OaiPublisherException {
		// MDFInfo mdf = this.configurationSetCollection.getConfiguration().getMetadataFormatInfo(metadataPrefix);
		MDFInfo mdf = this.configurationReader.getMetadataFormatInfo(metadataPrefix);
		if (mdf == null) throw new CannotDisseminateFormatException("Invalid metadataPrefix " + metadataPrefix);
		return mdf;

	}

	protected ListDocumentsInfo prepareListDocumentsInfo(final Cursor results,
			final MDFInfo mdf,
			final String query,
			final String requestedSet,
			final int read,
			final int totalNumber,
			final boolean hasDateRange) throws OaiPublisherException {
		ListDocumentsInfo documentList = new ListDocumentsInfo();
		documentList.setnMaxElements(totalNumber);
		documentList.setMetadataPrefix(mdf.getPrefix());
		documentList.setCursor(0);
		List<RecordInfo> theRecords = this.generateOAIRecords(mdf, requestedSet, results);
		if (documentList.getnMaxElements() > 0) {
			documentList.setDocs(theRecords);
		} else throw new NoRecordsMatchException("No records match the requested criteria");

		if ((theRecords == null) || (theRecords.isEmpty())) throw new NoRecordsMatchException("noRecordsMatch: 'documents' is null or empty");

		if (documentList.getnMaxElements() > (read + theRecords.size())) {
			String lastID = theRecords.get(theRecords.size() - 1).getInternalId();
			ResumptionTokenImpl nextToken = new ResumptionTokenImpl();
			nextToken.setDateRange(hasDateRange);
			nextToken.setLastObjIdentifier(lastID);
			nextToken.setMetadataPrefix(mdf.getPrefix());
			nextToken.setnMaxElements(totalNumber);
			nextToken.setnRead(read + theRecords.size());
			nextToken.setQuery(query);
			nextToken.setRequestedSet(requestedSet);
			documentList.setResumptionToken(nextToken);
		}

		return documentList;
	}

	protected Cursor getCursor(final String query, final boolean onlyIdentifiers, final MDFInfo mdfInfo) {
		PublisherStore<Cursor> store = this.publisherStoreDAO.getStore(mdfInfo.getSourceFormatName(), mdfInfo.getSourceFormatInterpretation(),
				mdfInfo.getSourceFormatLayout());
		if (store == null)
			throw new OaiPublisherRuntimeException("Missing store for metadata prefix " + mdfInfo.getPrefix() + ". Please check OAI publisher configuration.");
		Cursor results = null;
		if (StringUtils.isBlank(mdfInfo.getTransformationRuleID())) {
			results = store.getRecords(query, !onlyIdentifiers, pageSize);
		} else {
			UnaryFunction<String, String> function = this.lookupClient.getUnaryFunctionFromTDSRule(mdfInfo.getTransformationRuleID());
			results = store.getRecords(query, function, !onlyIdentifiers, pageSize);
		}
		return results;
	}

	/**
	 * Generates the List of RecordInfo to be delivered.
	 * 
	 * @param onlyIdentifiers
	 *            boolean, true if the request is a ListIdentifiers. false if it is a ListRecords.
	 * @param mdf
	 *            MDFInfo, the requested metadata format information.
	 * @param requestedSet
	 *            set specified in the request. It is blank if no set was requested.
	 * @param cursor
	 *            Cursor instance to use to get the records.
	 * @return List of RecordInfo instances
	 */
	protected List<RecordInfo> generateOAIRecords(final MDFInfo mdf, final String requestedSet, final Cursor cursor) {
		final List<RecordInfo> documents = Lists.newArrayList();
		Iterator<RecordInfo> cursorIterator = cursor.iterator();
		while (cursorIterator.hasNext()) {
			RecordInfo current = cursorIterator.next();
			current.addSetspec(requestedSet);
			current.setPrefix(mdf.getPrefix());
			documents.add(current);
		}
		return documents;
	}

	protected String generateQuery(final MDFInfo mdf, final String set, final String from, final String until, final boolean hasDateRange) {
		String datestampIndexName = OAIConfigurationReader.DATESTAMP_FIELD;

		String query = mdf.getBaseQuery();
		if (!StringUtils.isBlank(set)) {
			if (!StringUtils.isBlank(query)) {
				query += " AND ";
			}
			query += this.setCollection.getSetQuery(set);
		}
		if (hasDateRange) {
			if (query.length() > 0) {
				query += " AND ";
			}
			if ((from != null) && (until != null)) {
				query += datestampIndexName + " >= " + from + " AND " + datestampIndexName + " <= " + until;
			} else if (from != null) {
				query += datestampIndexName + " >= " + from;
			} else if (until != null) {

				query += datestampIndexName + " <= " + until;
			}
		}

		log.info("QUERY GENERATED: \n" + query);
		return query;
	}

	private int countTotal(final boolean hasDateRange, final String query, final String set, final MDFInfo mdFormat) {
		PublisherStore<Cursor> store = this.publisherStoreDAO.getStore(mdFormat.getSourceFormatName(), mdFormat.getSourceFormatInterpretation(),
				mdFormat.getSourceFormatLayout());
		int total = 0;
		if (hasDateRange) {
			total = store.count(query);
		} else {
			log.fatal("SET::: " + set);
			if (StringUtils.isNotBlank(set)) {
				total = setCollection.count(set, mdFormat.getPrefix());
			} else {
				total = setCollection.count("ALL", mdFormat.getPrefix());
			}
		}
		return total;
	}

	public void setLookupClient(final ISLookUpClient lookupClient) {
		this.lookupClient = lookupClient;
	}

	public String getDefaultDate() {
		return defaultDate;
	}

	public void setDefaultDate(final String defaultDate) {
		this.defaultDate = defaultDate;
	}

	public PublisherStoreDAO<PublisherStore<Cursor>, Cursor> getPublisherStoreDAO() {
		return publisherStoreDAO;
	}

	public void setPublisherStoreDAO(final PublisherStoreDAO<PublisherStore<Cursor>, Cursor> publisherStoreDAO) {
		this.publisherStoreDAO = publisherStoreDAO;
	}

	public int getPageSize() {
		return pageSize;
	}

	public void setPageSize(final int pageSize) {
		this.pageSize = pageSize;
	}

	public ISLookUpClient getLookupClient() {
		return lookupClient;
	}

	public MongoSetCollection getSetCollection() {
		return setCollection;
	}

	public void setSetCollection(final MongoSetCollection setCollection) {
		this.setCollection = setCollection;
	}

	public OAIConfigurationReader getConfigurationReader() {
		return configurationReader;
	}

	public void setConfigurationReader(final OAIConfigurationReader configurationReader) {
		this.configurationReader = configurationReader;
	}

}
