package eu.dnetlib.enabling.manager.msro.hope.groovy;

import java.util.List;

import javax.annotation.Resource;
import javax.xml.ws.wsaddressing.W3CEndpointReference;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.collect.Lists;
import com.googlecode.sarasvati.Engine;
import com.googlecode.sarasvati.NodeToken;
import com.googlecode.sarasvati.env.Env;

import eu.dnetlib.data.information.DataSourceResolver;
import eu.dnetlib.enabling.is.sn.EPRUtil;
import eu.dnetlib.enabling.resultset.ResultSetRegistry;
import eu.dnetlib.enabling.resultset.client.ResultSetClientFactory;
import eu.dnetlib.enabling.resultset.push.PushResultSet;
import eu.dnetlib.enabling.resultset.push.PushResultSetFactory;
import eu.dnetlib.workflow.AbstractJobNode;
import groovy.lang.GroovyShell;
import groovy.util.GroovyScriptEngine;
import groovy.util.ResourceException;
import groovy.util.ScriptException;

public class GroovyScriptNode extends AbstractJobNode {

	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(GroovyScriptNode.class);
	@Resource
	private ResultSetClientFactory resultSetClientFactory;

	@Resource
	private PushResultSetFactory pushResultSetFactory;
	@Resource
	private DefaultDNetGroovyEnablerFactory groovyEnablerFactory;

	@Resource
	private ResultSetRegistry rsRegistry;
	@Resource
	private DataSourceResolver dataSourceResolver;

	private String outputEprAttributeName = "epr_output";

	private List<String> defaultGroovyClassPath = Lists.newArrayList("/var/lib/hope-scripts", "/var/lib/hope-scripts/eu");

	/**
	 * Loads the groovy scripts needed for the metadata transformation.
	 * <p>
	 * Please modify this method to load every script supporting the transformation to the domain profile.
	 * </p>
	 * 
	 * @param gse
	 *            GroovyScriptEngine instance used to execute groovy code
	 * @throws ResourceException
	 * @throws ScriptException
	 */
	private void loadNeededScripts(GroovyScriptEngine gse) throws ResourceException, ScriptException {
		//TODO: currently supports only DC, EAD, LIDO and MARC(generic, archive visual and library domain profile). Complete with the audio/video when ready.
		gse.loadScriptByName("eu/dnetlib/hope/domains/common/DerivedResource.groovy");
		gse.loadScriptByName("eu/dnetlib/hope/domains/common/OAIHeader.groovy");

		gse.loadScriptByName("eu/dnetlib/hope/domains/archive/EADFields.groovy");
		gse.loadScriptByName("eu/dnetlib/hope/domains/archive/EADTemplateRoots.groovy");

		gse.loadScriptByName("eu/dnetlib/hope/domains/visual/LidoObject.groovy");

		gse.loadScriptByName("eu/dnetlib/hope/domains/library/MarcFields.groovy");
		gse.loadScriptByName("eu/dnetlib/hope/domains/library/MarcTemplateRoots.groovy");

		gse.loadScriptByName("eu/dnetlib/hope/domains/generic/DublinCoreObject.groovy");
	}

	/**
	 * {@inheritDoc}
	 * <p>
	 * Executes a Groovy class implementing the DNetGroovyClass interface. The name of the class is expected to be
	 * passed in the token full env together with the path to the folder where it is contained together with any other
	 * groovy scripts and classes needed for the execution.
	 * </p>
	 * <p>
	 * The input EPR is passed to the DNetGroovyClass instance together with the output push result set which has to be
	 * filled by the Groovy class during the execution.
	 * </p>
	 * <p>
	 * Note that the DnetGroovyClass instance is created here and launched as a Thread without waiting its termination.
	 * </p>
	 * 
	 * @see com.googlecode.sarasvati.mem.MemNode#execute(com.googlecode.sarasvati.Engine,
	 *      com.googlecode.sarasvati.NodeToken)
	 */
	@SuppressWarnings("rawtypes")
	@Override
	public void execute(final Engine engine, final NodeToken token) {
		Env env = token.getEnv();
		try {
			String mainClassName = token.getFullEnv().getAttribute("dnetGroovyClassName");
			String groovyClassPath = token.getFullEnv().getAttribute("groovyClassPath");
			if (groovyClassPath == null) {
				this.failed(engine, token, new RuntimeException("Null groovy classpath is not allowed!"));
			} else {
				if (!this.defaultGroovyClassPath.contains(groovyClassPath))
					this.defaultGroovyClassPath.add(groovyClassPath);
				//getting the datasource to generate the input EPR:
				final String dataSource = env.getAttribute("dataSource");

				final String repositoryId = token.getFullEnv().getAttribute("repositoryId");
				log.debug("datasource to resolve: " + dataSource);
				log.debug("repositoryId: " + repositoryId);
				W3CEndpointReference epr_input = this.dataSourceResolver.resolve(dataSource).retrieve();
				if (mainClassName != null && groovyClassPath != null) {
					log.info("groovy classpath");
					for (String root : this.defaultGroovyClassPath)
						log.info(root);
					W3CEndpointReference epr_out = pushResultSetFactory.createPushResultSet(12000);

					PushResultSet prs = (PushResultSet) this.rsRegistry.getResultSetById(EPRUtil.getResourceIdentifier(epr_out));

					GroovyScriptEngine gse = new GroovyScriptEngine(this.defaultGroovyClassPath.toArray(new String[0]));
					this.loadNeededScripts(gse);
					gse.getGroovyClassLoader().loadClass(mainClassName);
					log.debug("Loaded classes:");
					for (Class c : gse.getGroovyClassLoader().getLoadedClasses())
						log.debug(c.getCanonicalName());
					log.debug("***********************************************");
					GroovyShell groovyShell = new GroovyShell(gse.getGroovyClassLoader());
					RunnableGroovy groovyInstance = (RunnableGroovy) groovyShell.evaluate("new " + mainClassName + "()");
					DNetGroovyEnabler groovyEnabler = this.groovyEnablerFactory.getDnetGroovyEnabler();
					groovyEnabler.setInputEPR(epr_input);
					groovyEnabler.setPushResultSet(prs);
					groovyEnabler.setResultSetClientFactory(resultSetClientFactory);
					groovyEnabler.setRunnableGroovy(groovyInstance);
					groovyEnabler.setRepoId(repositoryId);
					Thread t = new Thread(groovyEnabler);
					t.start();

					token.getEnv().setAttribute(outputEprAttributeName + "_s", epr_out.toString());
					token.getEnv().setTransientAttribute(outputEprAttributeName, epr_out);

				} else {
					log.debug("Groovy transformation not requested, the input epr will be just passed as output epr");
					token.getEnv().setAttribute(outputEprAttributeName + "_s", epr_input.toString());
					token.getEnv().setTransientAttribute(outputEprAttributeName, epr_input);
				}
			}
		} catch (final Throwable e) {
			this.failed(engine, token, e);
		}

		super.execute(engine, token);
	}

	public ResultSetClientFactory getResultSetClientFactory() {
		return resultSetClientFactory;
	}

	public void setResultSetClientFactory(ResultSetClientFactory resultSetClientFactory) {
		this.resultSetClientFactory = resultSetClientFactory;
	}

	public PushResultSetFactory getPushResultSetFactory() {
		return pushResultSetFactory;
	}

	public void setPushResultSetFactory(PushResultSetFactory pushResultSetFactory) {
		this.pushResultSetFactory = pushResultSetFactory;
	}

	public ResultSetRegistry getRsRegistry() {
		return rsRegistry;
	}

	public void setRsRegistry(ResultSetRegistry rsRegistry) {
		this.rsRegistry = rsRegistry;
	}

	public String getOutputEprAttributeName() {
		return outputEprAttributeName;
	}

	public void setOutputEprAttributeName(String outputEprAttributeName) {
		this.outputEprAttributeName = outputEprAttributeName;
	}

	public DefaultDNetGroovyEnablerFactory getGroovyEnablerFactory() {
		return groovyEnablerFactory;
	}

	public void setGroovyEnablerFactory(DefaultDNetGroovyEnablerFactory groovyEnablerFactory) {
		this.groovyEnablerFactory = groovyEnablerFactory;
	}

	public List<String> getDefaultGroovyClassPath() {
		return defaultGroovyClassPath;
	}

	public void setDefaultGroovyClassPath(List<String> defaultGroovyClassPath) {
		this.defaultGroovyClassPath = defaultGroovyClassPath;
	}

}
