package eu.dnetlib.enabling.manager.msro.hope.thumbnails;

import java.util.List;

import javax.annotation.Resource;
import javax.xml.ws.wsaddressing.W3CEndpointReference;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.impl.HttpSolrServer;
import org.apache.solr.client.solrj.response.QueryResponse;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.gson.Gson;
import com.googlecode.sarasvati.Engine;
import com.googlecode.sarasvati.NodeToken;

import eu.dnetlib.enabling.resultset.IterableResultSetFactory;
import eu.dnetlib.workflow.AbstractJobNode;

/**
 * The node queries a Solr instance to get the URLs of files to use as sources for thumbnails. The expected response is
 * in json. Files are selected only if the associated thumbnail URLs reference the machine in the
 * services.msro.thumbnail.host property. This avoid generation of thumbnails when they are already provided.
 * 
 * @author alessia
 * 
 */
public class FindOriginalFiles extends AbstractJobNode {

	private static final Log log = LogFactory.getLog(FindOriginalFiles.class); // NOPMD by marko on 11/24/08 5:02 PM

	/**
	 * Endpoint address of the Solr instance.
	 */
	private String endpoint;
	private String thumbnailHost;
	private String query;
	private int pageSize = 100;
	private final Gson gson = new Gson();

	private final String returnedFields = "derivative2,derivative3url";
	@Resource
	private IterableResultSetFactory iterableResultSetFactory;

	@Override
	public void execute(Engine engine, NodeToken token) {
		query = "derivative3url:http\\:\\/\\/localhost*";
		SolrServer server;
		try {
			int start = 0;
			int processed = -1;
			int totalMatches = 0;
			server = new HttpSolrServer(endpoint);

			SolrQuery solrQuery = new SolrQuery();
			solrQuery.setQuery(query);
			solrQuery.setParam("fl", returnedFields);
			solrQuery.setRows(pageSize);
			Iterable<String> items = Lists.newArrayList();

			while (processed < totalMatches) {
				solrQuery.setStart(start);
				log.debug(solrQuery.toString());
				QueryResponse rsp = server.query(solrQuery);
				totalMatches = (int) rsp.getResults().getNumFound();
				List<String> toDownload = Lists.newArrayList();
				List<ThumbnailInfo> thumbInfos = rsp.getBeans(ThumbnailInfo.class);
				for (ThumbnailInfo t : thumbInfos) {
					toDownload.add(gson.toJson(t));
				}
				items = Iterables.concat(items, toDownload);
				processed += thumbInfos.size();
				start = processed;
			}
			log.debug(processed + " records in the result set for thumbnail generation");
			W3CEndpointReference epr = this.iterableResultSetFactory.createIterableResultSet(items);
			token.getEnv().setAttribute("downloadListEprString", epr.toString());
			token.getEnv().setTransientAttribute("downloadListEpr", epr);
			super.execute(engine, token);
		} catch (SolrServerException e) {
			log.fatal(e.getMessage());
			super.failed(engine, token, e);
		}
	}

	public String getEndpoint() {
		return endpoint;
	}

	public void setEndpoint(String endpoint) {
		this.endpoint = endpoint;
	}

	public String getQuery() {
		return query;
	}

	public void setQuery(String query) {
		this.query = query;
	}

	public String getThumbnailHost() {
		return thumbnailHost;
	}

	public void setThumbnailHost(String thumbnailHost) {
		this.thumbnailHost = thumbnailHost;
	}

	public IterableResultSetFactory getIterableResultSetFactory() {
		return iterableResultSetFactory;
	}

	public void setIterableResultSetFactory(IterableResultSetFactory iterableResultSetFactory) {
		this.iterableResultSetFactory = iterableResultSetFactory;
	}

	public int getPageSize() {
		return pageSize;
	}

	public void setPageSize(int pageSize) {
		this.pageSize = pageSize;
	}

}
