package eu.dnetlib.simplesso;

import java.io.IOException;
import java.util.List;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.security.Authentication;
import org.springframework.security.AuthenticationException;
import org.springframework.security.context.SecurityContextHolder;
import org.springframework.security.providers.anonymous.AnonymousAuthenticationToken;
import org.springframework.security.ui.AbstractProcessingFilter;

public class SimpleSSOAuthenticationFilter extends AbstractProcessingFilter {
	private static final Log log = LogFactory.getLog(SimpleSSOAuthenticationFilter.class); // NOPMD by marko on 11/24/08 5:02 PM

	private boolean enabled = false;

	private SimpleSSOAuthenticationRoleBuilder rolesBuilder;

	/**
	 * List of roles to set for anonymous users when the authentication is disabled.
	 */
	private List<String> anonymousRoles;

	public int getOrder() {
		return 0;
	}

	public void doFilterHttp(HttpServletRequest request, HttpServletResponse response, FilterChain chain) throws IOException, ServletException {

		if (requiresAuthentication(request, response)) {
			if (logger.isDebugEnabled()) {
				logger.debug("Request is to process authentication");
			}

			Authentication authResult;

			try {
				onPreAuthentication(request, response);
				authResult = attemptAuthentication(request);
			} catch (AuthenticationException failed) {
				// Authentication failed
				unsuccessfulAuthentication(request, response, failed);

				return;
			}

			successfulAuthentication(request, response, authResult);

			// DO NOT RETURN if auth is not enabled HACK
			if (enabled)
				return;
		}

		chain.doFilter(request, response);
	}

	public Authentication attemptAuthentication(HttpServletRequest request) throws AuthenticationException {
		Authentication authRequest;
		if (enabled)
			authRequest = ssoAuth(request);
		else
			authRequest = anonymousAuth();

		Authentication res = this.getAuthenticationManager().authenticate(authRequest);

		// hack
		if (!enabled) {
			log.debug("Setting HACK auth " + res);
			SecurityContextHolder.getContext().setAuthentication(res);
		}

		return res;
	}

	@Override
	public String getDefaultFilterProcessesUrl() {
		return "/auth";
	}

	@Override
	protected boolean requiresAuthentication(HttpServletRequest request, HttpServletResponse response) {
		if (!enabled) {
			if (SecurityContextHolder.getContext().getAuthentication() != null) {
				log.debug("already have some auth token " + SecurityContextHolder.getContext().getAuthentication());
				return false;
			}

			return true;
		}

		String authParam = request.getParameter("auth");
		if (authParam != null && !authParam.equals(""))
			return true;

		return false;
	}

	protected SimpleSSOAuthenticationToken ssoAuth(HttpServletRequest request) {
		return new SimpleSSOAuthenticationToken(request.getParameter("auth"));
	}

	protected Authentication anonymousAuth() {
		return new AnonymousAuthenticationToken("disabled", "anonymous", rolesBuilder.buildAuthorities(anonymousRoles));
	}

	protected void successfulAuthentication(HttpServletRequest request, HttpServletResponse response, Authentication authResult) throws IOException,
			ServletException {

		if (enabled)
			super.successfulAuthentication(request, response, authResult);
		else {
			log.debug("SETTING AUTHENTICATION " + authResult);
			// force authentication without redirect
			SecurityContextHolder.getContext().setAuthentication(authResult);
		}
	}

	public boolean isEnabled() {
		return enabled;
	}

	public void setEnabled(boolean enabled) {
		this.enabled = enabled;
	}

	public SimpleSSOAuthenticationRoleBuilder getRolesBuilder() {
		return rolesBuilder;
	}

	public void setRolesBuilder(SimpleSSOAuthenticationRoleBuilder rolesBuilder) {
		this.rolesBuilder = rolesBuilder;
	}

	public List<String> getAnonymousRoles() {
		return anonymousRoles;
	}

	public void setAnonymousRoles(List<String> anonymousRoles) {
		this.anonymousRoles = anonymousRoles;
	}

}
