package eu.dnetlib.actionmanager.hbase;

import java.io.IOException;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.NavigableMap;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.filter.Filter;
import org.apache.hadoop.hbase.filter.FilterList;
import org.apache.hadoop.hbase.filter.PageFilter;
import org.apache.hadoop.hbase.filter.PrefixFilter;
import org.apache.hadoop.hbase.util.Bytes;
import org.springframework.beans.factory.annotation.Required;

import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.hash.Hashing;
import com.google.gson.Gson;
import com.google.protobuf.InvalidProtocolBufferException;

import eu.dnetlib.actionmanager.ActionManagerConstants;
import eu.dnetlib.actionmanager.ActionManagerConstants.ACTION_TYPE;
import eu.dnetlib.actionmanager.ActionManagerConstants.COLUMN_FAMILIES;
import eu.dnetlib.actionmanager.rmi.ActionManagerException;
import eu.dnetlib.data.hadoop.HadoopServiceCore;
import eu.dnetlib.data.hadoop.config.ClusterName;
import eu.dnetlib.data.hadoop.rmi.HadoopServiceException;
import eu.dnetlib.data.proto.OafProtos.Oaf;

public class HBaseClient {

	private String tableName;

	private static final Log log = LogFactory.getLog(HBaseClient.class); // NOPMD by marko on 11/24/08 5:02 PM

	private final Executor executor = Executors.newSingleThreadExecutor();

	@Resource
	private HadoopServiceCore hadoopServiceCore;

	private final Function<Result, Map<String, String>> transformRow = new Function<Result, Map<String, String>>() {

		@Override
		public Map<String, String> apply(final Result r) {
			Map<String, String> map = Maps.newHashMap();
			map.put("rowId", Bytes.toString(r.getRow()));

			for (Entry<byte[], NavigableMap<byte[], NavigableMap<Long, byte[]>>> e1 : r.getMap().entrySet()) {
				String cf = Bytes.toString(e1.getKey());
				for (Entry<byte[], NavigableMap<Long, byte[]>> e2 : e1.getValue().entrySet()) {
					byte[] bb = e2.getValue().get(Collections.max(e2.getValue().keySet()));
					String key = cf + ":" + Bytes.toString(e2.getKey());
					String value = "";
					try {
						value = (key.equals("target:content")) ? Oaf.parseFrom(bb).toString() : Bytes.toString(bb);
					} catch (InvalidProtocolBufferException e) {
						log.error("Problem parsing protobuf !!!");
					}
					map.put(key, value);
				}
			}
			return map;
		}
	};

	public void init() {
		performInit(false);
	}

	public void initWithEmptyTable() {
		performInit(true);
	}

	private void performInit(final boolean delete) {
		executor.execute(new Runnable() {

			@Override
			public void run() {
				try {
					log.info("Initializing Action Manager");
					prepareTable(delete);
					log.info("ActionManager is ready");
				} catch (Exception e) {
					log.error("Error initializing action manager", e);
				}
			}
		});
	}

	private void prepareTable(final boolean delete) throws HadoopServiceException, IOException {
		final Set<String> columns = new HashSet<String>();
		for (COLUMN_FAMILIES cf : ActionManagerConstants.COLUMN_FAMILIES.values()) {
			columns.add(cf.toString());
		}

		if (delete) {
			hadoopServiceCore.truncateTable(ClusterName.DM, tableName);
		}
		hadoopServiceCore.ensureTable(ClusterName.DM, tableName, columns);
	}

	public void write(final List<Put> puts) throws ActionManagerException {

		try {
			log.info("Adding " + puts.size() + " action(s) to " + tableName);
			hadoopServiceCore.write(ClusterName.DM, tableName, puts);
			log.info("Number of action written: " + puts.size());
		} catch (IOException e) {
			throw new ActionManagerException(e);
		}
	}

	public void delete(final List<Delete> deletes) throws ActionManagerException {
		try {
			log.info("Deleting " + deletes.size() + " action(s) from " + tableName);
			hadoopServiceCore.delete(ClusterName.DM, tableName, deletes);
			log.info("Number of action deleted: " + deletes.size());
		} catch (IOException e) {
			throw new ActionManagerException(e);
		}
	}

	public Map<String, String> getRow(final byte[] id) throws ActionManagerException {
		try {
			final Result result = hadoopServiceCore.getRow(ClusterName.DM, tableName, id);
			return transformRow.apply(result);
		} catch (IOException e) {
			throw new ActionManagerException(e);
		}
	}

	public List<Map<String, String>> retrieveRows(final String prefix, final String start, final int limit) throws ActionManagerException {
		try {
			final Filter filter = ((prefix == null) || prefix.isEmpty()) ? new PageFilter(limit) : new FilterList(FilterList.Operator.MUST_PASS_ALL,
					new PageFilter(limit), new PrefixFilter(Bytes.toBytes(prefix)));

			final Scan scan = new Scan();

			if (start != null) {
				scan.setStartRow(Bytes.toBytes(start));
			}
			scan.setFilter(filter);

			return Lists.transform(hadoopServiceCore.getRows(ClusterName.DM, tableName, scan), transformRow);

		} catch (IOException e) {
			throw new ActionManagerException(e);
		}
	}

	public List<String> retrieveActionsByAgent(final String agentId, final int limit) throws ActionManagerException {

		try {

			final String prefix = ACTION_TYPE.pkg + "|" + Hashing.md5().hashString(agentId) + "|";

			final FilterList filter = new FilterList(FilterList.Operator.MUST_PASS_ALL);
			filter.addFilter(new PageFilter(limit));
			filter.addFilter(new PrefixFilter(Bytes.toBytes(prefix)));

			final Scan scan = new Scan();
			scan.setFilter(filter);

			final List<String> list = Lists.newArrayList();
			for (Result r : hadoopServiceCore.getRows(ClusterName.DM, tableName, scan)) {
				Map<String, String> map = Maps.newHashMap();
				map.put("id", Bytes.toString(r.getRow()));
				map.put("content", Bytes.toString(r.getValue(ActionManagerConstants.ACTION_COLFAMILY, Bytes.toBytes(ACTION_TYPE.pkg.toString()))));
				list.add(new Gson().toJson(map));
			}
			return list;
		} catch (IOException e) {
			throw new ActionManagerException(e);
		}

	}

	public String getTableName() {
		return tableName;
	}

	@Required
	public void setTableName(final String tableName) {
		this.tableName = tableName;
	}

}
