package eu.dnetlib.actionmanager.hbase;

import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;

import com.google.common.base.Joiner;
import com.google.common.collect.Sets;

import eu.dnetlib.actionmanager.is.ISClient;
import eu.dnetlib.actionmanager.rmi.ActionManagerException;
import eu.dnetlib.data.hadoop.config.ClusterName;
import eu.dnetlib.data.hadoop.rmi.HadoopBlackboardActions;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.tools.ServiceLocator;
import eu.dnetlib.enabling.tools.blackboard.BlackboardClientHandler;
import eu.dnetlib.enabling.tools.blackboard.BlackboardJob;
import eu.dnetlib.enabling.tools.blackboard.BlackboardJobRegistry;

public class HBaseJobLauncher {

	private static final Log log = LogFactory.getLog(HBaseJobLauncher.class); // NOPMD by marko on 11/24/08 5:02 PM

	public static final String ALL_SETS = "__ALL__";

	private final Executor executor = Executors.newCachedThreadPool();

	@Resource(name = "lookupLocator")
	private ServiceLocator<ISLookUpService> lookupLocator;

	private ISClient infomationServiceClient;

	/**
	 * blackboard handler.
	 */
	@Resource
	private BlackboardClientHandler blackboardClientHandler;

	/**
	 * blackboard job registry.
	 */
	@Resource
	private BlackboardJobRegistry jobRegistry;

	private String actionTable;
	private String dataTable;

	public void executeGarbageJob(final HBaseCallback callback) throws ActionManagerException {
		log.info("Starting clean garbage m/r job");

		final Set<String> rawSets = infomationServiceClient.listValidRawSets();

		execute("garbageActionsJob", rawSets, actionTable, actionTable, callback);
	}

	public void executePromoteJob(final String set, final HBaseCallback callback) throws ActionManagerException {
		log.info("Starting commit m/r job; set=" + set);

		final Set<String> rawSets = Sets.newHashSet();
		if (set == null || set.equals(ALL_SETS)) {
			rawSets.addAll(infomationServiceClient.listValidRawSets());
		} else if (infomationServiceClient.existsSet(set)) {
			rawSets.add(infomationServiceClient.geLatestRawSet(set).getId());
		} else {
			log.error("Invalid set " + set);
			throw new ActionManagerException("Invalid set " + set);
		}
		execute("promoteActionsJob", rawSets, actionTable, dataTable, callback);
	}

	private void execute(final String jobName, final Set<String> rawSets, final String inputTable, final String outputTable, final HBaseCallback callback)
			throws ActionManagerException {

		if (rawSets == null || rawSets.isEmpty()) { throw new ActionManagerException("Missing rawsets"); }

		executor.execute(new Runnable() {

			@Override
			public void run() {
				try {
					final String serviceId = findHadoopServiceProfileID();
					final BlackboardJob bbJob = blackboardClientHandler.newJob(serviceId);

					final String rawsets = Joiner.on(",").skipNulls().join(rawSets);
					log.info("promoting rawsets: " + rawsets);

					bbJob.setAction(HadoopBlackboardActions.SUBMIT_MAPREDUCE_JOB.toString());
					bbJob.getParameters().put("job.name", jobName);
					bbJob.getParameters().put("cluster", ClusterName.DM.toString());
					bbJob.getParameters().put("latestRawSets", rawsets);
					bbJob.getParameters().put("hbase.mapred.inputtable", inputTable);
					bbJob.getParameters().put("hbase.mapreduce.inputtable", inputTable);
					bbJob.getParameters().put("hbase.mapred.outputtable", outputTable);
					bbJob.getParameters().put("hbase.mapreduce.outputtable", outputTable);

					jobRegistry.registerJobListener(bbJob, new HBaseBlackboardJobListener(callback));
					blackboardClientHandler.assign(bbJob);
				} catch (Exception e) {
					log.error("Error launching m/r job: " + jobName, e);
					throw new RuntimeException("Error launching m/r job: " + jobName, e);
				}
			}

		});
	}

	private String findHadoopServiceProfileID() throws Exception {
		return lookupLocator.getService().getResourceProfileByQuery(
				"collection('/db/DRIVER/ServiceResources/HadoopServiceResourceType')//RESOURCE_IDENTIFIER/@value/string()");
	}

	public String getActionTable() {
		return actionTable;
	}

	@Required
	public void setActionTable(final String actionTable) {
		this.actionTable = actionTable;
	}

	public String getDataTable() {
		return dataTable;
	}

	@Required
	public void setDataTable(final String dataTable) {
		this.dataTable = dataTable;
	}

	public ISClient getInfomationServiceClient() {
		return infomationServiceClient;
	}

	@Required
	public void setInfomationServiceClient(final ISClient infomationServiceClient) {
		this.infomationServiceClient = infomationServiceClient;
	}

}
