package eu.dnetlib.actionmanager.is;

import java.io.StringReader;
import java.util.List;
import java.util.Set;

import javax.xml.ws.Endpoint;

import org.antlr.stringtemplate.StringTemplate;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Required;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import eu.dnetlib.actionmanager.rmi.ActionManagerException;
import eu.dnetlib.actionmanager.set.ActionManagerSet;
import eu.dnetlib.actionmanager.set.ActionManagerSet.ImpactTypes;
import eu.dnetlib.actionmanager.set.RawSet;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.is.registry.rmi.ISRegistryException;
import eu.dnetlib.enabling.is.registry.rmi.ISRegistryService;
import eu.dnetlib.enabling.tools.ServiceLocator;
import eu.dnetlib.miscutils.datetime.DateUtils;
import eu.dnetlib.soap.EndpointReferenceBuilder;

public class ISClient {

	private ServiceLocator<ISRegistryService> registryLocator;
	private ServiceLocator<ISLookUpService> lookupLocator;
	/**
	 * Template for ActionManagerSet profiles
	 */
	private StringTemplate actionManagerSetDsTemplate;

	/**
	 * service endpoint.
	 */
	private Endpoint endpoint;

	/**
	 * endpoint builder.
	 */
	private EndpointReferenceBuilder<Endpoint> eprBuilder;

	private static final Log log = LogFactory.getLog(ISClient.class); // NOPMD by marko on 11/24/08 5:02 PM

	public String registerSetProfile(final ActionManagerSet set) throws ActionManagerException {
		if (existsSet(set.getId())) { throw new ActionManagerException("Set " + set.getId() + " already registered"); }
		try {
			StringTemplate template = new StringTemplate(actionManagerSetDsTemplate.getTemplate());
			template.setAttribute("serviceUri", eprBuilder.getAddress(endpoint));
			template.setAttribute("set", set);
			template.setAttribute("latest", RawSet.newInstance());
			return registryLocator.getService().registerProfile(template.toString());
		} catch (ISRegistryException e) {
			throw new ActionManagerException("Error registering set " + set, e);
		}
	}

	public List<ActionManagerSet> listSets() {
		final String q = "for $x in collection('/db/DRIVER/ActionManagerSetDSResources/ActionManagerSetDSResourceType') return $x";

		final SAXReader reader = new SAXReader();

		final List<ActionManagerSet> list = Lists.newArrayList();

		try {
			for (String s : lookupLocator.getService().quickSearchProfile(q)) {

				final ActionManagerSet set = new ActionManagerSet();

				try {
					final Document doc = reader.read(new StringReader(s));

					set.setId(doc.valueOf("//SET/@id").trim());
					set.setName(doc.valueOf("//SET").trim());
					set.setImpact(ImpactTypes.valueOf(doc.valueOf("//IMPACT").trim()));
					set.setLatest(doc.valueOf("//RAW_SETS/LATEST/@id"), doc.valueOf("//RAW_SETS/LATEST/@creationDate"),
							doc.valueOf("//RAW_SETS/LATEST/@lastUpdate"));

					list.add(set);
				} catch (Exception e) {
					log.error("Error creating set from profile: " + s);
				}
			}
		} catch (ISLookUpException e) {
			log.error("Error accessing Sets, using query: " + q);
		}
		return list;
	}

	public boolean deleteSetProfile(final String setId) throws ActionManagerException {
		final String q = "for $x in collection('/db/DRIVER/ActionManagerSetDSResources/ActionManagerSetDSResourceType') where $x//SET/@id = '" + setId
				+ "' return $x//RESOURCE_IDENTIFIER/@value/string()";

		try {
			final String profId = lookupLocator.getService().getResourceProfileByQuery(q);
			return registryLocator.getService().deleteProfile(profId);
		} catch (Exception e) {
			log.error("Error deleting set " + setId, e);
			throw new ActionManagerException("Error deleting set " + setId, e);
		}
	}

	public boolean existsSet(final String id) throws ActionManagerException {
		final String q = "for $x in collection('/db/DRIVER/ActionManagerSetDSResources/ActionManagerSetDSResourceType') where $x//SET/@id = '" + id
				+ "' return $x";
		try {
			return !lookupLocator.getService().quickSearchProfile(q).isEmpty();
		} catch (ISLookUpException e) {
			log.error("Error accessing Sets, using query: " + q, e);
			throw new ActionManagerException("Error running xquery: " + q, e);
		}
	}

	public boolean existsRawSet(final String id) throws ActionManagerException {
		final String q = "for $x in collection('/db/DRIVER/ActionManagerSetDSResources/ActionManagerSetDSResourceType') where $x//RAW_SETS/*/@id = '" + id
				+ "' return $x";
		try {
			return !lookupLocator.getService().quickSearchProfile(q).isEmpty();
		} catch (ISLookUpException e) {
			log.error("Error accessing RawSets, using query: " + q, e);
			throw new ActionManagerException("Error running xquery: " + q, e);
		}
	}

	public Set<String> listValidRawSets() throws ActionManagerException {
		final String q = "for $x in collection('/db/DRIVER/ActionManagerSetDSResources/ActionManagerSetDSResourceType') return $x//RAW_SETS/LATEST/@id/string()";
		try {
			final List<String> list = lookupLocator.getService().quickSearchProfile(q);
			return Sets.newHashSet(list);
		} catch (Exception e) {
			log.error("Error running xquery: " + q, e);
			throw new ActionManagerException("Error running xquery: " + q, e);
		}
	}

	public RawSet geLatestRawSet(final String set) throws ActionManagerException {
		final String q = "for $x in collection('/db/DRIVER/ActionManagerSetDSResources/ActionManagerSetDSResourceType') where $x//SET/@id = '" + set
				+ "' return concat(' ' , $x//RAW_SETS/LATEST/@id, ' @@@ ', $x//RAW_SETS/LATEST/@creationDate, ' @@@ ', $x//RAW_SETS/LATEST/@lastUpdate, ' ')";
		try {
			String[] arr = lookupLocator.getService().getResourceProfileByQuery(q).split("@@@");
			return RawSet.newInstance(arr[0].trim(), arr[1].trim(), arr[2].trim());
		} catch (ISLookUpException e) {
			log.error("Error accessing Sets, using query: " + q, e);
			throw new ActionManagerException("Error running xquery: " + q, e);
		}
	}

	public void addLatestRawSet(final String set, final RawSet rawSet) throws ActionManagerException {
		final String q = "for $x in collection('/db/DRIVER/ActionManagerSetDSResources/ActionManagerSetDSResourceType') where $x//SET/@id = '" + set
				+ "' return $x";
		try {
			final String profile = lookupLocator.getService().getResourceProfileByQuery(q);
			final Document doc = (new SAXReader()).read(new StringReader(profile));
			final String profId = doc.valueOf("//RESOURCE_IDENTIFIER/@value");
			final Element latest = (Element) doc.selectSingleNode("//RAW_SETS/LATEST");
			final Element expired = ((Element) doc.selectSingleNode("//RAW_SETS")).addElement("EXPIRED");

			expired.setAttributes(latest.attributes());

			latest.addAttribute("id", rawSet.getId());
			latest.addAttribute("creationDate", rawSet.getCreationDate());
			latest.addAttribute("lastUpdate", rawSet.getLastUpdate());

			registryLocator.getService().updateProfile(profId, doc.asXML(), "ActionManagerSetDSResourceTyper");
		} catch (Exception e) {
			log.error("Error updating profile of set: " + set);
			throw new ActionManagerException("Error running xquery: " + q, e);
		}
	}

	public void updateLastUpdate(final String rawSet) throws ActionManagerException {
		final String q = "for $x in collection('/db/DRIVER/ActionManagerSetDSResources/ActionManagerSetDSResourceType')//RAW_SETS/*[@id='" + rawSet
				+ "']/@lastUpdate return update replace $x with '" + DateUtils.now_ISO8601() + "'";

		try {
			registryLocator.getService().executeXUpdate(q);
		} catch (Exception e) {
			log.error("Error updating lastUpdate using query: " + q, e);
			throw new ActionManagerException("Error updating lastUpdate using query: " + q, e);
		}
	}

	public StringTemplate getActionManagerSetDsTemplate() {
		return actionManagerSetDsTemplate;
	}

	@Required
	public void setActionManagerSetDsTemplate(final StringTemplate actionManagerSetDsTemplate) {
		this.actionManagerSetDsTemplate = actionManagerSetDsTemplate;
	}

	public ServiceLocator<ISRegistryService> getRegistryLocator() {
		return registryLocator;
	}

	@Required
	public void setRegistryLocator(final ServiceLocator<ISRegistryService> registryLocator) {
		this.registryLocator = registryLocator;
	}

	public Endpoint getEndpoint() {
		return endpoint;
	}

	@Required
	public void setEndpoint(final Endpoint endpoint) {
		this.endpoint = endpoint;
	}

	public EndpointReferenceBuilder<Endpoint> getEprBuilder() {
		return eprBuilder;
	}

	@Required
	public void setEprBuilder(final EndpointReferenceBuilder<Endpoint> eprBuilder) {
		this.eprBuilder = eprBuilder;
	}

	public ServiceLocator<ISLookUpService> getLookupLocator() {
		return lookupLocator;
	}

	@Required
	public void setLookupLocator(final ServiceLocator<ISLookUpService> lookupLocator) {
		this.lookupLocator = lookupLocator;
	}

}
