package eu.dnetlib.data.objectstore.modular.gridFS;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.List;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.mongodb.BasicDBObject;
import com.mongodb.DBObject;
import com.mongodb.gridfs.GridFS;
import com.mongodb.gridfs.GridFSDBFile;
import com.mongodb.gridfs.GridFSInputFile;

import eu.dnetlib.data.objectstore.modular.ObjectStoreRecord;
import eu.dnetlib.data.objectstore.modular.connector.ObjectStore;
import eu.dnetlib.data.objectstore.rmi.MetadataObjectRecord;
import eu.dnetlib.data.objectstore.rmi.ObjectStoreFile;
import eu.dnetlib.data.objectstore.rmi.Protocols;
import eu.dnetlib.enabling.resultset.ResultSetListener;

public class GridFSObjectStore implements ObjectStore {

	private static final Log log = LogFactory.getLog(GridFSObjectStore.class);

	private String id;
	private boolean upsert;
	private GridFS collection;
	private String baseURI;

	public GridFSObjectStore(final String id, final GridFS collection, final boolean upsert) {
		this.id = id;
		this.setUpsert(upsert);
		this.collection = collection;

	}

	@Override
	public String getId() {
		return id;
	}

	@Override
	public int feedMetadataRecord(final Iterable<MetadataObjectRecord> records, final boolean incremental) {
		long timestamp = System.currentTimeMillis();
		for (MetadataObjectRecord o : records) {
			if (o.getRecord() == null) {
				log.debug("Null object " + o.getId());
				continue;
			}

			GridFSInputFile currentFile = collection.createFile(new ByteArrayInputStream(o.getRecord().getBytes()));
			currentFile.setId(o.getId());
			BasicDBObject metadata = new BasicDBObject();
			metadata.put("id", o.getId());
			metadata.put("mime", o.getMime());
			metadata.put("timestamp", timestamp);
			try {
				String URI = (baseURI + "?objectStore=" + URLEncoder.encode(id, "UTF-8") + "&objectId=" + URLEncoder.encode(o.getId(), "UTF-8"));
				metadata.put("uri", URI);

			} catch (UnsupportedEncodingException e) {
				log.error("Got an exception during the feed ", e);
			}
			currentFile.setMetaData(metadata);
			currentFile.save();
		}
		return this.getSize();
	}

	@Override
	public int feed(final Iterable<ObjectStoreRecord> records, final boolean incremental) {
		long timestamp = System.currentTimeMillis();
		int time = 0;
		for (ObjectStoreRecord o : records) {
			if ((o == null) || (o.getInputStream() == null)) {
				if (o != null) {
					log.debug("Null object " + o.getFileMetadata().toJSON());
				} else {
					log.debug("Null Object");
				}
				try {
					Thread.sleep(1000);
				} catch (InterruptedException e) {
					log.error(e);
				}
				continue;
			}
			if (alreadyExist(o.getFileMetadata().getObjectID())) {
				try {
					o.getInputStream().close();
				} catch (IOException e) {

				}
				continue;
			}
			if (time++ > 100) {
				try {
					time = 0;
					// Thread.sleep(5000);
				} catch (Exception e) {

				}
			}
			GridFSInputFile currentFile = collection.createFile(o.getInputStream());
			currentFile.setId(o.getFileMetadata().getObjectID());
			currentFile.setFilename(o.getFileMetadata().getObjectID());
			BasicDBObject metadata = new BasicDBObject();
			metadata.put("id", o.getFileMetadata().getObjectID());
			metadata.put("mime", o.getFileMetadata().getMimeType());
			metadata.put("originalObject", o.getFileMetadata().toJSON());
			metadata.put("timestamp", timestamp);
			try {
				String URI = (baseURI + "?objectStore=" + URLEncoder.encode(id, "UTF-8") + "&objectId=" + URLEncoder.encode(o.getFileMetadata().getObjectID(),
						"UTF-8"));
				metadata.put("uri", URI);

			} catch (UnsupportedEncodingException e) {
				log.error("Got an exception during the feed ", e);
			}
			currentFile.setMetaData(metadata);
			currentFile.save();
		}
		return getSize();
	}

	@Override
	public ResultSetListener deliver(final Double from, final Double until) {
		GridFSObjectstoreResultSetListener resulset = new GridFSObjectstoreResultSetListener();
		resulset.setBaseURI(baseURI);
		resulset.setObjectStoreID(id);
		resulset.setCollection(collection);
		resulset.setFromDate(from);
		resulset.setUntilDate(until);
		return resulset;
	}

	@Override
	public ResultSetListener deliverIds(final Iterable<String> ids) {
		GridFSObjectstoreResultSetListener resulset = new GridFSObjectstoreResultSetListener();
		resulset.setBaseURI(baseURI);
		resulset.setObjectStoreID(id);
		resulset.setCollection(collection);
		resulset.setRecords((List<String>) ids);
		return resulset;
	}

	@Override
	public int getSize() {
		return collection.getFileList().count();
	}

	@Override
	public void deleteObject(final String objectId) {

	}

	@Override
	public String getObject(final String recordId) {
		return null;
	}

	@Override
	public String getInterpretation() {
		return (String) getMDStoreMetadata().get("interpretation");
	}

	public DBObject getMDStoreMetadata() {
		return collection.getDB().getCollection("metadataObjectStore").findOne(new BasicDBObject("obsId", Pattern.compile(getId())));
	}

	public String getBaseURI() {
		return baseURI;
	}

	public void setBaseURI(final String baseURI) {
		this.baseURI = baseURI;

	}

	public boolean isUpsert() {
		return upsert;
	}

	public void setUpsert(final boolean upsert) {
		this.upsert = upsert;
	}

	private boolean alreadyExist(final String objectId) {
		BasicDBObject query = new BasicDBObject("_id", objectId);
		List<GridFSDBFile> file = collection.find(query);
		if ((file != null) && (file.size() > 0)) return true;
		return false;
	}

	@Override
	public ObjectStoreFile deliverObject(final String objectId) {
		BasicDBObject query = new BasicDBObject("_id", objectId);
		List<GridFSDBFile> file = collection.find(query);
		for (GridFSDBFile f : file) {
			DBObject outObject = f.getMetaData();
			if (outObject.get("id").equals(objectId)) {
				ObjectStoreFile outFile = new ObjectStoreFile();
				outFile.setObjectID(objectId);
				outFile.setMimeType((String) outObject.get("mime"));
				outFile.setAccessProtocol(Protocols.HTTP);
				try {
					outFile.setURI(baseURI + "?objectStore=" + URLEncoder.encode(id, "UTF-8") + "&objectId=" + URLEncoder.encode(objectId, "UTF-8"));
				} catch (UnsupportedEncodingException e) {
					log.error(e);
				}
				return outFile;
			}

		}
		return null;
	}

	@Override
	public InputStream deliverStream(final String objectId) {
		BasicDBObject query = new BasicDBObject("_id", objectId);
		List<GridFSDBFile> file = collection.find(query);
		for (GridFSDBFile f : file) {
			if (f.getMetaData().get("id").equals(objectId)) return f.getInputStream();
		}
		return null;
	}

	@Override
	public String feedObjectRecord(final ObjectStoreRecord record) {
		ObjectStoreFile obj = deliverObject(record.getFileMetadata().getObjectID());
		if (obj.getObjectID() == record.getFileMetadata().getObjectID()) return obj.getURI();
		long timestamp = System.currentTimeMillis();
		String URI = "";
		if (record.getInputStream() == null) return null;
		GridFSInputFile currentFile = collection.createFile(record.getInputStream());
		currentFile.setId(record.getFileMetadata().getObjectID());
		currentFile.setFilename(record.getFileMetadata().getObjectID());
		BasicDBObject metadata = new BasicDBObject();
		metadata.put("id", record.getFileMetadata().getObjectID());
		metadata.put("mime", record.getFileMetadata().getMimeType());
		metadata.put("originalObject", record.getFileMetadata().toJSON());
		metadata.put("timestamp", timestamp);
		try {
			URI = (baseURI + "?objectStore=" + URLEncoder.encode(id, "UTF-8") + "&objectId=" + URLEncoder.encode(record.getFileMetadata().getObjectID(),
					"UTF-8"));
			metadata.put("uri", URI);

		} catch (UnsupportedEncodingException e) {
			log.error(e);
		}
		currentFile.setMetaData(metadata);
		currentFile.save();
		return URI;
	}

	@Override
	public boolean existIDStartsWith(final String startId) {
		GridFSDBFile out = collection.findOne(new BasicDBObject("_id", Pattern.compile(startId)));
		return out != null;
	}

}
