package eu.dnetlib.data.objectstore.modular.gridFS;

import java.util.List;
import java.util.regex.Pattern;

import org.bson.BSONObject;
import org.springframework.beans.factory.annotation.Required;

import com.mongodb.BasicDBObject;
import com.mongodb.DB;
import com.mongodb.DBCollection;
import com.mongodb.DBObject;
import com.mongodb.gridfs.GridFS;

import eu.dnetlib.data.objectstore.modular.connector.ObjectStore;
import eu.dnetlib.data.objectstore.modular.connector.ObjectStoreDao;
import eu.dnetlib.miscutils.collections.MappedCollection;
import eu.dnetlib.miscutils.functional.UnaryFunction;

public class GridFSObjectstoreDaoImpl implements ObjectStoreDao {

	private final static String OBJECTSTOREMETADATANAME = "metadataObjectStore";
	private final static String OBJECTSTOREIDFIELD = "obsId";

	private DB db;
	private boolean upsert;
	private String objectStoreRESTURI;

	@Override
	public ObjectStore getObjectStore(String obsId) {
		obsId = obsId.substring(0, 36);
		GridFSObjectStore objectStore = new GridFSObjectStore(obsId, new GridFS(db, obsId), isUpsert());
		objectStore.setBaseURI(objectStoreRESTURI);
		checkIndexes(obsId);
		return objectStore;
	}

	private void checkIndexes(final String id) {
		DBCollection coll = db.getCollection(id + ".files");
		List<DBObject> indexInfos = coll.getIndexInfo();
		boolean hasTimestampIndex = false;
		for (DBObject info : indexInfos) {
			BSONObject boj = (BSONObject) info.get("key");
			if (boj.get("metadata.timestamp") != null) {
				hasTimestampIndex = true;
			}
		}
		if (hasTimestampIndex == false) {
			coll.ensureIndex("metadata.timestamp");
		}
	}

	@Override
	public List<String> listObjectStores() {
		DBCollection metadata = db.getCollection(OBJECTSTOREMETADATANAME);
		return MappedCollection.listMap(metadata.find(), new UnaryFunction<String, DBObject>() {

			@Override
			public String evaluate(final DBObject object) {
				return (String) object.get(OBJECTSTOREIDFIELD);
			}
		});

	}

	@Override
	public boolean createObjectStore(final String obsId, final String interpretation) {
		DBCollection coll = db.getCollection(OBJECTSTOREMETADATANAME);
		final BasicDBObject obj = new BasicDBObject();
		obj.put(OBJECTSTOREIDFIELD, obsId);
		obj.put("interpretation", interpretation);
		coll.save(obj);
		return true;
	}

	@Override
	public boolean updateObjectStore(final String obsId, final String interpretation) {
		DBCollection coll = db.getCollection(OBJECTSTOREMETADATANAME);
		final BasicDBObject obj = new BasicDBObject();
		obj.put(OBJECTSTOREIDFIELD, obsId);
		obj.put("interpretation", interpretation);
		coll.update(new BasicDBObject("obsId", obsId), obj, true, false);
		coll.save(obj);
		return true;
	}

	@Override
	public boolean deleteObjectStore(String obsId) {
		DBCollection coll = db.getCollection(OBJECTSTOREMETADATANAME);
		coll.remove(new BasicDBObject(OBJECTSTOREIDFIELD, obsId));
		obsId = obsId.substring(0, 36);
		GridFS objectStore = new GridFS(db, obsId);
		Pattern any = Pattern.compile(".");
		BasicDBObject query = new BasicDBObject("md5", any);
		objectStore.remove(query);
		return true;
	}

	public DB getDb() {
		return db;
	}

	@Required
	public void setDb(final DB db) {
		this.db = db;
	}

	public boolean isUpsert() {
		return upsert;
	}

	@Required
	public void setUpsert(final boolean upsert) {
		this.upsert = upsert;
	}

	public String getObjectStoreRESTURI() {
		return objectStoreRESTURI;
	}

	@Required
	public void setObjectStoreRESTURI(final String objectStoreRESTURI) {
		this.objectStoreRESTURI = objectStoreRESTURI;
	}

}
