package eu.dnetlib.data.mapreduce.hbase.index;

import java.io.IOException;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableMapper;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Counter;

import com.google.protobuf.InvalidProtocolBufferException;

import eu.dnetlib.data.mapreduce.JobParams;
import eu.dnetlib.data.mapreduce.hbase.OafRowDecoder;
import eu.dnetlib.data.mapreduce.hbase.index.config.EntityLinkTable;
import eu.dnetlib.data.mapreduce.hbase.index.config.IndexConfig;
import eu.dnetlib.data.mapreduce.hbase.index.config.LinkDescriptor;
import eu.dnetlib.data.mapreduce.util.UpdateMerger;
import eu.dnetlib.data.proto.KindProtos.Kind;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.OafProtos.OafRel;
import eu.dnetlib.data.proto.OafProtos.OafRelOrBuilder;
import eu.dnetlib.data.proto.RelTypeProtos.RelType;
import eu.dnetlib.data.proto.TypeProtos.Type;

public class PrepareFeedMapper extends TableMapper<Text, ImmutableBytesWritable> {

	private EntityLinkTable entityLinkTable;

	@Override
	protected void setup(final Context context) throws IOException, InterruptedException {
		String json = context.getConfiguration().get(JobParams.INDEX_ENTITY_LINKS);
		System.out.println(JobParams.INDEX_ENTITY_LINKS + ":\n" + json);
		entityLinkTable = IndexConfig.load(json).getLinkMap();
	}

	@Override
	protected void map(final ImmutableBytesWritable keyIn, final Result result, final Context context) throws IOException, InterruptedException {

		final String keyOut = new String(keyIn.copyBytes());
		final OafRowDecoder decoder = OafRowDecoder.decode(keyIn);

		if (!decoder.isValid()) {
			System.err.println("invalid key: " + keyOut);
		} else {

			if (decoder.hasType()) {
				final Type type = decoder.getType();
				final byte[] body = UpdateMerger.mergeBodyUpdates(result.getFamilyMap(Bytes.toBytes(type.toString())));

				if (isValid(body)) {

					final Oaf oaf = decodeProto(context, body);

					if (!deletedByInference(oaf)) {
						emit(keyOut, context, body);
						incrementCounter(context, Kind.entity.toString(), type.toString(), 1);
					} else {
						incrementCounter(context, "deleted by inference", type.toString(), 1);
					}

					for (LinkDescriptor ld : entityLinkTable.getDescriptors(type)) {
						emitRel(oaf, context, result, type, ld.getRelType(), ld.getTarget(), ld.isChild());
					}

				} else {
					incrementCounter(context, "missing body", type.toString(), 1);
				}
			} else {
				throw new IllegalStateException("unknow row type: " + keyOut);
			}
		}
	}

	private void emitRel(final Oaf body,
			final Context context,
			final Result result,
			final Type source,
			final RelType relType,
			final Type target,
			final boolean isChild) throws IOException, InterruptedException {

		final Oaf.Builder oafBuilder = Oaf.newBuilder().setKind(Kind.relation);
		final Map<byte[], byte[]> columnMap = result.getFamilyMap(Bytes.toBytes(relType.toString()));

		if (isValid(columnMap)) {
			// iterates the column map
			for (Entry<byte[], byte[]> e : columnMap.entrySet()) {

				final Oaf decodedOaf = decodeProto(context, e.getValue());
				if (!deletedByInference(decodedOaf)) {

					OafRel.Builder relBuilder = OafRel.newBuilder(decodedOaf.getRel());
					relBuilder.setCachedTarget(body.getEntity());

					if (relType.equals(RelType.dedupRel) && isDedupSelf(relBuilder)) {
						System.out.println("avoid to emit dedup self: " + relBuilder.getSource());
						continue;
					}

					OafRel oafRel = relBuilder.setRelType(relType).setChild(isChild).build();

					// String rowKey = patchTargetId(target, OafRelDecoder.decode(oafRel).getRelTargetId());

					emit(oafRel.getTarget(), context, merge(oafBuilder, decodedOaf).setRel(oafRel).build().toByteArray());
				} else {
					incrementCounter(context, "deleted by inference", relType.toString(), 1);
				}
			}
			incrementCounter(context, source.toString(), relType.toString(), columnMap.size());
		} else {
			incrementCounter(context, source.toString(), relType.toString() + "_empty", 1);
		}
	}

	protected String patchTargetId(final Type target, final String id) {
		return id.replaceFirst("^.*\\|", target.getNumber() + "|");
	}

	private Oaf.Builder merge(final Oaf.Builder builder, final Oaf prototype) {
		return builder.setDataInfo(prototype.getDataInfo()).setTimestamp(prototype.getTimestamp());
	}

	private boolean isDedupSelf(final OafRelOrBuilder rel) {
		return rel.getSource().contains(rel.getTarget());
	}

	private boolean isValid(final Map<byte[], byte[]> columnMap) {
		return (columnMap != null) && !columnMap.isEmpty();
	}

	private boolean isValid(final byte[] body) {
		return (body != null) && (body.length > 0);
	}

	private boolean deletedByInference(final Oaf oaf) {
		return oaf.getDataInfo().getDeletedbyinference();
	}

	private Oaf decodeProto(final Context context, final byte[] body) {
		try {
			return Oaf.parseFrom(body);
		} catch (InvalidProtocolBufferException e) {
			context.getCounter("decodeProto", e.getClass().getName()).increment(1);
		}
		return null;
	}

	// private byte[] prefix(final Type name, final byte[] bytes) {
	// return concat(concat(name.toString().getBytes(), PrepareFeedJob.bSEPARATOR), bytes);
	// }

	private void emit(final String key, final Context context, final byte[] bytes) throws IOException, InterruptedException {
		// Text keyOut = new Text(Hashing.murmur3_128().hashString(key).toString());
		Text keyOut = new Text(key);
		context.write(keyOut, new ImmutableBytesWritable(bytes));
	}

	protected byte[] concat(final byte[] a, final byte[] b) {
		byte[] c = new byte[a.length + b.length];
		System.arraycopy(a, 0, c, 0, a.length);
		System.arraycopy(b, 0, c, a.length, b.length);

		return c;
	}

	private void incrementCounter(final Context context, final String k, final String t, final int n) {
		getCounter(context, k, t).increment(n);
	}

	private Counter getCounter(final Context context, final String k, final String t) {
		return context.getCounter(k, t);
	}

}
