package eu.dnetlib.data.mapreduce.util;

import static eu.dnetlib.miscutils.collections.MappedCollection.listMap;

import java.io.StringReader;
import java.text.Normalizer;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.UUID;

import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.StringUtils;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.json.JSONObject;

import com.google.common.base.Predicate;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.protobuf.Descriptors.FieldDescriptor;
import com.google.protobuf.GeneratedMessage;

import eu.dnetlib.data.mapreduce.hbase.index.config.ContextDef;
import eu.dnetlib.data.mapreduce.hbase.index.config.ContextMapper;
import eu.dnetlib.data.mapreduce.hbase.index.config.EntityLinkTable;
import eu.dnetlib.data.proto.KeyValueProtos.KeyValue;
import eu.dnetlib.data.proto.OafProtos.OafEntity;
import eu.dnetlib.data.proto.OafProtos.OafRel;
import eu.dnetlib.data.proto.QualifierProtos.Qualifier;
import eu.dnetlib.data.proto.RelMetadataProtos.RelMetadata;
import eu.dnetlib.data.proto.RelTypeProtos.RelType;
import eu.dnetlib.data.proto.ResultProtos.Result;
import eu.dnetlib.data.proto.ResultProtos.Result.Context;
import eu.dnetlib.data.proto.ResultProtos.Result.ExternalReference;
import eu.dnetlib.data.proto.ResultProtos.Result.Instance;
import eu.dnetlib.data.proto.ResultProtos.Result.Journal;
import eu.dnetlib.data.proto.StructuredPropertyProtos.StructuredProperty;
import eu.dnetlib.data.proto.TypeProtos.Type;
import eu.dnetlib.miscutils.functional.UnaryFunction;

public class XmlRecordFactory {

	protected Set<String> specialDatasourceTypes = Sets.newHashSet("scholarcomminfra", "infospace", "pubsrepository::mock", "entityregistry");

	protected TemplateFactory templateFactory = new TemplateFactory();

	protected OafEntityDecoder mainEntity = null;

	protected String id = null;

	protected Map<String, OafRel> relations = Maps.newHashMap();
	protected Map<String, OafRel> children = Maps.newHashMap();

	protected EntityLinkTable entityLinkTable;

	protected ContextMapper contextMapper;

	protected boolean entityDefaults;
	protected boolean relDefaults;
	protected boolean childDefaults;

	public XmlRecordFactory(final EntityLinkTable entityLinkTable, final ContextMapper contextMapper, final boolean entityDefaults, final boolean relDefaults,
			final boolean childDefeaults) {
		this.entityLinkTable = entityLinkTable;
		this.contextMapper = contextMapper;
		this.entityDefaults = entityDefaults;
		this.relDefaults = relDefaults;
		this.childDefaults = childDefeaults;
	}

	public String getId() {
		return id;
	}

	public boolean isValid() {
		return mainEntity != null;
	}

	public void setMainEntity(final OafEntity entity) {
		this.mainEntity = OafEntityDecoder.decode(entity);
		this.id = mainEntity.getId();
	}

	public void addRelation(final OafRel rel) {
		addRelOrChild(relations, rel);
	}

	public void addChild(final OafRel child) {
		addRelOrChild(children, child);
	}

	private void addRelOrChild(final Map<String, OafRel> map, final OafRel rel) {
		if (!rel.getSource().equals(getId())) {
			map.put(rel.getSource(), rel);
		} else {
			map.put(UUID.randomUUID().toString(), rel);
		}
	}

	public String build() {

		// System.out.println("building");
		// System.out.println("main: " + mainEntity);
		// System.out.println("rel:  " + relations);
		// System.out.println("chi:  " + children);
		// System.out.println("=============");

		final Type type = mainEntity.getType();
		final List<String> metadata = decodeType(mainEntity, null, entityDefaults, false);
		final String body = templateFactory.buildBody(type, metadata, listRelations(), listChildren());

		// System.out.println("record id: " + recordId);
		return templateFactory.buildRecord(type, mainEntity.getId(), mainEntity.getDateOfCollection(), body);
	}

	private List<String> decodeType(final OafEntityDecoder decoder, final Set<String> filter, final boolean defaults, final boolean expandingRel) {

		final List<String> metadata = Lists.newArrayList();
		metadata.addAll(listFields(decoder.getMetadata(), filter, defaults, expandingRel));
		metadata.addAll(listFields(decoder.getOafEntity(), filter, defaults, expandingRel));

		if ((mainEntity.getEntity() instanceof Result) && !expandingRel) {
			metadata.add(asXmlElement("bestlicense", "", getBestLicense()));
		}

		return metadata;
	}

	private Qualifier getBestLicense() {
		Qualifier bestLicense = null;
		LicenseComparator lc = new LicenseComparator();
		for (Instance instance : ((Result) mainEntity.getEntity()).getInstanceList()) {
			if (lc.compare(bestLicense, instance.getLicence()) > 0) {
				bestLicense = instance.getLicence();
			}
		}
		return bestLicense;
	}

	private List<String> listRelations() {

		final List<String> rels = Lists.newArrayList();

		for (OafRel rel : this.relations.values()) {

			final OafEntity cachedTarget = rel.getCachedTarget();
			final OafRelDecoder relDecoder = OafRelDecoder.decode(rel);

			if (!relDecoder.getRelType().equals(RelType.personResult) || relDecoder.getRelTargetId().equals(id)) {
				final Set<String> filter = entityLinkTable.getFilter(cachedTarget.getType(), relDecoder.getRelType());
				final List<String> metadata = decodeType(OafEntityDecoder.decode(cachedTarget), filter, relDefaults, true);
				metadata.addAll(listFields(relDecoder.getRel(), null, false, true));
				RelMetadata relMetadata = relDecoder.getRelMetadata();

				String semanticclass = "";
				String semanticscheme = "";
				// debug
				if (relMetadata == null) {
					System.err.println(this);
					semanticclass = semanticscheme = "UNKNOWN";
				} else {
					semanticclass = relMetadata.getSemantics().getClassname();
					semanticscheme = relMetadata.getSemantics().getSchemename();
				}

				rels.add(templateFactory.getRel(cachedTarget.getType(), relDecoder.getRelSourceId(), metadata, semanticclass, semanticscheme));
			}
		}
		return rels;
	}

	private List<String> listChildren() {

		final List<String> children = Lists.newArrayList();
		for (OafRel rel : this.children.values()) {
			addChildren(children, rel.getCachedTarget(), rel.getRelType());
		}
		if (mainEntity.getType().equals(Type.result)) {
			for (Instance instance : ((Result) mainEntity.getEntity()).getInstanceList()) {
				children.add(templateFactory.getInstance(instance.getHostedby().getKey(), listFields(instance, null, false, false),
						listMap(instance.getUrlList(), new UnaryFunction<String, String>() {

							@Override
							public String evaluate(final String identifier) {
								return templateFactory.getWebResource(identifier);
							}
						})));
			}
			for (ExternalReference er : ((Result) mainEntity.getEntity()).getExternalReferenceList()) {
				Set<String> filters = entityLinkTable.getFilter(Type.result, RelType.resultResult);
				List<String> fields = listFields(er, filters, false, false);
				children.add(templateFactory.getChild("externalreference", null, fields));
			}
		}

		return children;
	}

	private void addChildren(final List<String> children, final OafEntity target, final RelType relType) {
		final OafEntityDecoder decoder = OafEntityDecoder.decode(target);
		Set<String> filters = entityLinkTable.getFilter(target.getType(), relType);
		children.add(templateFactory.getChild(decoder.getType().toString(), decoder.getId(), listFields(decoder.getMetadata(), filters, childDefaults, false)));
	}

	// //////////////////////////////////

	private List<String> listFields(final GeneratedMessage fields, final Set<String> filter, final boolean defaults, final boolean expandingRel) {

		final List<String> metadata = Lists.newArrayList();

		if (fields != null) {

			Set<String> seen = Sets.newHashSet();
			for (Entry<FieldDescriptor, Object> e : filterFields(fields, filter)) {

				// final String name = getFieldName(e.getKey().getName());
				final String name = e.getKey().getName();
				seen.add(name);

				addFieldValue(metadata, e.getKey(), e.getValue(), expandingRel);
			}

			if (defaults) {
				for (FieldDescriptor fd : fields.getDescriptorForType().getFields()) {
					if (!seen.contains(fd.getName())) {
						addFieldValue(metadata, fd, getDefault(fd), expandingRel);
					}
				}
			}
		}
		return metadata;
	}

	private Object getDefault(final FieldDescriptor fd) {
		switch (fd.getType()) {
		case BOOL:
			return false;
		case BYTES:
			return "".getBytes();
		case MESSAGE: {
			if ("Qualifier".equals(fd.getMessageType().getName())) { return defaultQualifier(); }
			if ("StructuredProperty".equals(fd.getMessageType().getName())) { return StructuredProperty.newBuilder().setValue("")
					.setQualifier(defaultQualifier()).build(); }
			if ("KeyValue".equals(fd.getMessageType().getName())) { return KeyValue.newBuilder().setKey("").setValue("").build(); }
			return null;
		}
		case SFIXED32:
		case SFIXED64:
		case SINT32:
		case SINT64:
		case INT32:
		case INT64:
		case DOUBLE:
		case FIXED32:
		case FIXED64:
		case FLOAT:
			return 0;
		case STRING:
			return "";
		default:
			return null;
		}
	}

	private Qualifier defaultQualifier() {
		return Qualifier.newBuilder().setClassid("").setClassname("").setSchemeid("").setSchemename("").build();
	}

	@SuppressWarnings("unchecked")
	private void addFieldValue(final List<String> metadata, final FieldDescriptor fd, final Object value, final boolean expandingRel) {
		if (fd.getName().equals("dateofcollection") || fd.getName().equals("id")) { return; }

		if (fd.getName().equals("datasourcetype")) {
			String classid = ((Qualifier) value).getClassid();

			Qualifier.Builder q = Qualifier.newBuilder((Qualifier) value);
			if (specialDatasourceTypes.contains(classid)) {
				q.setClassid("other").setClassname("other");
			}
			metadata.add(asXmlElement("datasourcetypeui", "", q.build()));
		}

		if (fd.isRepeated() && (value instanceof List<?>)) {
			for (Object o : (List<Object>) value) {
				guessType(metadata, fd, o, expandingRel);
			}
		} else {
			guessType(metadata, fd, value, expandingRel);
		}
	}

	private void guessType(final List<String> metadata, final FieldDescriptor fd, final Object o, final boolean expandingRel) {

		if (fd.getType().equals(FieldDescriptor.Type.MESSAGE)) {

			if (Qualifier.getDescriptor().equals(fd.getMessageType())) {
				Qualifier qualifier = (Qualifier) o;
				metadata.add(asXmlElement(fd.getName(), "", qualifier));
			}

			if (StructuredProperty.getDescriptor().equals(fd.getMessageType())) {
				StructuredProperty sp = (StructuredProperty) o;
				metadata.add(asXmlElement(fd.getName(), sp.getValue(), sp.getQualifier()));
			}

			if (KeyValue.getDescriptor().equals(fd.getMessageType())) {
				KeyValue kv = (KeyValue) o;
				metadata.add("<" + fd.getName() + " name=\"" + escapeXml(kv.getValue()) + "\" id=\"" + escapeXml(removePrefix(kv.getKey())) + "\"/>");
			}

			if (Journal.getDescriptor().equals(fd.getMessageType()) && (o != null)) {
				Journal j = (Journal) o;
				metadata.add("<journal " + "issn=\"" + escapeXml(j.getIssnPrinted()) + "\" " + "eissn=\"" + escapeXml(j.getIssnOnline()) + "\" " + "lissn=\""
						+ escapeXml(j.getIssnLinking()) + "\">" + j.getName() + "</journal>");
			}

			if (Context.getDescriptor().equals(fd.getMessageType()) && (o != null)) {
				handleContext(metadata, fd, o, expandingRel);
			}
		} else if (fd.getType().equals(FieldDescriptor.Type.ENUM)) {
			// Skipped
		} else if (fd.getName().contains("fundingtree")) {
			handleFundingTree(metadata, fd, o, expandingRel);
		} else {
			metadata.add(asXmlElement(fd.getName(), o.toString(), null));
		}
	}

	private void handleContext(final List<String> metadata, final FieldDescriptor fd, final Object o, final boolean expandingRel) {
		Result.Context context = (Result.Context) o;
		String fullid = context.getId();
		List<String> ids = Lists.newArrayList();
		if (contextMapper.isEmpty()) { return; }

		while (fullid.contains("::")) {
			fullid = StringUtils.substringBeforeLast(fullid, "::");
			ids.add(fullid);
		}
		for (String id : ids) {
			ContextDef def = contextMapper.get(id);
			metadata.add("<" + def.getElement() + " id=\"" + id + "\" label=\"" + def.getLabel() + "\"/>");
		}
	}

	@SuppressWarnings("unchecked")
	private void handleFundingTree(final List<String> metadata, final FieldDescriptor fd, final Object o, final boolean expandingRel) {
		String xmlTree = asXmlJSon(fd.getName(), o.toString());
		if (expandingRel) {
			try {
				Document ftree = new SAXReader().read(new StringReader(xmlTree));

				int i = 0;
				String funding = "<funding>";
				String _id = "";

				for (Object id : Lists.reverse(ftree.selectNodes("//fundingtree//name"))) {
					_id += ((Element) id).getText();
					funding += "<funding_level_" + i + ">" + escapeXml(_id) + "</funding_level_" + i + ">";
					_id += "::";
					i++;
				}
				funding += "</funding>";
				// System.out.println("-------------------------------\n" + xmlTree + "\n" + funding);
				metadata.add(funding);
			} catch (DocumentException e) {
				System.err.println("unable to parse funding tree: " + xmlTree + "\n" + e.getMessage());
			}
		} else {
			metadata.add(xmlTree);
		}
	}

	private String asXmlJSon(final String root, final String json) {
		try {
			if ((json == null) || json.isEmpty()) { return "<" + root + "/>"; }
			JSONObject o = new JSONObject(json);

			String xml = org.json.XML.toString(o, root);
			return xml;
		} catch (Exception e) {
			System.err.println("unable to parse json: " + json + "\n" + e.getMessage());
			return "<" + root + "/>";
		}
	}

	private String asXmlElement(final String name, final String value, final Qualifier q) {
		StringBuilder sb = new StringBuilder();
		sb.append("<");
		sb.append(name);
		sb.append(getAttributes(q));
		if ((value == null) || value.isEmpty()) {
			sb.append("/>");
			return sb.toString();
			// return "<" + name + getAttributes(q) + "/>";
		}

		sb.append(">");
		sb.append(escapeXml(Normalizer.normalize(value, Normalizer.Form.NFD)));
		sb.append("</");
		sb.append(name);
		sb.append(">");

		return sb.toString();
		// return "<" + name + getAttributes(q) + ">" + escapeXml(value) + "</" + name + ">";
	}

	private String getAttributes(final Qualifier q) {
		if (q == null) { return ""; }

		StringBuilder sb = new StringBuilder();
		for (Entry<FieldDescriptor, Object> e : q.getAllFields().entrySet()) {
			// sb.append(" " + e.getKey().getName() + "=\"" + escapeXml(e.getValue().toString()) + "\"");
			sb.append(" ");
			sb.append(e.getKey().getName());
			sb.append("=\"");
			sb.append(escapeXml(Normalizer.normalize(e.getValue().toString(), Normalizer.Form.NFD)));
			sb.append("\"");
		}
		return sb.toString();
	}

	private Set<Entry<FieldDescriptor, Object>> filterFields(final GeneratedMessage fields, final Set<String> filter) {

		if (filter != null) {
			Predicate<FieldDescriptor> p = new Predicate<FieldDescriptor>() {

				@Override
				public boolean apply(final FieldDescriptor descriptor) {
					if (fields == null) { return false; }
					return filter.contains(descriptor.getName());
				}
			};
			Map<FieldDescriptor, Object> filtered = Maps.filterKeys(fields.getAllFields(), p);
			// System.out.println(
			// "filtered " + type.toString() + ": " + toString(filterEntries.keySet()) + "\n" +
			// "builder  " + fields.getDescriptorForType().getFullName() + ": " + toString(fields.getAllFields().keySet()));
			return filtered.entrySet();
		}
		return fields.getAllFields().entrySet();
	}

	public static String removePrefix(final String s) {
		if (s.contains("|")) { return StringUtils.substringAfter(s, "|"); }
		return s;
	}

	public static String escapeXml(final String value) {
		return StringEscapeUtils.escapeXml(value).replaceAll("&", "&amp;").replaceAll("<", "&lt;").replaceAll(">", "&gt;");
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append("################################################\n");
		sb.append("ID: ").append(id).append("\n");
		if (mainEntity != null) {
			sb.append("MAIN ENTITY:\n").append(mainEntity.getEntity().toString() + "\n");
		}
		if (relations != null) {
			sb.append("\nRELATIONS:\n");
			for (OafRel rel : relations.values()) {
				sb.append(rel.toString() + "\n");
			}
		}
		if (children != null) {
			sb.append("\nCHILDREN:\n");
			for (OafRel rel : children.values()) {
				sb.append(rel.toString() + "\n");
			}
		}
		return sb.toString();
	}

}
