package eu.dnetlib.data.mapreduce.hbase.dedup;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Map;

import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableMapper;
import org.apache.hadoop.hbase.util.Bytes;

import com.google.protobuf.InvalidProtocolBufferException;

import eu.dnetlib.data.mapreduce.hbase.dedup.config.DedupConfig;
import eu.dnetlib.data.mapreduce.hbase.dedup.config.DedupConfigLoader;
import eu.dnetlib.data.mapreduce.util.DedupRootUtils;
import eu.dnetlib.data.proto.DedupRelProtos.DedupRel;
import eu.dnetlib.data.proto.KindProtos.Kind;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.OafProtos.OafRel;
import eu.dnetlib.data.proto.OafProtos.OafRel.Builder;
import eu.dnetlib.data.proto.RelTypeProtos.RelType;
import eu.dnetlib.data.transform.xml.AbstractDNetOafXsltFunctions;

public class DedupFindRootsMapper extends TableMapper<ImmutableBytesWritable, Put> {
	
	private static final boolean WRITE_TO_WAL = false;

	private DedupConfig dedupConf;

	@Override
	protected void setup(Context context) throws IOException, InterruptedException {
		dedupConf = DedupConfigLoader.load(context.getConfiguration().get("dedup.wf.conf"));
		System.out.println("dedup findRoots mapper\nwf conf: " + dedupConf.toString());
	}

	@Override
	protected void map(ImmutableBytesWritable rowkey, Result value, Context context) throws IOException, InterruptedException {
		//System.out.println("Find root mapping: " + new String(rowkey.copyBytes()));

		Map<byte[], byte[]> similarRels = value.getFamilyMap(Bytes.toBytes(RelType.similarRel.toString()));

		if (similarRels != null && !similarRels.isEmpty()) {
			ByteBuffer min = findMin(ByteBuffer.wrap(rowkey.get()), similarRels.keySet());
			
			byte[] row = rowkey.copyBytes();
			byte[] root = DedupRootUtils.newIdBytes(min, dedupConf.getDedupRun());

			System.out.println("Found root: " + new String(root));

			emitDedupRel(context, "dedupRel", row, root, buildRel(row, root));
			emitDedupRel(context, "dedupRel", root, row, buildRel(root, row));

			context.getCounter("DedupFindRootsMapper", "dedupRel (x2)").increment(1);

			//marks the original body deleted
			emitBody(context, row, value.getValue(Bytes.toBytes(dedupConf.getEntityName()), Bytes.toBytes("body")));

		} else {
			context.getCounter("DedupFindRootsMapper", "row not in similarity mesh").increment(1);
		}
	}

	private ByteBuffer findMin(ByteBuffer min, Iterable<byte[]> keys) {
		for (byte[] q : keys) {
			ByteBuffer iq = ByteBuffer.wrap(q);
			if (min.compareTo(iq) > 0) {
				min = iq;
			}
		}
		return min;
	}

	private void emitBody(Context context, byte[] row, byte[] body) throws InvalidProtocolBufferException, IOException, InterruptedException {
		if (body == null) {
			context.getCounter(dedupConf.getEntityName(), "missing body").increment(1);
			System.err.println("missing body: " + new String(row));
			return;
		}
		final Oaf prototype = Oaf.parseFrom(body);

		if (prototype.getDataInfo().getDeletedbyinference()) {
			context.getCounter("DedupFindRootsMapper", "bodies already deleted").increment(1);
		} else {
			Oaf.Builder oafRoot = Oaf.newBuilder(prototype);
			oafRoot.getDataInfoBuilder().setDeletedbyinference(true).setInferred(true).setInferenceprovenance("dedup");
			byte[] family = Bytes.toBytes(dedupConf.getEntityName());
			byte[] qualifier = Bytes.toBytes("body");
			Put put = new Put(row).add(family, qualifier, oafRoot.build().toByteArray());
			put.setWriteToWAL(WRITE_TO_WAL);
			context.write(new ImmutableBytesWritable(row), put);
			context.getCounter("DedupFindRootsMapper", "bodies marked deleted").increment(1);
		}
	}

	private byte[] buildRel(byte[] from, byte[] to) {
		Builder oafRel = OafRel.newBuilder().setRelType(RelType.dedupRel).setDedupRel(DedupRel.newBuilder()).setChild(false).setSource(new String(from))
				.setTarget(new String(to));
		Oaf oaf = Oaf.newBuilder().setKind(Kind.relation).setTimestamp(System.currentTimeMillis())
				.setDataInfo(AbstractDNetOafXsltFunctions.getDataInfo(null, "0.8", false, true).setInferenceprovenance("dedup")).setRel(oafRel).build();
		return oaf.toByteArray();
	}

	private void emitDedupRel(Context context, String relName, byte[] from, byte[] to, byte[] value) throws IOException, InterruptedException {
		Put put = new Put(from).add(Bytes.toBytes(relName), to, value);
		context.write(new ImmutableBytesWritable(from), put);
	}

}
