package eu.dnetlib.data.mapreduce.hbase.index;

import java.io.IOException;

import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Reducer;

import eu.dnetlib.data.mapreduce.JobParams;
import eu.dnetlib.data.mapreduce.hbase.index.config.ContextMapper;
import eu.dnetlib.data.mapreduce.hbase.index.config.EntityLinkTable;
import eu.dnetlib.data.mapreduce.hbase.index.config.IndexConfig;
import eu.dnetlib.data.mapreduce.util.OafDecoder;
import eu.dnetlib.data.mapreduce.util.XmlRecordFactory;
import eu.dnetlib.data.proto.TypeProtos.Type;

public class PrepareFeedReducer extends Reducer<Text, ImmutableBytesWritable, Text, Text> {

	private EntityLinkTable entityLinkTable;

	private ContextMapper contextMapper = new ContextMapper();

	private final boolean entityDefaults = true;
	private final boolean relDefaults = false;
	private final boolean childDefaults = false;

	@Override
	protected void setup(final Context context) throws IOException, InterruptedException {
		String json = context.getConfiguration().get(JobParams.INDEX_ENTITY_LINKS);
		System.out.println(JobParams.INDEX_ENTITY_LINKS + ":\n" + json);
		entityLinkTable = IndexConfig.load(json).getLinkMap();

		json = context.getConfiguration().get("contextmap");
		System.out.println("contextmap:\n" + json);
		contextMapper.fromJson(json);
	}

	@Override
	protected void reduce(final Text key, final Iterable<ImmutableBytesWritable> values, final Context context) throws IOException, InterruptedException {

		final XmlRecordFactory builder = new XmlRecordFactory(entityLinkTable, contextMapper, entityDefaults, relDefaults, childDefaults);
		decodeValues(values, builder);

		if (builder.isValid()) {
			context.write(key, new Text(builder.build()));
			// context.getCounter("entity", keyDecoder.getType().toString()).increment(1);
		} else {
			// System.err.println("invalid " + getType(key) + ": " + key.toString());
			context.getCounter("missing body", getType(key)).increment(1);
		}
	}

	private void decodeValues(final Iterable<ImmutableBytesWritable> values, final XmlRecordFactory builder) {
		for (ImmutableBytesWritable bytes : values) {
			final OafDecoder decoder = OafDecoder.decode(bytes);

			switch (decoder.getKind()) {
			case entity:
				builder.setMainEntity(decoder.getEntity());
				break;
			case relation:
				if (decoder.getOafRel().getChild()) {
					builder.addChild(decoder.getOafRel());
				} else {
					builder.addRelation(decoder.getOafRel());
				}
				break;
			default:
				throw new IllegalArgumentException("unknow type: " + decoder.getKind());
			}
		}
	}

	private String getType(final Text key) {
		try {
			return Type.valueOf(Type.getDescriptor().findValueByNumber(Integer.parseInt(StringUtils.left(key.toString(), 2)))).toString();
		} catch (Throwable e) {
			return "record";
		}
	}

}
