package eu.dnetlib.data.mapreduce.hbase.dedup;

import java.io.IOException;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.PriorityQueue;
import java.util.Queue;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableReducer;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.Text;

import com.google.common.collect.Lists;

import eu.dnetlib.data.proto.RelTypeProtos.RelType;
import eu.dnetlib.pace.clustering.NGramUtils;
import eu.dnetlib.pace.config.Config;
import eu.dnetlib.pace.config.DynConf;
import eu.dnetlib.pace.distance.PaceDocumentDistance;
import eu.dnetlib.pace.model.Field;
import eu.dnetlib.pace.model.MapDocument;
import eu.dnetlib.pace.model.MapDocumentComparator;
import eu.dnetlib.pace.model.MapDocumentSerializer;
import eu.dnetlib.pace.util.DedupConfig;
import eu.dnetlib.pace.util.DedupConfigLoader;

public class DedupReducer extends TableReducer<Text, ImmutableBytesWritable, ImmutableBytesWritable> {

	private static final boolean WRITE_TO_WAL = false;
	private static final int LIMIT = 2000;
	private static final int FIELD_LIMIT = 10;
	private static final int WINDOW_SIZE = 200;

	private Config paceConf;
	private DedupConfig dedupConf;

	@Override
	protected void setup(Context context) throws IOException, InterruptedException {
		paceConf = DynConf.load(context.getConfiguration().get("dedup.pace.conf"));
		dedupConf = DedupConfigLoader.load(context.getConfiguration().get("dedup.wf.conf"));

		System.out.println("dedup reduce phase \npace conf: " + paceConf.fields() + "\nwf conf: " + dedupConf.toString());
	}

	@Override
	protected void reduce(Text key, Iterable<ImmutableBytesWritable> values, Context context) throws IOException, InterruptedException {
		System.out.println("\nReducing key: '" + key + "'");

		final Queue<MapDocument> q = prepare(context, key, values);
		switch (dedupConf.getEntityType()) {
		case person:
			process(q, context);
			break;
		case result:
			process(simplifyQueue(q, key.toString(), context), context);
			break;
		case organization:
			process(q, context);
			break;
		default:
			throw new IllegalArgumentException("dedup not implemented for type: " + dedupConf.getEntityName());
		}
	}

	private Queue<MapDocument> prepare(Context context, Text key, Iterable<ImmutableBytesWritable> values) {
		final Queue<MapDocument> queue = new PriorityQueue<MapDocument>(100, new MapDocumentComparator(dedupConf.getOrderField()));

		final Set<String> seen = new HashSet<String>();

		for (ImmutableBytesWritable i : values) {
			try {
				MapDocument doc = MapDocumentSerializer.decode(i.copyBytes());
				String id = doc.getIdentifier();

				if (!seen.contains(id)) {
					seen.add(id);
					queue.add(doc);
				}

				if (queue.size() > LIMIT) {
					//context.getCounter("ngram size > " + LIMIT, "'" + key.toString() + "', --> " + context.getTaskAttemptID()).increment(1);
					context.getCounter("ngram size > " + LIMIT, "N").increment(1);
					System.out.println("breaking out after limit (" + LIMIT + ") for ngram '" + key);
					break;
				}
			} catch (Throwable e) {
				System.out.println("Got exception: " + e);
			}
		}

		return queue;
	}

	private Queue<MapDocument> simplifyQueue(Queue<MapDocument> queue, String ngram, Context context) {
		final Queue<MapDocument> q = new LinkedList<MapDocument>();

		String fieldRef = "";
		List<MapDocument> tempResults = Lists.newArrayList();

		while (!queue.isEmpty()) {
			MapDocument result = queue.remove();

			if (!result.values(dedupConf.getOrderField()).isEmpty()) {
				String field = NGramUtils.cleanupForOrdering(result.values(dedupConf.getOrderField()).get(0).getValue().toString());
				if (field.equals(fieldRef)) {
					tempResults.add(result);
				} else {
					populateSimplifiedQueue(q, tempResults, context, fieldRef, ngram);
					tempResults.clear();
					tempResults.add(result);
					fieldRef = field;
				}
			} else {
				context.getCounter("Records", "Without " + dedupConf.getOrderField()).increment(1);
			}
		}
		populateSimplifiedQueue(q, tempResults, context, fieldRef, ngram);

		return q;
	}

	private void populateSimplifiedQueue(Queue<MapDocument> q, List<MapDocument> tempResults, Context context, String fieldRef, String ngram) {
		if (tempResults.size() < FIELD_LIMIT) {
			q.addAll(tempResults);
		} else {
			context.getCounter(getClass().getSimpleName(), "Skipped records for count(" + dedupConf.getOrderField() + ") >= " + FIELD_LIMIT).increment(
					tempResults.size());
			System.out.println("Skipped field: " + fieldRef + " - size: " + tempResults.size() + " - ngram: " + ngram);
		}
	}

	private void process(Queue<MapDocument> queue, Context context) throws IOException, InterruptedException {

		final PaceDocumentDistance algo = new PaceDocumentDistance();

		while (!queue.isEmpty()) {

			final MapDocument pivot = queue.remove();
			final String idPivot = pivot.getIdentifier();

			final List<Field> fieldsPivot = pivot.values(dedupConf.getOrderField());
			final String fieldPivot = fieldsPivot == null || fieldsPivot.isEmpty() ? null : fieldsPivot.get(0).getValue().toString();

			if (fieldPivot != null) {
				System.out.println(idPivot + " --> " + fieldPivot);

				int i = 0;
				for (MapDocument curr : queue) {
					final String idCurr = curr.getIdentifier();

					if (mustSkip(idCurr)) {
						context.getCounter(dedupConf.getEntityName(), "skip list").increment(1);
						break;
					}

					if (i > WINDOW_SIZE) {
						break;
					}

					final List<Field> fieldsCurr = curr.values(dedupConf.getOrderField());
					final String fieldCurr = fieldsCurr == null || fieldsCurr.isEmpty() ? null : fieldsCurr.get(0).getValue().toString();

					if (!idCurr.equals(idPivot) && fieldCurr != null) {

						double d = algo.between(pivot, curr, paceConf);

						if (d >= dedupConf.getThreshold()) {
							writeSimilarity(context, idPivot, idCurr);
							context.getCounter(dedupConf.getEntityName(), "similarRel (x2)").increment(1);
						} else {
							context.getCounter(dedupConf.getEntityName(), "d < " + dedupConf.getThreshold()).increment(1);
						}
						i++;
					}
				}
			}
		}
	}

	private boolean mustSkip(final String idPivot) {
		return dedupConf.getSkipList().contains(getNsPrefix(idPivot));
	}

	private String getNsPrefix(String id) {
		return StringUtils.substringBetween(id, "|", "::");
	}

	private void writeSimilarity(Context context, String idPivot, String id) throws IOException, InterruptedException {
		byte[] rowKey = Bytes.toBytes(idPivot);
		byte[] target = Bytes.toBytes(id);

		emitRel(context, rowKey, target);
		emitRel(context, target, rowKey);
	}

	private void emitRel(Context context, byte[] from, byte[] to) throws IOException, InterruptedException {
		Put put = new Put(from).add(Bytes.toBytes(RelType.similarRel.toString()), to, Bytes.toBytes(""));
		put.setWriteToWAL(WRITE_TO_WAL);
		context.write(new ImmutableBytesWritable(from), put);
	}

}
