package eu.dnetlib.data.mapreduce.util;

import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.protobuf.Descriptors.FieldDescriptor;
import com.google.protobuf.Message.Builder;

import eu.dnetlib.data.mapreduce.hbase.dedup.TrustOrdering;
import eu.dnetlib.data.proto.KeyValueProtos.KeyValue;
import eu.dnetlib.data.proto.KindProtos.Kind;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.PersonProtos.Person;
import eu.dnetlib.data.proto.ResultProtos.Result;
import eu.dnetlib.data.proto.SpecialTrustProtos.SpecialTrust;
import eu.dnetlib.data.proto.StructuredPropertyProtos.StructuredProperty;

public class OafEntityMerger {

	private final Predicate<StructuredProperty> filterMainTitles = new Predicate<StructuredProperty>() {

		@Override
		public boolean apply(StructuredProperty sp) {
			return sp.getQualifier() != null && sp.getQualifier().getClassname().equals("main title");
		}
	};

	private final Predicate<String> skipEmpty = new Predicate<String>() {

		@Override
		public boolean apply(String s) {
			return s != null && !s.isEmpty();
		}
	};

	public Oaf.Builder mergeEntities(String id, Iterable<Oaf> entities) {

		Oaf.Builder builder = Oaf.newBuilder();
		String trust = "0.0";
		for (Oaf oaf : new TrustOrdering().immutableSortedCopy(entities)) {
			// doublecheck we're dealing only with main entities
			if (!oaf.getKind().equals(Kind.entity)) { throw new IllegalArgumentException("expected OafEntity!"); }

			String currentTrust = oaf.getDataInfo().getTrust();
			if (!currentTrust.equals(SpecialTrust.NEUTRAL.toString())) {
				trust = currentTrust;
			}
			builder.mergeFrom(oaf);
		}

		doMergeEntities(builder);

		builder.getEntityBuilder().setId(id);
		builder.getDataInfoBuilder().setInferred(true).setDeletedbyinference(false).setTrust(trust);

		return builder;
	}

	public void doMergeEntities(Oaf.Builder builder) {

		switch (builder.getEntity().getType()) {
		case datasource:
			break;
		case organization:
			break;
		case person:
			Person.Metadata.Builder person = builder.getEntityBuilder().getPersonBuilder().getMetadataBuilder();
			for (String field : Lists.newArrayList("secondnames")) {
				setSingleString(person, field);
			}
			break;
		case project:
			break;
		case result:
			Result.Metadata.Builder result = builder.getEntityBuilder().getResultBuilder().getMetadataBuilder();
			setTitle(result);

			for (String field : Lists.newArrayList("subject", "relevantdate")) {
				setStructuredProperty(result, field);
			}
			for (String field : Lists.newArrayList("description")) {
				setSingleString(result, field);
			}
			for (String field : Lists.newArrayList("collectedfrom")) {
				setKeyValues(builder.getEntityBuilder(), field);
			}
			for (String field : Lists.newArrayList("pid")) {
				setStructuredProperty(builder.getEntityBuilder(), field);
			}
			for (String field : Lists.newArrayList("originalId")) {
				setStrings(builder.getEntityBuilder(), field);
			}
			break;
		default:
			break;
		}
	}

	/**
	 * Helper method, avoid duplicated StructuredProperties in the given builder for the given fieldName
	 * 
	 * @param builder
	 * @param fieldName
	 */
	@SuppressWarnings("unchecked")
	private void setStructuredProperty(Builder builder, String fieldName) {
		final Map<String, StructuredProperty> map = Maps.newHashMap();
		final FieldDescriptor fd = builder.getDescriptorForType().findFieldByName(fieldName);
		final List<StructuredProperty> sps = (List<StructuredProperty>) builder.getField(fd);

		if (sps != null && !sps.isEmpty()) {
			for (StructuredProperty sp : sps) {
				map.put(sp.getValue(), sp);
			}

			if (!map.isEmpty()) {
				builder.clearField(fd).setField(fd, Lists.newArrayList(map.values()));
			}
		}
	}

	/**
	 * Helper method, avoid duplicated KeyValues in the given builder for the given fieldName
	 * 
	 * @param builder
	 * @param fieldName
	 */
	@SuppressWarnings("unchecked")
	private void setKeyValues(Builder builder, String fieldName) {
		final Map<String, KeyValue> map = Maps.newHashMap();
		final FieldDescriptor fd = builder.getDescriptorForType().findFieldByName(fieldName);
		final List<KeyValue> kvs = (List<KeyValue>) builder.getField(fd);

		if (kvs != null && !kvs.isEmpty()) {
			for (KeyValue sp : kvs) {
				map.put(sp.getKey(), sp);
			}

			if (!map.isEmpty()) {
				builder.clearField(fd).setField(fd, Lists.newArrayList(map.values()));
			}
		}
	}

	@SuppressWarnings("unchecked")
	private void setSingleString(Builder builder, String fieldName) {

		final FieldDescriptor fd = builder.getDescriptorForType().findFieldByName(fieldName);
		final List<String> field = (List<String>) builder.getField(fd);
		if (field != null && !field.isEmpty()) {
			final String s = Iterables.getLast(Iterables.filter(field, skipEmpty), "");

			if (s != null && !s.isEmpty()) {
				builder.clearField(fd).setField(fd, Lists.newArrayList(s));
			}
		}
	}

	@SuppressWarnings("unchecked")
	private void setStrings(Builder builder, String fieldName) {

		final FieldDescriptor fd = builder.getDescriptorForType().findFieldByName(fieldName);
		final List<String> field = (List<String>) builder.getField(fd);
		final Set<String> set = Sets.newHashSet();
		if (field != null && !field.isEmpty()) {
			for (String s : Iterables.filter(field, skipEmpty)) {
				set.add(s);
			}

			builder.clearField(fd).setField(fd, Lists.newArrayList(set));
		}
	}

	private void setTitle(Result.Metadata.Builder metadata) {
		Iterable<StructuredProperty> filtered = Iterables.filter(metadata.getTitleList(), filterMainTitles);

		if (!Iterables.isEmpty(filtered)) {
			metadata.clearTitle().addTitle(Iterables.getLast(filtered));
		}
	}

}
