package eu.dnetlib.data.mapreduce.util;

import static eu.dnetlib.miscutils.collections.MappedCollection.listMap;

import java.io.StringReader;
import java.io.StringWriter;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.UUID;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.commons.lang.StringUtils;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.json.JSONException;
import org.json.JSONObject;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.protobuf.Descriptors.EnumValueDescriptor;
import com.google.protobuf.Descriptors.FieldDescriptor;
import com.google.protobuf.GeneratedMessage;
import com.mycila.xmltool.XMLDoc;
import com.mycila.xmltool.XMLTag;

import eu.dnetlib.data.mapreduce.hbase.index.config.ContextDef;
import eu.dnetlib.data.mapreduce.hbase.index.config.ContextMapper;
import eu.dnetlib.data.mapreduce.hbase.index.config.EntityConfigTable;
import eu.dnetlib.data.proto.DataInfoProtos.DataInfo;
import eu.dnetlib.data.proto.KeyValueProtos.KeyValue;
import eu.dnetlib.data.proto.OafProtos.OafEntity;
import eu.dnetlib.data.proto.OafProtos.OafRel;
import eu.dnetlib.data.proto.QualifierProtos.Qualifier;
import eu.dnetlib.data.proto.RelMetadataProtos.RelMetadata;
import eu.dnetlib.data.proto.RelTypeProtos.RelType;
import eu.dnetlib.data.proto.ResultProtos.Result;
import eu.dnetlib.data.proto.ResultProtos.Result.Context;
import eu.dnetlib.data.proto.ResultProtos.Result.ExternalReference;
import eu.dnetlib.data.proto.ResultProtos.Result.Instance;
import eu.dnetlib.data.proto.ResultProtos.Result.Journal;
import eu.dnetlib.data.proto.StructuredPropertyProtos.StructuredProperty;
import eu.dnetlib.data.proto.TypeProtos.Type;
import eu.dnetlib.miscutils.functional.UnaryFunction;

public class XmlRecordFactory {

	protected Set<String> specialDatasourceTypes = Sets.newHashSet("scholarcomminfra", "infospace", "pubsrepository::mock", "entityregistry");

	protected TemplateFactory templateFactory = new TemplateFactory();

	protected OafDecoder mainEntity = null;

	protected String key = null;

	protected Map<String, OafDecoder> relations = Maps.newHashMap();
	protected Map<String, OafDecoder> children = Maps.newHashMap();

	protected EntityConfigTable entityConfigTable;

	protected ContextMapper contextMapper;

	protected String schemaLocation;

	protected boolean entityDefaults;
	protected boolean relDefaults;
	protected boolean childDefaults;

	protected Set<String> contextes = Sets.newHashSet();

	protected Transformer transformer;

	public XmlRecordFactory(final EntityConfigTable entityConfigTable, final ContextMapper contextMapper, final String schemaLocation,
			final boolean entityDefaults, final boolean relDefaults, final boolean childDefeaults) throws TransformerConfigurationException,
			TransformerFactoryConfigurationError {
		this.entityConfigTable = entityConfigTable;
		this.contextMapper = contextMapper;
		this.schemaLocation = schemaLocation;
		this.entityDefaults = entityDefaults;
		this.relDefaults = relDefaults;
		this.childDefaults = childDefeaults;

		transformer = TransformerFactory.newInstance().newTransformer();
		transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
	}

	public String getId() {
		return key;
	}

	public boolean isValid() {
		return mainEntity != null;
	}

	public void setMainEntity(final OafDecoder mainEntity) {
		this.mainEntity = mainEntity;
		this.key = mainEntity.decodeEntity().getId();
	}

	public void addRelation(final OafDecoder rel) {
		addRelOrChild(relations, rel);
	}

	public void addChild(final OafDecoder child) {
		addRelOrChild(children, child);
	}

	private void addRelOrChild(final Map<String, OafDecoder> map, final OafDecoder rel) {
		if (!rel.relSourceId().equals(getId())) {
			map.put(rel.relSourceId(), rel);
		} else {
			map.put(UUID.randomUUID().toString(), rel);
		}
	}

	public String build() {

		OafEntityDecoder entity = mainEntity.decodeEntity();
		// System.out.println("building");
		// System.out.println("main: " + mainEntity);
		// System.out.println("rel:  " + relations);
		// System.out.println("chi:  " + children);
		// System.out.println("=============");

		final Type type = entity.getType();
		final List<String> metadata = decodeType(entity, null, entityDefaults, false);
		List<String> rels = listRelations();
		metadata.addAll(buildContexts(type));
		metadata.add(parseDataInfo(mainEntity));
		final String body = templateFactory.buildBody(type, metadata, rels, listChildren());

		// System.out.println("record id: " + recordId);
		return templateFactory.buildRecord(type, key, entity.getDateOfCollection(), schemaLocation, body);
	}

	private String parseDataInfo(final OafDecoder decoder) {
		DataInfo dataInfo = decoder.getOaf().getDataInfo();

		StringBuilder sb = new StringBuilder();
		sb.append("<datainfo>");
		sb.append(asXmlElement("inferred", dataInfo.getInferred() + "", null));
		sb.append(asXmlElement("deletedbyinference", dataInfo.getDeletedbyinference() + "", null));
		sb.append(asXmlElement("trust", dataInfo.getTrust() + "", null));
		sb.append(asXmlElement("inferenceprovenance", dataInfo.getInferenceprovenance() + "", null));
		sb.append(asXmlElement("provenanceaction", null, dataInfo.getProvenanceaction()));
		sb.append("</datainfo>");

		return sb.toString();
	}

	private List<String> decodeType(final OafEntityDecoder decoder, final Set<String> filter, final boolean defaults, final boolean expandingRel) {

		final List<String> metadata = Lists.newArrayList();
		metadata.addAll(listFields(decoder.getMetadata(), filter, defaults, expandingRel));
		metadata.addAll(listFields(decoder.getOafEntity(), filter, defaults, expandingRel));

		if (decoder.getEntity() instanceof Result && !expandingRel) {
			metadata.add(asXmlElement("bestlicense", "", getBestLicense()));
		}

		return metadata;
	}

	private Qualifier getBestLicense() {
		Qualifier bestLicense = null;
		LicenseComparator lc = new LicenseComparator();
		for (Instance instance : ((Result) mainEntity.decodeEntity().getEntity()).getInstanceList()) {
			if (lc.compare(bestLicense, instance.getLicence()) > 0) {
				bestLicense = instance.getLicence();
			}
		}
		return bestLicense;
	}

	private List<String> listRelations() {

		final List<String> rels = Lists.newArrayList();

		for (OafDecoder decoder : this.relations.values()) {

			final OafRel rel = decoder.getOafRel();
			final OafEntity cachedTarget = rel.getCachedTarget();
			final OafRelDecoder relDecoder = OafRelDecoder.decode(rel);

			if (!relDecoder.getRelType().equals(RelType.personResult) || relDecoder.getRelTargetId().equals(key)) {

				final List<String> metadata = Lists.newArrayList();
				metadata.addAll(listFields(relDecoder.getRel(), null, false, true));

				String semanticclass = "";
				String semanticscheme = "";
				Type type = null;

				if (cachedTarget != null && cachedTarget.isInitialized()) {
					final Set<String> filter = entityConfigTable.getFilter(cachedTarget.getType(), relDecoder.getRelType());
					metadata.addAll(decodeType(OafEntityDecoder.decode(cachedTarget), filter, relDefaults, true));
					RelMetadata relMetadata = relDecoder.getRelMetadata();
					type = cachedTarget.getType();

					// debug
					if (relMetadata == null) {
						// System.err.println(this);
						semanticclass = semanticscheme = "UNKNOWN";
					} else {
						semanticclass = relMetadata.getSemantics().getClassname();
						semanticscheme = relMetadata.getSemantics().getSchemename();
					}
				} else {
					semanticclass = semanticscheme = rel.getRelType().toString();
					type = Type.result;
				}

				DataInfo info = decoder.getOaf().getDataInfo();
				rels.add(templateFactory.getRel(type, relDecoder.getRelSourceId(), metadata, semanticclass, semanticscheme, info.getInferred(),
						info.getTrust(), info.getInferenceprovenance(), info.getProvenanceaction().getClassid()));
			}
		}
		return rels;
	}

	private List<String> listChildren() {

		final List<String> children = Lists.newArrayList();
		for (OafDecoder decoder : this.children.values()) {
			final OafRel rel = decoder.getOafRel();
			addChildren(children, rel.getCachedTarget(), rel.getRelType());
		}
		OafEntityDecoder entity = mainEntity.decodeEntity();
		if (entity.getType().equals(Type.result)) {
			for (Instance instance : ((Result) entity.getEntity()).getInstanceList()) {
				children.add(templateFactory.getInstance(instance.getHostedby().getKey(), listFields(instance, null, false, false),
						listMap(instance.getUrlList(), new UnaryFunction<String, String>() {

							@Override
							public String evaluate(final String identifier) {
								return templateFactory.getWebResource(identifier);
							}
						})));
			}
			for (ExternalReference er : ((Result) entity.getEntity()).getExternalReferenceList()) {
				// Set<String> filters = entityConfigTable.getFilter(Type.result, RelType.resultResult);
				List<String> fields = listFields(er, null, false, false);
				children.add(templateFactory.getChild("externalreference", null, fields));
			}
		}

		return children;
	}

	private void addChildren(final List<String> children, final OafEntity target, final RelType relType) {
		final OafEntityDecoder decoder = OafEntityDecoder.decode(target);
		Set<String> filters = entityConfigTable.getFilter(target.getType(), relType);
		children.add(templateFactory.getChild(decoder.getType().toString(), decoder.getId(), listFields(decoder.getMetadata(), filters, childDefaults, false)));
	}

	// //////////////////////////////////

	private List<String> listFields(final GeneratedMessage fields, final Set<String> filter, final boolean defaults, final boolean expandingRel) {

		final List<String> metadata = Lists.newArrayList();

		if (fields != null) {

			Set<String> seen = Sets.newHashSet();
			for (Entry<FieldDescriptor, Object> e : filterFields(fields, filter)) {

				// final String name = getFieldName(e.getKey().getName());
				final String name = e.getKey().getName();
				seen.add(name);

				addFieldValue(metadata, e.getKey(), e.getValue(), expandingRel);
			}

			if (defaults) {
				for (FieldDescriptor fd : fields.getDescriptorForType().getFields()) {
					if (!seen.contains(fd.getName())) {
						addFieldValue(metadata, fd, getDefault(fd), expandingRel);
					}
				}
			}
		}
		return metadata;
	}

	private Object getDefault(final FieldDescriptor fd) {
		switch (fd.getType()) {
		case BOOL:
			return false;
		case BYTES:
			return "".getBytes();
		case MESSAGE: {
			if (Qualifier.getDescriptor().equals(fd.getMessageType())) { return defaultQualifier(); }
			if (StructuredProperty.getDescriptor().equals(fd.getMessageType())) { return StructuredProperty.newBuilder().setValue("")
					.setQualifier(defaultQualifier()).build(); }
			if (KeyValue.getDescriptor().equals(fd.getMessageType())) { return KeyValue.newBuilder().setKey("").setValue("").build(); }
			return null;
		}
		case SFIXED32:
		case SFIXED64:
		case SINT32:
		case SINT64:
		case INT32:
		case INT64:
		case DOUBLE:
		case FIXED32:
		case FIXED64:
		case FLOAT:
			return 0;
		case STRING:
			return "";
		default:
			return null;
		}
	}

	private Qualifier defaultQualifier() {
		return Qualifier.newBuilder().setClassid("").setClassname("").setSchemeid("").setSchemename("").build();
	}

	@SuppressWarnings("unchecked")
	private void addFieldValue(final List<String> metadata, final FieldDescriptor fd, final Object value, final boolean expandingRel) {
		if (fd.getName().equals("dateofcollection") || fd.getName().equals("id") || fd.getName().equals("url")) { return; }

		if (fd.getName().equals("datasourcetype")) {
			String classid = ((Qualifier) value).getClassid();

			Qualifier.Builder q = Qualifier.newBuilder((Qualifier) value);
			if (specialDatasourceTypes.contains(classid)) {
				q.setClassid("other").setClassname("other");
			}
			metadata.add(asXmlElement("datasourcetypeui", "", q.build()));
		}

		if (fd.isRepeated() && value instanceof List<?>) {
			for (Object o : (List<Object>) value) {
				guessType(metadata, fd, o, expandingRel);
			}
		} else {
			guessType(metadata, fd, value, expandingRel);
		}
	}

	private void guessType(final List<String> metadata, final FieldDescriptor fd, final Object o, final boolean expandingRel) {

		if (fd.getType().equals(FieldDescriptor.Type.MESSAGE)) {

			if (Qualifier.getDescriptor().equals(fd.getMessageType())) {
				Qualifier qualifier = (Qualifier) o;
				metadata.add(asXmlElement(fd.getName(), "", qualifier));
			}

			if (StructuredProperty.getDescriptor().equals(fd.getMessageType())) {
				StructuredProperty sp = (StructuredProperty) o;
				metadata.add(asXmlElement(fd.getName(), sp.getValue(), sp.getQualifier()));
			}

			if (KeyValue.getDescriptor().equals(fd.getMessageType())) {
				KeyValue kv = (KeyValue) o;
				metadata.add("<" + fd.getName() + " name=\"" + escapeXml(kv.getValue()) + "\" id=\"" + escapeXml(removePrefix(kv.getKey())) + "\"/>");
			}

			if (Journal.getDescriptor().equals(fd.getMessageType()) && o != null) {
				Journal j = (Journal) o;
				metadata.add("<journal " + "issn=\"" + escapeXml(j.getIssnPrinted()) + "\" " + "eissn=\"" + escapeXml(j.getIssnOnline()) + "\" " + "lissn=\""
						+ escapeXml(j.getIssnLinking()) + "\">" + escapeXml(j.getName()) + "</journal>");
			}

			if (Context.getDescriptor().equals(fd.getMessageType()) && o != null) {
				contextes.add(((Result.Context) o).getId());
			}
		} else if (fd.getType().equals(FieldDescriptor.Type.ENUM)) {
			if (fd.getFullName().equals("eu.dnetlib.data.proto.OafEntity.type")) { return; }
			metadata.add(asXmlElement(fd.getName(), ((EnumValueDescriptor) o).getName(), null));
		} else if (fd.getName().contains("fundingtree")) {
			handleFundingTree(metadata, fd, o, expandingRel);
		} else {
			metadata.add(asXmlElement(fd.getName(), o.toString(), null));
		}
	}

	private List<String> buildContexts(final Type type) {
		final List<String> res = Lists.newArrayList();

		if (contextMapper != null && !contextMapper.isEmpty() && type.equals(Type.result)) {

			XMLTag document = XMLDoc.newDocument(true).addRoot("contextRoot");

			for (String id : contextes) {

				StringTokenizer st = new StringTokenizer(id, "::");
				String token = "";
				while (st.hasMoreTokens()) {
					token += st.nextToken();

					final ContextDef def = contextMapper.get(token);

					if (def == null) { throw new IllegalStateException("cannot find context for id: " + token); }

					if (def.getName().equals("context")) {
						String xpath = "//context/@id='" + def.getId() + "'";
						if (!document.gotoRoot().rawXpathBoolean(xpath, new Object())) {
							document = addContextDef(document.gotoRoot(), def);
						}
					}

					if (def.getName().equals("category")) {
						String rootId = StringUtils.substringBefore(def.getId(), "::");
						document = addContextDef(document.gotoRoot().gotoTag("//context[./@id='" + rootId + "']", new Object()), def);
					}

					if (def.getName().equals("concept")) {
						document = addContextDef(document, def).gotoParent();
					}
					token += "::";
				}
			}

			for (org.w3c.dom.Element x : document.gotoRoot().getChildElement()) {
				try {
					res.add(asStringElement(x));
				} catch (TransformerException e) {
					throw new RuntimeException(e);
				}
			}
		}

		return res;
	}

	private XMLTag addContextDef(final XMLTag tag, final ContextDef def) {
		tag.addTag(def.getName()).addAttribute("id", def.getId()).addAttribute("label", def.getLabel());
		if (def.getType() != null && !def.getType().isEmpty()) {
			tag.addAttribute("type", def.getType());
		}
		return tag;
	}

	private String asStringElement(final org.w3c.dom.Element element) throws TransformerException {
		StringWriter buffer = new StringWriter();
		transformer.transform(new DOMSource(element), new StreamResult(buffer));
		return buffer.toString();
	}

	@SuppressWarnings("unchecked")
	private void handleFundingTree(final List<String> metadata, final FieldDescriptor fd, final Object o, final boolean expandingRel) {
		String xmlTree = asXmlJSon(fd.getName(), o.toString());
		if (expandingRel) {
			try {
				Document ftree = new SAXReader().read(new StringReader(xmlTree));

				int i = 0;
				String funding = "<funding>";
				String _id = "";

				for (Object id : Lists.reverse(ftree.selectNodes("//fundingtree//name"))) {
					_id += ((Element) id).getText();
					funding += "<funding_level_" + i + ">" + escapeXml(_id) + "</funding_level_" + i + ">";
					_id += "::";
					i++;
				}
				funding += "</funding>";
				// System.out.println("-------------------------------\n" + xmlTree + "\n" + funding);
				metadata.add(funding);
			} catch (DocumentException e) {
				System.err.println("unable to parse funding tree: " + xmlTree + "\n" + e.getMessage());
			}
		} else {
			metadata.add(xmlTree);
		}
	}

	private String asXmlJSon(final String root, final String json) {
		try {
			if (json == null || json.isEmpty()) { return "<" + root + "/>"; }
			JSONObject o = new JSONObject(json.replace("'", ""));

			String contextId = parseFundingJson(o).toLowerCase();
			contextes.add(contextId);

			String xml = org.json.XML.toString(o, root);
			return xml;
		} catch (Exception e) {
			System.err.println("unable to parse json: " + json + "\n" + e.getMessage());
			return "<" + root + "/>";
		}
	}

	private String parseFundingJson(final JSONObject o) {
		try {
			String key = (String) Iterators.getOnlyElement(o.keys());
			JSONObject obj = o.getJSONObject(key);

			String id = obj.getString("id").toLowerCase();
			if (id.startsWith("welcometrust::")) {
				id = StringUtils.substringBeforeLast("uk::" + id.replace("welcometrust", "wt"), "::") + "::" + cleanup(id);
			} else if (id.startsWith("wt::wt")) {
				id = StringUtils.substringBeforeLast(id.replaceFirst("wt", "uk"), "::") + "::" + cleanup(id);
			} else if (id.startsWith("corda_______::")) {
				id = id.replace("corda_______::", "ec::");
			}

			String label = obj.getString("name");

			if (key.endsWith("level_0")) {

				if (id.equals("uk::wt")) {
					label = "Wellcome Trust Funding Stream";
				}
				contextMapper.put(id, new ContextDef(id, label, "category", ""));

				if (id.startsWith("ec::")) {
					contextMapper.put("ec", new ContextDef("ec", "European Community", "context", "funding"));
				} else if (id.startsWith("uk::")) {
					contextMapper.put("uk", new ContextDef("uk", "United Kingdom", "context", "funding"));
				}
			} else {
				contextMapper.put(id, new ContextDef(id, label, "concept", ""));
				parseFundingJson(obj.getJSONObject("parent"));
			}

			return id;
		} catch (JSONException e) {
			throw new RuntimeException(e);
		}
	}

	private String cleanup(final String id) {
		return StringUtils.substring(StringUtils.deleteWhitespace(StringUtils.substringAfterLast(id, "::").replaceAll("[^a-zA-Z]", "")), 0, 20);
	}

	private String asXmlElement(final String name, final String value, final Qualifier q) {
		StringBuilder sb = new StringBuilder();
		sb.append("<");
		sb.append(name);
		if (q != null) {
			sb.append(getAttributes(q));
		}
		if (value == null || value.isEmpty()) {
			sb.append("/>");
			return sb.toString();
			// return "<" + name + getAttributes(q) + "/>";
		}

		sb.append(">");
		// sb.append(escapeXml(Normalizer.normalize(value, Normalizer.Form.NFD)));
		sb.append(escapeXml(value));
		sb.append("</");
		sb.append(name);
		sb.append(">");

		return sb.toString();
		// return "<" + name + getAttributes(q) + ">" + escapeXml(value) + "</" + name + ">";
	}

	private String getAttributes(final Qualifier q) {
		if (q == null) { return ""; }

		StringBuilder sb = new StringBuilder();
		for (Entry<FieldDescriptor, Object> e : q.getAllFields().entrySet()) {
			// sb.append(" " + e.getKey().getName() + "=\"" + escapeXml(e.getValue().toString()) + "\"");
			sb.append(" ");
			sb.append(e.getKey().getName());
			sb.append("=\"");
			sb.append(escapeXml(e.getValue().toString()));
			sb.append("\"");
		}
		return sb.toString();
	}

	private Set<Entry<FieldDescriptor, Object>> filterFields(final GeneratedMessage fields, final Set<String> filter) {

		if (filter != null) {
			Predicate<FieldDescriptor> p = new Predicate<FieldDescriptor>() {

				@Override
				public boolean apply(final FieldDescriptor descriptor) {
					if (fields == null) { return false; }
					return filter.contains(descriptor.getName());
				}
			};
			Map<FieldDescriptor, Object> filtered = Maps.filterKeys(fields.getAllFields(), p);
			// System.out.println(
			// "filtered " + type.toString() + ": " + toString(filterEntries.keySet()) + "\n" +
			// "builder  " + fields.getDescriptorForType().getFullName() + ": " + toString(fields.getAllFields().keySet()));
			return filtered.entrySet();
		}
		return fields.getAllFields().entrySet();
	}

	public static String removePrefix(final String s) {
		if (s.contains("|")) { return StringUtils.substringAfter(s, "|"); }
		return s;
	}

	public static String escapeXml(final String value) {
		// return StringEscapeUtils.escapeXml(value).replaceAll("&", "&amp;").replaceAll("<", "&lt;").replaceAll(">", "&gt;");
		return value.replaceAll("&", "&amp;").replaceAll("<", "&lt;").replaceAll(">", "&gt;");
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append("################################################\n");
		sb.append("ID: ").append(key).append("\n");
		if (mainEntity != null) {
			sb.append("MAIN ENTITY:\n").append(mainEntity.getEntity().toString() + "\n");
		}
		if (relations != null) {
			sb.append("\nRELATIONS:\n");
			for (OafDecoder decoder : relations.values()) {
				sb.append(decoder.getOafRel().toString() + "\n");
			}
		}
		if (children != null) {
			sb.append("\nCHILDREN:\n");
			for (OafDecoder decoder : children.values()) {
				sb.append(decoder.getOafRel().toString() + "\n");
			}
		}
		return sb.toString();
	}

}
