package eu.dnetlib.functionality.modular.ui.vocabularies.controllers;

import java.io.IOException;
import java.io.StringReader;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.bind.JAXBException;

import lombok.extern.apachecommons.CommonsLog;

import org.antlr.stringtemplate.StringTemplate;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringEscapeUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;

import com.google.common.collect.Lists;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.is.registry.rmi.ISRegistryException;
import eu.dnetlib.enabling.is.registry.rmi.ISRegistryService;
import eu.dnetlib.enabling.tools.ServiceLocator;
import eu.dnetlib.functionality.modular.ui.vocabularies.adapters.SynonymAdapter;
import eu.dnetlib.functionality.modular.ui.vocabularies.adapters.TermAdapter;
import eu.dnetlib.functionality.modular.ui.vocabularies.adapters.VocabularyAdapter;
import eu.dnetlib.functionality.modular.ui.vocabularies.model.Synonym;
import eu.dnetlib.functionality.modular.ui.vocabularies.model.Term;
import eu.dnetlib.functionality.modular.ui.vocabularies.model.Vocabulary;
import eu.dnetlib.miscutils.datetime.DateUtils;

/**
 * Web controller for the UI
 * 
 * @author Andrea Mannocci
 */
@CommonsLog
@Controller
public class VocabulariesController {

	@Resource(name = "lookupLocator")
	private ServiceLocator<ISLookUpService> lookupLocator;

	@Resource(name = "registryLocator")
	private ServiceLocator<ISRegistryService> registryLocator;

	/**
	 * Returns all vocabularies serialized in json. Invoked by angularJS
	 * 
	 * @param response
	 * @throws ISLookUpException
	 * @throws IOException
	 */
	@RequestMapping("/ui/vocabularies.json")
	public void getVocabularies(final HttpServletResponse response) throws ISLookUpException, IOException {
		log.info("vocabularies.json");
		String query = "for $x in collection('/db/DRIVER/VocabularyDSResources/VocabularyDSResourceType') order by $x//VOCABULARY_NAME "
				+ "return concat ($x//RESOURCE_IDENTIFIER/@value,' §§§ ',$x//VOCABULARY_NAME,' §§§ ',$x//VOCABULARY_DESCRIPTION,' §§§ ',$x//VOCABULARY_NAME/@code)";

		List<Vocabulary> vocabularies = Lists.transform(lookupLocator.getService().quickSearchProfile(query), new VocabularyAdapter());
		Map<String, Vocabulary> map = new HashMap<String, Vocabulary>();
		for (Vocabulary voc : vocabularies) {
			map.put(voc.getId(), voc);
		}
		IOUtils.copy(new StringReader(new Gson().toJson(map)), response.getOutputStream());
	}

	/**
	 * Returns all terms and their synonyms belonging to a vocabulary. Invoked by AngularJS
	 * 
	 * @param response
	 * @param vocabularyId
	 * @throws ISLookUpException
	 * @throws IOException
	 */
	@RequestMapping("/ui/terms.json")
	public void getTerms(final HttpServletResponse response, @RequestParam(value = "vocabularyId", required = true) final String vocabularyId)
			throws ISLookUpException, IOException {
		log.info("terms.json?vocabularyId=" + vocabularyId);

		String queryTerms = "for $x in /*[.//RESOURCE_IDENTIFIER/@value=\"{vocabularyId}\"]//TERM order by $x/@english_name "
				+ "return concat ($x/@english_name,' §§§ ',$x/@native_name,' §§§ ',$x/@encoding,' §§§ ',$x/@code)";
		queryTerms = queryTerms.replace("{vocabularyId}", vocabularyId);
		List<Term> terms = Lists.newArrayList(Lists.transform(lookupLocator.getService().quickSearchProfile(queryTerms), new TermAdapter()));

		for (Term term : terms) {
			String querySynonyms = "for $x in /*[.//RESOURCE_IDENTIFIER/@value=\"{vocabularyId}\"]//TERM[@english_name=\"{term}\"]//SYNONYM order by $x/@term "
					+ "return concat ($x/@term,' §§§ ',$x/@encoding)";
			querySynonyms = querySynonyms.replace("{vocabularyId}", vocabularyId).replace("{term}", term.getEnglishName()).replace("&", "&amp;");
			// log.info(querySynonyms);
			List<Synonym> synonyms = Lists.transform(lookupLocator.getService().quickSearchProfile(querySynonyms), new SynonymAdapter());
			term.getSynonyms().addAll(synonyms);
		}

		IOUtils.copy(new StringReader(new Gson().toJson(terms)), response.getOutputStream());
	}

	@RequestMapping(value = "/ui/commitVocabulary", method = RequestMethod.POST)
	public void commitTerms(final HttpServletResponse response,
			final HttpServletRequest request,
			@RequestParam(value = "vocabularyId", required = true) final String vocabularyId) throws IOException, ISRegistryException, JAXBException {
		log.info("committing vocabulary id = " + vocabularyId);

		Type collectionType = new TypeToken<ArrayList<Term>>() {}.getType();
		ArrayList<Term> terms = new Gson().fromJson(request.getReader(), collectionType);

		// prepare terms for XML
		for (Term t : terms) {
			t.setCode(StringEscapeUtils.escapeXml(t.getCode()));
			t.setEncoding(StringEscapeUtils.escapeXml(t.getEncoding()));
			t.setEnglishName(StringEscapeUtils.escapeXml(t.getEnglishName()));
			t.setNativeName(StringEscapeUtils.escapeXml(t.getNativeName()));
			for (Synonym s : t.getSynonyms()) {
				s.setEncoding(StringEscapeUtils.escapeXml(s.getEncoding()));
				s.setTerm(StringEscapeUtils.escapeXml(s.getTerm()));
			}
		}

		StringTemplate st = new StringTemplate(IOUtils.toString(getClass().getResourceAsStream("/eu/dnetlib/functionality/modular/templates/terms.xml.st")));
		st.setAttribute("terms", terms);
		registryLocator.getService().updateProfileNode(vocabularyId, "//TERMS", st.toString());

		IOUtils.copy(new StringReader("Vocabulary saved correctly!"), response.getOutputStream());
	}

	@RequestMapping(value = "/ui/commitVocabularyInfo", method = RequestMethod.POST)
	public void commitVocabularyInfo(final HttpServletResponse response,
			final HttpServletRequest request,
			@RequestParam(value = "vocabularyId", required = true) final String vocabularyId) throws IOException, ISRegistryException, JAXBException {
		log.info("committing vocabulary info id = " + vocabularyId);

		Vocabulary voc = new Gson().fromJson(request.getReader(), Vocabulary.class);

		// String xml = "<VOCABULARY_NAME code='{code}'>{name}</VOCABULARY_NAME>";
		// xml = xml.replace("{code}", StringEscapeUtils.escapeXml(voc.getCode())).replace("{name}",
		// StringEscapeUtils.escapeXml(voc.getName()));
		// registryLocator.getService().updateProfileNode(vocabularyId, "//VOCABULARY_NAME", xml);

		String xml = "<VOCABULARY_DESCRIPTION>{desc}</VOCABULARY_DESCRIPTION>";
		xml = xml.replace("{desc}", StringEscapeUtils.escapeXml(voc.getDescription()));
		registryLocator.getService().updateProfileNode(vocabularyId, "//VOCABULARY_DESCRIPTION", xml);

		IOUtils.copy(new StringReader("Vocabulary info saved correctly!"), response.getOutputStream());
	}

	@RequestMapping(value = "/ui/createVocabulary", method = RequestMethod.POST)
	public void createVocabulary(final HttpServletResponse response, final HttpServletRequest request) throws IOException, ISRegistryException, JAXBException {
		log.info("create vocabulary");

		Vocabulary voc = new Gson().fromJson(request.getReader(), Vocabulary.class);

		StringTemplate st = new StringTemplate(
				IOUtils.toString(getClass().getResourceAsStream("/eu/dnetlib/functionality/modular/templates/vocabulary.xml.st")));
		st.setAttribute("name", voc.getName());
		st.setAttribute("description", voc.getDescription());
		st.setAttribute("code", voc.getCode());
		st.setAttribute("date", DateUtils.now_ISO8601());
		String newVocabularyId = registryLocator.getService().registerProfile(st.toString());

		IOUtils.copy(new StringReader(newVocabularyId), response.getOutputStream());
	}

	@RequestMapping(value = "/ui/dropVocabulary", method = RequestMethod.GET)
	public void dropVocabulary(final HttpServletResponse response,
			final HttpServletRequest request,
			@RequestParam(value = "vocabularyId", required = true) final String vocabularyId) throws IOException, ISRegistryException, JAXBException {
		log.info("delete vocabulary id=" + vocabularyId);

		registryLocator.getService().deleteProfile(vocabularyId);
	}
}
