/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package eu.dnetlib.espas.spatial.impl;

import eu.dnetlib.espas.spatial.QShape;
import eu.dnetlib.espas.spatial.QueryCRS;
import eu.dnetlib.espas.spatial.TimePeriodConstraint;
import static eu.dnetlib.espas.spatial.impl.ObservationFilter._logger;
import eu.dnetlib.espas.spatial.utils.QueryDBUtils;
import eu.dnetlib.espas.spatial.utils.SQDiskMonitor;
import eu.dnetlib.espas.spatial.utils.SatDataFileUtils;
import eu.dnetlib.espas.util.GeometryMetadataHandler;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.logging.Level;
import javax.xml.bind.JAXBException;
   import org.apache.log4j.Logger;
import org.joda.time.Period;
import org.xml.sax.SAXException;

/**
 *
 * @author gathanas
 */
public class SatelliteObservationFilter extends ObservationFilter{
   private static final Logger _logger = Logger.getLogger(SatelliteObservationFilter.class);
   static final int BATCH_LIST_QUOTA= 20000;
   private SQDiskMonitor diskMonitor;
   private SatDataFileUtils dataUtils;
   private Map<String,String> satNameToTLEFileMap = new HashMap<String, String>();
   private List<SatelliteProcessor> satelliteWorkers;
   private boolean cancel = false;
   
   public SatelliteObservationFilter(QueryDBUtils dBUtils, TimePeriodConstraint timeConstraint, String userId, QueryCRS crs, String uniqQueryId,
                                     TimePeriodCalculator timeCalculator, QShape spaceConstraint) {
      super(dBUtils, timeConstraint, userId, crs, uniqQueryId, timeCalculator, spaceConstraint);
      dataUtils = new SatDataFileUtils();
   }

   @Override
   public void cancel(){
       cancel=true;
       if(satelliteWorkers!=null)
           for(SatelliteProcessor proc:satelliteWorkers)
               proc.cancelExecution();
   }
   
   @Override
   public void execute() {
       try {
            List<Object[]> observationSatList = dBUtils.getSatelliteObservations(timeConstraint);
            List<String[]> uniqueSats = dBUtils.getUniqueSatellitesinTimeRange(timeConstraint);
           findSatelliteLocations(uniqueSats);
           fuseObservationLocations(observationSatList,uniqueSats);           
       } catch (IOException ex) {
           _logger.error(null, ex);
       }
      
   }

   @Override
   protected void performLocationQuery() {
   }

   
   public void setDiskMonitor(SQDiskMonitor diskMonitor) throws FileNotFoundException {
      this.diskMonitor = diskMonitor;
      satNameToTLEFileMap=dataUtils.getSatNameLocationFileMap(new File(diskMonitor.getSatelliteListFilePath()));
   }

   private void findSatelliteLocations(List<String[]> uniqueSats) throws IOException {
       File epochFile = diskMonitor.createReqSatelliteEpochFile(this.uniqQueryId); 
       
       dataUtils.generateEpochFile(epochFile, timeCalculator, timeConstraint);
       
       for(String[] satDetails: uniqueSats){
           try {
               File satelliteOutputFile = diskMonitor.createReqSatelliteOutputFile(this.uniqQueryId, satDetails[0]);
               File commandAttributes = prepareSatelliteInputNameList(epochFile,satelliteOutputFile,satDetails[0]);
               executeSatTrajectoryProcess(commandAttributes);
           }
           catch (InterruptedException ex) {
               _logger.error("Execution of trajectory process for "+satDetails[0]+" has failed.", ex);
               continue;
           }
       }
       
   }

  private File prepareSatelliteInputNameList(File epochFile, File satelliteOutputFile, String satelliteName) throws IOException{
      File processInputFile = diskMonitor.createTrajectoryProcInputFile(this.uniqQueryId, satelliteName);
      FileWriter fwritter = new FileWriter(processInputFile);
      fwritter.write(" &orbit\n");
      fwritter.append("  ELEMENTFILE = '"+getSatElementFilePath(satelliteName)+"'\n");
      fwritter.append("  OUTPUTFILE = '"+satelliteOutputFile.getPath()+"'\n");
      fwritter.append("  EPOCHFILE = '"+epochFile.getPath()+"'\n");
      fwritter.append("  REFERENCEFRAME = 'GEOspherical'\n");
      fwritter.append("  KERNELPATH = '"+diskMonitor.getSatelliteUtilityProcessKernelPath()+"'\n /\n");
      fwritter.flush();
      fwritter.close();
      return  processInputFile;
  }
  
  private int executeSatTrajectoryProcess(File inputFile) throws IOException, InterruptedException{
      String trajectoryCommand = diskMonitor.getSatelliteUtilityProcessPath();
      String trajectoryAttribute = inputFile.getAbsolutePath().replaceFirst("file://", "").trim();
      ProcessBuilder systemProcBuilder = new ProcessBuilder(trajectoryCommand,trajectoryAttribute);
      Process trajectoryProcess = systemProcBuilder.start();
      int terminationCode = trajectoryProcess.waitFor();
      return terminationCode;
  }
  
  private String getSatElementFilePath(String espasSatName){
     return diskMonitor.getSatelliteTLEPath()+File.separator+this.satNameToTLEFileMap.get(espasSatName);
  }

   private void fuseObservationLocations(List<Object[]> observationSatList,List<String[]> uniqueSats) {
       List<Date[]> timePeriods = timeCalculator.findPeriodicTimestamps(timeConstraint.getFromDate(), timeConstraint.getToDate(), timeCalculator.estimatePeriod(timeConstraint));
       List<String> uniqueSatNames = getUniqueSatNames(uniqueSats);
       Executor workerExecutor = Executors.newFixedThreadPool(uniqueSats.size());
       
       satelliteWorkers = new LinkedList<SatelliteProcessor>();
       CountDownLatch satLatch = new CountDownLatch(uniqueSatNames.size());
       for(String satellite:uniqueSatNames){
           SatelliteProcessor worker = new SatelliteProcessor(satLatch,this.dataUtils,this.diskMonitor,this.dBUtils,observationSatList,satellite,this.uniqQueryId,timePeriods, crs.espasValue());
           satelliteWorkers.add(worker);
           workerExecutor.execute(worker);
       }

       try {
           satLatch.await();
       } catch (InterruptedException ex) {
           _logger.error("Failed while waiting for spawned threads to complete", ex);
       }
               
   }

   private List<String> getUniqueSatNames(List<String[]> uniqueSats){
      List<String> results = new LinkedList<String>();
      for(String[] record:uniqueSats)
         results.add(record[0]);
      return results;
   }
}


class SatelliteHandler implements Runnable{
    
    private File commandNamedListFile;
    private CountDownLatch syncronizationLatch;
    
    public SatelliteHandler(File outputFile,CountDownLatch latch) {
        this.commandNamedListFile = outputFile;
        this.syncronizationLatch = latch;
    }
    
    
    @Override
    public void run() {
    }
    
}


class SatelliteProcessor implements Runnable{
    private List<Object[]> observationDetails;
    private String satelliteId;
    private String queryId;
    private QueryDBUtils dBUtils;
    private SQDiskMonitor diskMonitor;
    private List<Date[]> timePeriods;
    private SatDataFileUtils dataUtils;
    private String crs;
    private CountDownLatch _latch;
    
    private boolean cancel = false;

    public SatelliteProcessor(CountDownLatch latch,SatDataFileUtils dataUtils, SQDiskMonitor diskMonitor,QueryDBUtils dBUtils, List<Object[]> observationDetails, String satelliteId, String queryId,List<Date[]> timePeriods,String crs) {
        this.observationDetails = observationDetails;
        this.satelliteId = satelliteId;
        this.queryId = queryId;
        this.dBUtils = dBUtils;
        this.diskMonitor = diskMonitor;
        this.timePeriods= timePeriods;
        this.dataUtils = dataUtils;
        this.crs = crs;
        this._latch = latch;
    }
    
    public void cancelExecution(){
        this.cancel = true;
    }
    
    @Override
    public void run() {
        try {
            SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ssz");         
            String type = SQueryObservationType.Satellite.name();
            
            File satLocationFile = diskMonitor.createReqSatelliteOutputFile(this.queryId, satelliteId);

            File tempF = this.dBUtils.createTempDataFile(queryId);
            FileWriter tempFWriter = new FileWriter(tempF);

            Map<Date, String> observationLocations = this.dataUtils.getSatLocationXMLRecords(satLocationFile);
            for (Date[] timePeriod : timePeriods) 
            if(!cancel){
                String locationXML = null;
                try {
                synchronized(diskMonitor){
                    locationXML = GeometryMetadataHandler.transformGeoLocation(timePeriod[1], (String) observationLocations.get(timePeriod[1]), crs);
                    }
                } catch (IOException ex) {
                    _logger.error(null, ex);
                } catch (JAXBException ex) {
                    _logger.error(null, ex);
                } catch (ParseException ex) {
                    _logger.error(null, ex);
                } catch (SAXException ex) {
                    _logger.error(null, ex);
                }
                observationLocations.put(timePeriod[1], locationXML);
            }
            else{
                this._latch.countDown();
                return;
            }
            
            int _counter = 0;
            for (Object[] obsDetails : observationDetails) {
                if (((String) obsDetails[4]).equals(satelliteId)) {
                    try {
                        for (Date[] timePeriod : timePeriods) 
                            if(cancel){
                                this._latch.countDown();
                                return;
                            }
                            else {
                            _counter++;

                            String locationXML = observationLocations.get(timePeriod[1]);
                            Object[] record = new Object[]{this.queryId, obsDetails[0], timePeriod[0], timePeriod[2], locationXML};
                            
                            String insertedRecord = "('"+(String)record[0]+"';'"+(String)record[1]+"';'"+type+"'"
                                    + ";to_timestamp('"+dateFormat.format((Date)record[2])+"','YYYY-MM-DD HH24:MI:SS')"
                                    + ";to_timestamp('"+dateFormat.format((Date)record[3])+"','YYYY-MM-DD HH24:MI:SS')"
                                    + ";'"+locationXML+"')";
                            _logger.trace("About to check record :"+insertedRecord);

                        if (locationXML != null && !locationXML.isEmpty()) {
                                prepareMassImportFile(record,tempFWriter);
                            }


//                    added to support processing in batch mode in order to decrease memory footprint
                            
                            
                                if(_counter>=SatelliteObservationFilter.BATCH_LIST_QUOTA){
                                tempFWriter.flush();
                                tempFWriter.close();
                                    
                                dBUtils.registerFilteredObservations(this.queryId, SQueryObservationType.Satellite.name(), tempF);
                                
                                tempF = this.dBUtils.createTempDataFile(queryId);
                                tempFWriter = new FileWriter(tempF);
                                
                                _counter=0;
                            }
                                
                        }
                    } catch (IOException ex) {
                        _logger.error(null, ex);
                        continue;
                    }
                }
            }
        tempFWriter.flush();
        tempFWriter.close();
         
        if(!cancel)
            dBUtils.registerFilteredObservations(this.queryId, SQueryObservationType.Satellite.name(), tempF);
        
        } catch (IOException ex) {
            _logger.error("failed to retrieve sattellite data file", ex);
        } finally {
            _latch.countDown();
        }
    }
    
    private void prepareMassImportFile(Object[] recordRow, Writer tempFWriter) throws IOException {
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ssz");
        String type = SQueryObservationType.Satellite.name();
        if (recordRow[4] != null && !((String) recordRow[4]).isEmpty()) {
            tempFWriter.append("" + ((String) recordRow[0]).trim() + ";" + ((String) recordRow[1]).trim() + ";" + type + ""
                    + ";" + (new Timestamp(((Date) recordRow[2]).getTime())).toString().trim()
                    + ";" + (new Timestamp(((Date) recordRow[3]).getTime())).toString().trim()
                    + ";" + ((String) recordRow[4]).replaceAll("\\s", " ").trim() + "\n");

        } else {
            _logger.trace("Record :" + (String) recordRow[1] + ", " + dateFormat.format((Date) recordRow[2]) + " has  null  or empty geometry");
        }

    }
    
}