package eu.dnetlib.api.data;

import java.util.List;

import eu.dnetlib.api.DriverService;
import eu.dnetlib.domain.data.Repository;
import eu.dnetlib.domain.data.RepositoryInterface;

public interface DatasourceManagerService extends DriverService {

	/**
	 * Register a datasource.
	 * 
	 * @param ds
	 *            the new datasource
	 * @throws DatasourceManagerServiceException
	 */
	public boolean addDatasource(Repository repository)
			throws DatasourceManagerServiceException;

	/**
	 * Delete a datasource.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @throws DatasourceManagerServiceException
	 */
	public boolean deleteDatasource(String dsId)
			throws DatasourceManagerServiceException;

	/**
	 * Returns a datasource.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @throws DatasourceManagerServiceException
	 */
	public Repository getDatasource(String dsId)
			throws DatasourceManagerServiceException;

	/**
	 * Returns all the datasources.
	 * 
	 * @return a list of XML profiles
	 * @throws DatasourceManagerServiceException
	 */
	public List<Repository> listAllDatasources()
			throws DatasourceManagerServiceException;

	/**
	 * Returns the datasources with some properties.
	 * 
	 * @param compliance
	 *            the level of compliance (or NULL to skip the test)
	 * @param fileManagementMode
	 *            the fileManagementMode property (or NULL to skip the test)
	 * @param contentDescription
	 *            the contentDescription property (or NULL to skip the test)
	 * @param iisProcessingWorkflow
	 *            the iisProcessingWorkflow property (or NULL to skip the test)
	 * @return a list of XML profiles
	 * @throws DatasourceManagerServiceException
	 */
	public List<Repository> listDatasourcesUsingFilter(String compliance,
			String contentDescription, String iisProcessingWorkflow, String collectedFrom)
			throws DatasourceManagerServiceException;

	/**
	 * updates the level of compliance of a datasource.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param ifaceId
	 *            the interface id
	 * @param level
	 *            the new level of compliance
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean updateLevelOfCompliance(String dsId, String ifaceId,
			String level) throws DatasourceManagerServiceException;

	/**
	 * update the baseUrl of an interface.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param ifaceId
	 *            the interface id
	 * @param baseUrl
	 *            the new value
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean updateBaseUrl(String dsId, String ifaceId, String baseUrl)
			throws DatasourceManagerServiceException;

	/**
	 * update the activation status of an interface (true = there is a related
	 * wf).
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param ifaceId
	 *            the interface id
	 * @param active
	 *            true if there is a related wf
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean updateActivationStatus(String dsId, String ifaceId,
			boolean active) throws DatasourceManagerServiceException;

	/**
	 * update the "contentDescription" property of an interface.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param ifaceId
	 *            the interface id
	 * @param desc
	 *            the new value
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean updateContentDescription(String dsId, String ifaceId,
			String desc) throws DatasourceManagerServiceException;

	/**
	 * update the "iis_processing_workflow" property of an interface.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param ifaceId
	 *            the interface id
	 * @param wf
	 *            the new value
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean setIisProcessingWorkflow(String dsId, String ifaceId,
			String wf) throws DatasourceManagerServiceException;

	/**
	 * Set a generic extra field of an interface.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param ifaceId
	 *            the interface id
	 * @param field
	 *            the field name
	 * @param value
	 *            the new value (or null for removing it)
	 * @param preserveOriginal
	 *            preserve the original value in the DB (the value is stored in
	 *            the "edited" column)
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean updateExtraField(String dsId, String ifaceId, String field,
			String value, boolean preserveOriginal)
			throws DatasourceManagerServiceException;

	/**
	 * Update an access param of an interface.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param ifaceId
	 *            the interface id
	 * @param field
	 *            the field name
	 * @param value
	 *            the new value (or null for removing it)
	 * @param preserveOriginal
	 *            preserve the original value in the DB (the value is stored in
	 *            the "edited" column)
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean updateAccessParam(String dsId, String ifaceId, String field,
			String value, boolean preserveOriginal)
			throws DatasourceManagerServiceException;

	/**
	 * delete a generic extra field or an access param of an interface.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param ifaceId
	 *            the interface id
	 * @param field
	 *            the field name
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean deleteAccessParamOrExtraField(String dsId, String ifaceId,
			String field) throws DatasourceManagerServiceException;

	/**
	 * Add an interface to a datasource.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param iface
	 *            the interface description
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean addInterface(String dsId, RepositoryInterface iface)
			throws DatasourceManagerServiceException;

	/**
	 * Delete the interface of a datasource.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param ifcId
	 *            the id of the interface
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean deleteInterface(String dsId, String ifaceId)
			throws DatasourceManagerServiceException;

	/**
	 * Perform a generic SQL update on a datasource.
	 * 
	 * @param dsId
	 *            the OpenAireDataSourceId (example: opendoar____::2101)
	 * @param sql
	 *            the sql query to insert/update/delete the datasource
	 * @param delete
	 *            true if the profile must be deleted
	 * @return true if the operation ends successfully
	 * @throws DatasourceManagerServiceException
	 */
	public boolean updateSQL(String dsId, String sql, boolean delete)
			throws DatasourceManagerServiceException;
}
