package eu.dnetlib.api.functionality;

import java.util.List;

import eu.dnetlib.api.DriverService;
import eu.dnetlib.domain.functionality.Community;
import eu.dnetlib.domain.functionality.CommunitySearchCriteria;

/**
 * The class that manages Community objects
 * 
 */
public interface CommunityService extends DriverService {
	/**
	 * Create or edit a community.
	 * @param community the community to save 
	 * @return the community just saved
	 * @throws CommunityServiceException if any errors occur
	 */
	public Community saveCommunity(Community community) throws CommunityServiceException;

	/**
	 * Delete a community.
	 * @param community the community to delete
	 * @throws CommunityServiceException if any errors occur
	 */
	public void deleteCommunity(Community community) throws CommunityServiceException;
	
	/**
	 * Delete a community.
	 * @param id the id of the community to delete
	 * @throws CommunityServiceException if any errors occur
	 */
	public void deleteCommunityById(String id) throws CommunityServiceException;
	
	/**
	 * Search for a community by id
	 * @param id the id of the community to search for
	 * @return the community with the specified id or null if no such community exists
	 * @throws CommunityServiceException
	 */
	public Community getCommunityById(String id) throws CommunityServiceException;

	/**
	 * Search for communities by the specified criteria.
	 * @param criteria the search criteria to match
	 * @return a list containing all the communities that match the specified criteria
	 * @throws CommunityServiceException if any errors occur
	 */
	public List<Community> searchCommunities(CommunitySearchCriteria criteria)
			throws CommunityServiceException;

	/**
	 * Search for communities by the specified criteria.
	 * @param criteria the search criteria to match
	 * @return a list containing the ids of all the communities that match the specified criteria
	 * @throws CommunityServiceException if any errors occur
	 */
	public List<String> searchCommunityIds(CommunitySearchCriteria criteria)
			throws CommunityServiceException;
	
	/**
	 * Check if a user is owner of a community.
	 * @param communityId the id of the community to check for
	 * @param userId the id of the user to check for
	 * @return true if the user is owner of the community; false otherwise
	 * @throws CommunityServiceException if any errors occur
	 */
	public boolean isOwner(String communityId, String userId) throws CommunityServiceException;
	
	/**
	 * Check if a user is manager of a community.
	 * @param communityId the id of the community to check for
	 * @param userId the id of the user to check for
	 * @return true if the user is manager of the community; false otherwise
	 * @throws CommunityServiceException
	 */
	public boolean isManager(String communityId, String userId) throws CommunityServiceException;
}
