package eu.dnetlib.api.data;

import java.util.Collection;

import eu.dnetlib.api.DriverService;
import eu.dnetlib.domain.data.FormattedSearchResult;
import eu.dnetlib.domain.data.SearchResult;
import eu.dnetlib.domain.data.SuggestiveResult;

public interface SearchService extends DriverService {

	/**
	 * Returns search results for given query containing the request parameters and 
	 * the list of search result XML records as it is derived after the application of 
	 * the xslt tranfrormer.
	 * 
	 * @param queryText the query
	 * @param transformer the name of the xslt transformer for search results 
	 * @param locale the locale
	 * @param page the search result page 
	 * @param size the search result page size
	 * @return {@link SearchResult} containing the request parameters and the list of 
	 * search result xml records
	 * 
	 * @throws SearchServiceException
	 */
	public SearchResult search(String queryText, String transformer,
			String locale, int page, int size) throws SearchServiceException;

	/**
	 * Returns refine results of given query containing the request parameters and 
	 * the list of refine XML records as it is derived after the application of 
	 * the xslt transfrormer.
	 * @param queryText the query
	 * @param transformer the name of the xslt transformer for refine results
	 * @param format the format
	 * @param locale the locale
	 * @param fields the fields to include in refine results
	 * @return {@link SearchResult} containing the request parameters and the list of refine xml records
	 * 
	 * @throws SearchServiceException 
	 */
	public SearchResult refine(String queryText, String transformer,
			String locale, Collection<String> fields) throws SearchServiceException;

	/**
	 * Returns search and refine results  containing the request parameters, the list 
	 * of XML search records and the list of XML refine records as it is derived after 
	 * the application of the xslt tranfrormers.
	 * @param queryText the query 
	 * @param searchTransformer the xslt transformer of search results
	 * @param refineTransformer the xslt transformer of refine results
	 * @param format the format
	 * @param locale the locale
	 * @param page the search result page
	 * @param size the search result page size
	 * @param fields the fields to include in refine
	 * @return {@link SearchResult} containing the request parameters, the list 
	 * of search records and the list of refine records 
	 * 
	 * @throws SearchServiceException
	 */
	public SearchResult searchNrefine(String queryText, String searchTransformer, String refineTransformer, 
			String locale, int page, int size, Collection<String> fields) throws SearchServiceException;

	/**
	 * Returns a {@link FormattedSearchResult} containing the formatted results of search 
	 * @param queryText the query
	 * @param transformer the name of the transformer for refine results
	 * @param format the format
	 * @param locale the locale
	 * @param fields the fields to include in refine results
	 * @return {@link FormattedSearchResult} containing the formatted results of refine 
	 * 
	 * @throws SearchServiceException	 */
	public FormattedSearchResult search(String queryText, String transformer, String format,
			String locale, int page, int size) throws SearchServiceException;
	
	/**
	 * Returns a  {@link FormattedSearchResult} containing the formatted results of refine 
	 * @param queryText the query 
	 * @param searchTransformer the transformer of search results
	 * @param refineTransformer the transformer of refine results
	 * @param format the format
	 * @param locale the locale
	 * @param page the search result page
	 * @param size the search result page size
	 * @param fields the fields to include in refine
	 * @return  {@link FormattedSearchResult} containing the formatted results of refine and search  
	 * @throws SearchServiceException
	 */
	public FormattedSearchResult refine(String queryText, String refineTransformer, 
			String format, String locale, Collection<String> fields)
				throws SearchServiceException;
	
	/**
	 * Returns a  {@link FormattedSearchResult} containing the formatted results of refine and search 
	 * @param queryText the query 
	 * @param searchTransformer the transformer of search results
	 * @param refineTransformer the transformer of refine results
	 * @param format the format
	 * @param locale the locale
	 * @param page the search result page
	 * @param size the search result page size
	 * @param fields the fields to include in refine
	 * @return  {@link FormattedSearchResult} containing the formatted results of refine and search  */
	public FormattedSearchResult searchNrefine(String queryText, String searchTransformer, String refineTransformer, 
			String format, String locale, int page, int size, Collection<String> fields) throws SearchServiceException;
	
	public SuggestiveResult suggestiveSearch(String query) throws SearchServiceException;

}