package eu.dnetlib.api.data.espas;

import eu.dnetlib.api.DriverService;
import eu.dnetlib.domain.data.espas.HarvestHistory;
import eu.dnetlib.domain.data.espas.HarvestSchedule;

import java.util.Date;
import java.util.List;

/**
 * Created by antleb on 6/17/14.
 */
public interface DataProviderService extends DriverService {

    /**
     * Starts a harvesting process
     *
     * @param types          the list of ESPAS resource types to harvest
     * @param dateFrom       the earliest modification date of the resources to harvest (inclusive)
     * @param dateTo         the latest modification date of the resources to harvest (inclusive)
     * @param dataProviderId the dataProvider identifier
     */
    public void harvest(List<String> types, Date dateFrom, Date dateTo, String dataProviderId) throws DataProviderServiceException;

    /**
     * Schedules a periodic harvesting process. Each data provider may have more than one scheduled processes.
     *
     * @param types          the list of ESPAS resource types to harvest
     * @param initialModificationDate       the earliest modification date of the resources to harvest when the harvest process first runs.
     * @param dataProviderId the dataProvider identifier
     * @param cronExpression the cron expression describing the scheduling of the harvesting process
     * @return the assigned id of the new schedule
     */
    public String scheduleHarvest(List<String> types, Date initialModificationDate, String dataProviderId,
                                  String cronExpression) throws DataProviderServiceException;

    /**
     * Updates an existing schedule of a harvesting process
     *
     * @param scheduleId     the assigned id of the existing schedule
     * @param types          the list of ESPAS resource types to harvest
     * @param dataProviderId the dataProvider identifier
     * @param cronExpression the cron expression describing the scheduling of the harvesting process
     */
    public void updateHarvestSchedule(String scheduleId, List<String> types, String dataProviderId, String cronExpression) throws DataProviderServiceException;

	/**
	 * Pauses the harvesting process without deleting it from the service.
	 *
	 * @param scheduleId The id of the process to pause.
	 */
	public void pauseHarvestSchedule(String scheduleId) throws DataProviderServiceException;

	/**
	 * Resumes a previously paused harvesting process.
	 *
	 * @param scheduleId The id of the process to resume.
	 */
	public void resumeHarvestScedule(String scheduleId) throws DataProviderServiceException;

	/**
	 * Cancels the scheduling and deletes a harvesting process.
	 *
	 * @param scheduleId the assigned id of the existing schedule
	 */
	public void cancelHarvestSchedule(String scheduleId) throws DataProviderServiceException;

    /**
     * Returns all the scheduled harvesting process for a data provider
     *
     * @param dataProviderId the dataProvider identifier
     * @return all the scheduled harvesting process for a data provider
     */
    public List<HarvestSchedule> getHarvestSchedules(String dataProviderId) throws DataProviderServiceException;

    /**
     * Returns the history of all harvesting process that have been executed for a data provider.
     *
     * @param dataProviderId the dataProvider identifier
     * @return the history of all harvesting process that have been executed for a data provider.
     */
    public List<HarvestHistory> getHarvestHistory(String dataProviderId) throws DataProviderServiceException;

    /**
     * Removes the history of all harvesting process that have been executed for a data provider for a given period.
     *
     * @param dataProviderId the dataProvider identifier
     * @param from           the earliest harvest date to delete (inclusive)
     * @param to             the latest harvest date to delete (inclusive)
     */
    public void deleteHarvestHistory(String dataProviderId, Date from, Date to) throws DataProviderServiceException;

    /**
     * Removes the history of harvesting processes that have been executed.
     *
     * @param harvestIds The ids of the harvesting process history objects to delete
     */
    public void deleteHarvestHistory(List<String> harvestIds) throws DataProviderServiceException;
}