package eu.dnetlib.api.functionality;

import java.util.Date;
import java.util.List;
import java.util.Set;

import eu.dnetlib.api.DriverService;
import eu.dnetlib.domain.functionality.Recommendation;

/**
 * The class that manages Recommendation objects
 */
public interface RecommendationService extends DriverService {

	/**
	 * creates a new recommendation of type announcement
	 * 
	 * @param recommendationText
	 *            the text of the recommendation
	 * @param creationDate
	 *            the creation date of the announcement
	 * @param expirationDate
	 *            the date the announcement expires
	 * @return the id of the new recommendation
	 * @throws RecommendationWebServiceException
	 *             if the announcement cannot be created
	 */
	public String generateAnnouncement(int index, boolean active,
			String announcementTitle, String announcementText,
			Date creationDate, Date expirationDate)
			throws RecommendationServiceException;

	/**
	 * Swap the index of 2 announcements
	 * 
	 * @param announcementId1
	 *            the id of the fist announcement
	 * @param announcementId2
	 *            the id of the second announcement
	 */
	public void swapAnnouncements(String announcementId1, String announcementId2)
			throws RecommendationServiceException;

	/**
	 * Swap the index of 2 user recommendaitons
	 * 
	 * @param announcementId1
	 *            the id of the fist announcement
	 * @param announcementId2
	 *            the id of the second announcement
	 */
	public void swapUserRecommendations(String announcementId1, String announcementId2) 
			throws RecommendationServiceException;;
		
	/**
	 * Swap the index of 2 community recommendations
	 * 
	 * @param announcementId1
	 *            the id of the fist announcement
	 * @param announcementId2
	 *            the id of the second announcement
	 */
	public void swapCommunityRecommendations(String announcementId1, String announcementId2)
			throws RecommendationServiceException;
		
	/**
	 * creates a new recommendation
	 * 
	 * @param index
	 *            the recommendation index
	 * @param active
	 *            the state of the recommendaiton, is it active or not #param
	 *            title the title of the recommendation
	 * @param rText
	 *            the recommendation text
	 * @param creationDate
	 *            the date the recommendation was created
	 * @param expirationDate
	 *            the date the recommendation expires
	 * @return the id of the recommendation
	 * @throws RecommendationServiceException
	 *             if the recommendation cannot be created
	 */
	public Recommendation generateRecommendation(int index, boolean active,
			String title, String recommendationText, Date creationDate,
			Date expirationDate) throws RecommendationServiceException;

	/**
	 * creates a new recommendation of type community
	 * 
	 * @param recommendationText
	 *            the recommendation text
	 * @param creationDate
	 * @param expirationDate
	 * @param communityId
	 *            the ids of the associated communities
	 * @return the id of the recommendation
	 * @throws RecommendationServiceException
	 *             if the community recommendation cannot be created
	 */
	public String generateCommunityRecommendation(int index, boolean active,
			String title, String recommendationText, Date creationDate,
			Date expirationDate, Set<String> communityIds)
			throws RecommendationServiceException;

	/**
	 * creates a new recommendation of type user
	 * 
	 * @param recommendationText
	 * @param creationDate
	 * @param expirationDate
	 * @return the id of the recommendation
	 * @throws RecommendationServiceException
	 *             if the user recommendation cannot be created
	 */
	public String generateUserRecommendation(int index, boolean active,
			String title, String userId, String recommendationText,
			Date creationDate, Date expirationDate)
			throws RecommendationServiceException;

	/**
	 * retrieves the recommendation text of all announcements that haven't
	 * expired
	 * 
	 * @return the announcements' recommendation text
	 * @throws RecommendationServiceException
	 */
	public List<String> getAnnouncements()
			throws RecommendationServiceException;

	/**
	 * retrieves ths ids of all announcements
	 * 
	 * @throws RecommendationServiceException
	 */
	public List<String> getAllAnnouncementIds()
			throws RecommendationServiceException;

	/**
	 * retrieves the recommendation text of all announcements
	 * 
	 * @return the recommendation text of all announcements
	 * @throws RecommendationServiceException
	 */
	public List<Recommendation> getAllAnnouncements()
			throws RecommendationServiceException;

	/**
	 * retrieves all recommendations of type community, only the content
	 * 
	 * @return the recommendation text of all community recommendations
	 * @throws RecommendationServiceException
	 */
	public List<String> getAllCommunityRecommendations()
			throws RecommendationServiceException;

	/**
	 * retrieves all recommendations of type community, the objects
	 * 
	 * @return the recommendation text of all community recommendations
	 * @throws RecommendationServiceException
	 */
	public List<Recommendation> getAllCommunityRecommendationsObj()
			throws RecommendationServiceException;

	public List<String> getCommunityRecommendations(String communityId)
			throws RecommendationServiceException;

	public List<Recommendation> getCommunityRecommendationsObj(
			String communityId) throws RecommendationServiceException;

	/**
	 * retrieves all community recommendation content that are associated with a
	 * community
	 * 
	 * @param communityId
	 *            the id of the community
	 * @return the recommendation text of the community recommendations
	 * @throws RecommendationServiceException
	 */
	public List<String> getAllCommunityRecommendations(String communityId)
			throws RecommendationServiceException;

	/**
	 * retrieves all community recommendation objects that are associated with a
	 * community.
	 * 
	 * @param communityId
	 *            the id of the community
	 * @return the recommendation text of the community recommendations
	 * @throws RecommendationServiceException
	 */
	public List<Recommendation> getAllCommunityRecommendationsObj(
			String communityId) throws RecommendationServiceException;

	public List<String> getAllUserRecommendations(String userId)
			throws RecommendationServiceException;

	/**
	 * retrieves a recommendation
	 * 
	 * @param recommendationId
	 *            the recommendation id
	 * @return the recommendation or null if the recommendation does not exist
	 * @throws RecommendationServiceException
	 * 
	 */

	public Recommendation getRecommendation(String recommendationId)
			throws RecommendationServiceException;

	/**
	 * retrieves the text of a recommendation
	 * 
	 * @param recommendationId
	 *            the id of the recommendation
	 * @return the recommendation text or null if the recommendation does not
	 *         exist
	 * @throws RecommendationServiceException
	 * 
	 */
	public String getRecommendationText(String recommendationId)
			throws RecommendationServiceException;

	/**
	 * retrieves the recommendation text of a Set of recommendations
	 * 
	 * @param recommendationIds
	 *            the ids of the recommendations
	 * @return a Set of the recommendations' text
	 * @throws RecommendationServiceException
	 * 
	 */
	public List<String> getRecommendations(List<String> recommendationIds)
			throws RecommendationServiceException;

	/**
	 * removes a recommendation
	 * 
	 * @param recommendationId
	 *            the id of the recommendation
	 * @throws RecommendationServiceException
	 *             if the recommendation does not exist or cannot be updated
	 */
	public void removeRecommendation(String recommendationId)
			throws RecommendationServiceException;

	/**
	 * removes a recommendation of type announcement
	 * 
	 * @param announcementId
	 *            the id of the announcement
	 * @throws RecommendationServiceException
	 *             if the announcement does not exist or cannot be updated
	 */
	public void removeAnnouncement(String announcementId)
			throws RecommendationServiceException;

	/**
	 * removes a recommendation of type community
	 * 
	 * @param recommendationId
	 *            the id of the recommendation
	 * @param communityId
	 *            the id of the community
	 * @throws RecommendationServiceException
	 */
	public void removeCommunityRecommendation(String recommendationId)
			throws RecommendationServiceException;

	public void removeUserRecommendation(String recommendationId)
			throws RecommendationServiceException;

	/**
	 * updates a recommendation
	 * 
	 * @param recommendationId
	 *            the id of the recommendation
	 * @param recommendationText
	 *            the text of the recommendation
	 * @param creationDate
	 *            the date the recommendation is created
	 * @param expirationDate
	 *            the date the recommendation expires
	 * @throws RecommendationServiceException
	 *             if the recommendation does not exist or cannot be updated
	 */
	public void updateRecommendation(String recommendationId, int index,
			boolean active, String title, String recommendationText,
			Date creationDate, Date expirationDate)
			throws RecommendationServiceException;

	/**
	 * updates an announcement and the corresponding recommendation
	 * 
	 * @param announcementId
	 *            the id of the announcement
	 * @param recommendationText
	 *            the text of the corresponding recommendation
	 * @param creationDate
	 *            the creation date of the announcement
	 * @param expirationDate
	 *            the expiration date of the announcement
	 * @throws RecommendationServiceException
	 *             if the announcement recommendation does not exist or cannot
	 *             be updated
	 */
	public void updateAnnouncement(String announcementId, int index,
			boolean active, String announcementTitle, String announcementText,
			Date creationDate, Date expirationDate)
			throws RecommendationServiceException;

	public void updateCommunityRecommendation(String recommendationId,
			int index, boolean active, String title, String recommendationText,
			Date creationDate, Date expirationDate, Set<String> communityIds)
			throws RecommendationServiceException;

	/*
	 * /** sends an e-mail to all users with the recommendation text of an
	 * announcement
	 * 
	 * @param announcementId the id of the announcement @throws
	 * RecommendationServiceException if the announcement or recommendation does
	 * not exist
	 * 
	 * public void sendAnnouncement( String announcementId ) throws
	 * RecommendationServiceException;
	 */

	public void addRecommendationToUser(String recommendationId, String userId)
			throws RecommendationServiceException;

	/**
	 * gets all the community recommendation for a specific user
	 * 
	 * @throws RecommendationServiceException
	 */
	public List<String> getCommunityRecommendationsForUser(String userId)
			throws RecommendationServiceException;

	/**
	 * gets all the community recommendation objects for a specific user
	 * 
	 * @throws RecommendationServiceException
	 */
	public List<Recommendation> getCommunityRecommendationsForUserObj(
			String userId) throws RecommendationServiceException;
}
