package eu.dnetlib.espas.data.harvest;

import java.io.StringReader;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Formatter;
import java.util.LinkedList;
import java.util.List;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.apache.log4j.Logger;

import eu.dnetlib.api.enabling.ResultSetService;
import eu.dnetlib.domain.EPR;
import eu.dnetlib.enabling.tools.blackboard.BlackboardJob;
import eu.dnetlib.espas.data.harvest.csw.CSWGetRecordsRequest;
import eu.dnetlib.espas.data.harvest.csw.common.CSWConstants;
import eu.dnetlib.espas.data.harvest.csw.common.CSWOutputFormatEnum;
import eu.dnetlib.espas.data.harvest.csw.common.CSWOutputSchemaStrs;
import eu.dnetlib.espas.data.harvest.csw.common.CSWResultSetTypeEnum;
import eu.dnetlib.espas.data.harvest.service.BlackboardHarvestParameters;
import eu.dnetlib.espas.data.harvest.service.CSWJobListener;
import eu.dnetlib.utils.EPRUtils;
import gr.uoa.di.driver.util.ServiceLocator;

public class RecordsConsumerTransImpl implements RecordsConsumer {
	{ // Class wide initialization.
		logger = Logger.getLogger(RecordsConsumerTransImpl.class);
	}

	public static int DEFAULT_BATCH_SIZE = 100;
	public static int DEFAULT_NUM_OF_BATCH_SIZES = 3;

	private static Logger logger = null;
	private int batchSize = DEFAULT_BATCH_SIZE;
	private int numOfBatchSizes = DEFAULT_NUM_OF_BATCH_SIZES;
	private CSWHarvester cswHarvester = null;
	CSWJobListener cswJobListener = null;
	private BlackboardJob job = null;
	private ServiceLocator<ResultSetService> resultSetServiceLocator = null;

	public RecordsConsumerTransImpl() {
	}

	@Override
	public void consume() {
		if(!this.isRecordsConsumerValid()) {
			logger.debug("The Record consumer is not properly configured!");
			return;
		}
		try {
			if (job.getAction().equals(BlackboardHarvestParameters.VALUE_ACTION_START_HARVESTING)) {
				logger.info("Start harvesting job.");
				String[] recordTypes = job.getParameters().get(BlackboardHarvestParameters.NAME_RECORD_TYPE).split(",");
				if(0 == recordTypes.length) {
					throw new Exception("Passed list of record types is empty.");
				}
				LinkedList<CSWGetRecordsRequest> cswGetRecordsRequests = new LinkedList<CSWGetRecordsRequest>();
			    for(int i = 0; i < recordTypes.length; i++) {
			    	CSWGetRecordsRequest cswGetRecordsRequest = new CSWGetRecordsRequest();
					cswGetRecordsRequest.setBaseURL(job.getParameters().get(BlackboardHarvestParameters.NAME_URL));
					cswGetRecordsRequest.setService(CSWConstants.CSW_Service_NAME_STR);
					cswGetRecordsRequest.setVersion(CSWConstants.VERSION_202);
					cswGetRecordsRequest.setOutputFormat(CSWOutputFormatEnum.CSW_OUTPUT_FORMAT_APPLICATION_XML);
					cswGetRecordsRequest.setOutputSchema(CSWOutputSchemaStrs.CSW_OUTPUT_SCHEMA_CORE_DEFAULT);
					cswGetRecordsRequest.setResultType(CSWResultSetTypeEnum.RESULTS);
					cswGetRecordsRequest.setToDate(job.getParameters().get(BlackboardHarvestParameters.NAME_TO_DATE));
					cswGetRecordsRequest.setFromDate(job.getParameters().get(BlackboardHarvestParameters.NAME_FROM_DATE));
			    	cswGetRecordsRequest.setType(recordTypes[i].trim());
					cswGetRecordsRequest.setStartPosition(1);
					cswGetRecordsRequest.setMaxRecords(this.batchSize);
					logger.debug("CSW Get Records Request\n" + cswGetRecordsRequest.toString());
					// Check whether the just created CSWGetRecordsRequest is valid, so as to be fed into a harvester.
					if(!cswGetRecordsRequest.isValid()) {
						logger.error("Invalid request");
						job.setAction(BlackboardHarvestParameters.VALUE_ACTION_IDLE);
						job.getParameters().put(BlackboardHarvestParameters.NAME_HARVESTING_STATUS, BlackboardHarvestParameters.VALUE_HARVESTING_STATUS_FAILED);
						cswJobListener.getBlackboardHandler().failed(job, new Exception("Invalid " + CSWGetRecordsRequest.class.getCanonicalName() + "request."));
						throw new Exception("Invalid CSW Get Records request.");
					}
				    cswGetRecordsRequests.addLast(cswGetRecordsRequest);
			    }
				logger.debug("Creating harvester.");
				RecordsHarvester recordsHarvester = new RecordsHarvesterTransImpl(this.numOfBatchSizes*this.batchSize, cswGetRecordsRequests);
				this.cswHarvester.execute(recordsHarvester);

				// Find a suitable service and keep the reference
				ResultSetService resultSetService = resultSetServiceLocator.getService();
				// Create a new result set
				EPR epr = resultSetService.createPushRS(86400, 0);
				// Read the result set id from the epr
				String resultSetID = epr.getParameter(BlackboardHarvestParameters.NAME_RESULT_RESOURCE_IDENTIFIER);

				logger.debug("CREATED RS: " + resultSetID);

				int numOfHarvestedRecords = 0;
				List<String> arrayList = new ArrayList<String>(0);

				for (String recordStr : (RecordsHarvesterTransImpl) recordsHarvester) {
					logger.debug(recordStr);
                                        if(recordStr!=null && !recordStr.isEmpty())
                                            arrayList.add(recordStr);

					if (arrayList.size() >= this.batchSize) {
						resultSetService.populateRS(resultSetID, arrayList);
						numOfHarvestedRecords += arrayList.size();
						logger.debug("Currently, batch harvested records [" + arrayList.size() + "]");
						logger.debug("Totally, harvested records [" + numOfHarvestedRecords + "]");
						arrayList.clear();
						job.getParameters().put(BlackboardHarvestParameters.NAME_NUM_OF_HARVESTED_RECORDS, Integer.toString(numOfHarvestedRecords));
						job.getParameters().put(BlackboardHarvestParameters.NAME_HARVESTING_STATUS, BlackboardHarvestParameters.VALUE_HARVESTING_STATUS_RUNNING);
						cswJobListener.getBlackboardHandler().ongoing(job);
					}
				}
				if(arrayList.size() > 0) {
					resultSetService.populateRS(resultSetID, arrayList);
					numOfHarvestedRecords += arrayList.size();

					logger.debug("Currently, batch harvested records [" + arrayList.size() + "]");
					logger.debug("Totally, harvested records [" + numOfHarvestedRecords + "]");
					arrayList.clear();
					job.getParameters().put(BlackboardHarvestParameters.NAME_NUM_OF_HARVESTED_RECORDS, Integer.toString(numOfHarvestedRecords));
					job.getParameters().put(BlackboardHarvestParameters.NAME_HARVESTING_STATUS, BlackboardHarvestParameters.VALUE_HARVESTING_STATUS_RUNNING);
					cswJobListener.getBlackboardHandler().ongoing(job);
				} else {
					logger.debug("Records list is empty.");
				}
				// Harvesting completed.
				resultSetService.closeRS(resultSetID);
				job.setAction(BlackboardHarvestParameters.VALUE_ACTION_IDLE);
				// Add the result set EPR (in xml format) so that the manager
				// service knows where to find the results.
				String eprStr = EPRUtils.eprToXml(epr);
				if(logger.isDebugEnabled()) {
					Formatter formatter = new Formatter();
					TransformerFactory transformerFactory = TransformerFactory.newInstance();
					transformerFactory.setAttribute("indent-number", 3);
					Transformer transformer = transformerFactory.newTransformer();
					transformer.setOutputProperty(OutputKeys.INDENT, "yes");
					Source xmlInput = new StreamSource(new StringReader(eprStr));
					StreamResult xmlOutput = new StreamResult(new StringWriter());
					transformer.transform(xmlInput, xmlOutput);
					formatter.format("\n%s\n", xmlOutput.getWriter().toString());
					logger.debug(formatter.toString());
					formatter.close();
				}
				logger.info("Harvesting completed.");
				logger.debug("EPR: " + this.eprToDNETUrl(eprStr));
				// No more records will be harvested, thus set the start position to zero.
				job.getParameters().put(BlackboardHarvestParameters.NAME_RESULT_SET_APR, this.eprToDNETUrl(eprStr));
				job.getParameters().put(BlackboardHarvestParameters.NAME_HARVESTING_STATUS, BlackboardHarvestParameters.VALUE_HARVESTING_STATUS_COMPLETED);
				job.getParameters().put(BlackboardHarvestParameters.NAME_CSW_SEARCH_STATUS_TIMESTAMP, recordsHarvester.getCSWGetRecordsRequest().getLast().getSearchStatus().get(Record.SS_ATT_TIMESTAMP));
				// Notify the blackboard that the current job listener
				// has finished its job successfully.
				logger.info("updating blackboard.");
				cswJobListener.getBlackboardHandler().done(job);

			}
		} catch (Exception e) {
			logger.error("Fatal exception, harvesting failed.", e);
			job.getParameters().put(BlackboardHarvestParameters.NAME_HARVESTING_STATUS, BlackboardHarvestParameters.VALUE_HARVESTING_STATUS_FAILED);
			cswJobListener.getBlackboardHandler().failed(job, e);
		}
	}

	public boolean isRecordsConsumerValid()
	{
		boolean isValid = true;
		logger.info("Configured batch size is [" + this.batchSize + "]");
		isValid = isValid && this.batchSize > 0; if(!isValid) return isValid;
		isValid = isValid && null != this.cswHarvester; if(!isValid) return isValid;
		isValid = isValid && null != this.cswJobListener;  if(!isValid) return isValid;
		isValid = isValid && null != this.job;  if(!isValid) return isValid;
		isValid = isValid && null != this.resultSetServiceLocator;  if(!isValid) return isValid;
		return isValid;
	}

	@Override
	public void run() {
		this.consume();
	}

	/**
	 * A simple method that encodes the result set epr to the dnet format:
	 * dnet://EPR/Base64Encoding(eprXml)
	 *
	 * @param eprXml
	 *
	 * @return
	 * @throws UnsupportedEncodingException
	 */
	private String eprToDNETUrl(String eprXml) throws UnsupportedEncodingException {
		eprXml = URLEncoder.encode(eprXml, "UTF-8").replace("+", "%20");

		return eprXml;
	}

	@Override
	public void setBatchSize(int batchSize) {
		this.batchSize = batchSize;
	}

	@Override
	public int getBatchSize() {
		return this.batchSize;
	}

	public int getNumOfBatchSizes()
   {
      return numOfBatchSizes;
   }

   public void setNumOfBatchSizes(int numOfBatchSizes)
   {
      this.numOfBatchSizes = numOfBatchSizes;
   }

   @Override
	public CSWHarvester getCswHarvester() {
		return cswHarvester;
	}

	@Override
	public void setCswHarvester(CSWHarvester cswHarvester) {
		this.cswHarvester = cswHarvester;
	}

	@Override
	public CSWJobListener getCswJobListener() {
		return cswJobListener;
	}

	@Override
	public void setCswJobListener(CSWJobListener cswJobListener) {
		this.cswJobListener = cswJobListener;
	}

	@Override
	public BlackboardJob getJob() {
		return job;
	}

	@Override
	public void setJob(BlackboardJob job) {
		this.job = job;
	}

	@Override
	public ServiceLocator<ResultSetService> getResultSetServiceLocator() {
		return resultSetServiceLocator;
	}

	@Override
	public void setResultSetServiceLocator(ServiceLocator<ResultSetService> resultSetServiceLocator) {
		this.resultSetServiceLocator = resultSetServiceLocator;
	}
}
