package eu.dnetlib.espas.data.harvest.csw.common;

/*
 * Some part of this file has been used form the deegree prroject,
 * "http://www.deegree.org"
 * Markus Schneider <schneider@lat-lon.de>
 */

import java.net.URI;
import java.net.URISyntaxException;

import javax.xml.namespace.QName;

/**
 *
 * Container for, in the specification defined, static specified elements.
 */
public final class CSWConstants
{
   /**
    * Namespace for elements from the CSW 2.0.2 specification <br>
    * Namespace="http://www.opengis.net/cat/csw/2.0.2"
    */
   public static final String CSW_202_NS = "http://www.opengis.net/cat/csw/2.0.2";

   /**
    * ISO application profile <br> "http://www.isotc211.org/2005/gmd"
    */
   public static final String ISO_19115_NS = "http://www.isotc211.org/2005/gmd";

   /**
    * DC application profile <br> "http://purl.org/dc/elements/1.1/"
    */
   public static final String DC_NS = "http://purl.org/dc/elements/1.1/";

   /**
    * DCT application profile <br> "http://purl.org/dc/terms/"
    */
   public static final String DCT_NS = "http://purl.org/dc/terms/";

   /**
    * LOCAL_PART = "dct"
    */
   public static final String DCT_PREFIX = "dct";

   /**
    * APISO application profile <br> "http://www.opengis.net/cat/csw/apiso/1.0"
    */
   public static final String APISO_NS = "http://www.opengis.net/cat/csw/apiso/1.0";

   /**
    * Namespace for elements from the ISO AP 1.0 specification <br>
    * Namespace="http://www.isotc211.org/2005/gmd"
    * */
   public static final String GMD_NS = "http://www.isotc211.org/2005/gmd";

   /**
    * Namespace for elements from the ISO AP 1.0 specification <br>
    * Namespace="http://www.isotc211.org/2005/srv"
    * */
   public static final String SRV_NS = "http://www.isotc211.org/2005/srv";

   /**
    * Location of the schema <br>
    * "http://schemas.opengis.net/csw/2.0.2/CSW-discovery.xsd"
    */
   public static final String CSW_202_DISCOVERY_SCHEMA = "http://schemas.opengis.net/csw/2.0.2/CSW-discovery.xsd";

   /**
    * Location of the schema <br>
    * "http://schemas.opengis.net/csw/2.0.2/CSW-publication.xsd"
    */
   public static final String CSW_202_PUBLICATION_SCHEMA = "http://schemas.opengis.net/csw/2.0.2/CSW-publication.xsd";

   /**
    * Location of the schema <br>
    * "http://schemas.opengis.net/csw/2.0.2/record.xsd"
    */
   public static final String CSW_202_RECORD = "http://schemas.opengis.net/csw/2.0.2/record.xsd";

   // Currently, the only supported service type is the "CSW".
   // If more than the above service need to be supported, then a corresponding
   // enumerarion is more appropriate.
   public static final String CSW_Service_NAME_STR = "CSW";

   // Common namespace prefix for elements from the CSW specification
   public static final String CSW_PREFIX = "csw";

   /**
    * Common namespace prefix for elements from the ISO AP specification for the
    * types "Dataset", "DatasetCollection" and "Application"
    */
   public static final String GMD_PREFIX = "gmd";

   /**
    * Common namespace prefix for elements from the ISO AP specification for the
    * types "Service"
    */
   public static final String SRV_PREFIX = "srv";

   /** Common namespace prefix for elements from the ISO AP specification */
   public static final String APISO_PREFIX = "apiso";

   /**
    * Common local part of a qualified name for elements from the CSW
    * specification <br>
    * LOCAL_PART = "Record"
    */
   public static final String DC_LOCAL_PART = "Record";

   /**
    * LOCAL_PART = "dc"
    */
   public static final String DC_PREFIX = "dc";

   /**
    * Common local part of a qualified name for elements from the ISO AP
    * specification <br>
    * LOCAL_PART = "MD_Metadata"
    */
   public static final String GMD_LOCAL_PART = "MD_Metadata";

   /**
    * Common local part of a qualified name for elements from the ISO AP
    * specification <br>
    * LOCAL_PART = "Service"
    */
   public static final String  SRV_LOCAL_PART = "Service";

   // CSW protocol version 2.0.2
   public static final Version VERSION_202 = Version.parseVersion("2.0.2");

   // CSW protocol version 2.0.2 as String representation
   public static final String VERSION_202_STRING = "2.0.2";

   // CSW ebrim protocol version 1.0.0
   public static final Version VERSION_100 = new Version(1, 0, 0);

   // CSW default application format
   public static final String CSW_DEFAULT_APPLICATION_FORMAT = "application/xml";

   // CSW CQL constraint language
   public static final String CSW_CQL_CONSTRAINT_LANGUAGE = "CQL_TEXT";

   /**
    *
    * Sections are informations about the service represented in the
    * GetCapabilities operation <br>
    * <li>ServiceIdentification</li> <li>ServiceProvider</li> <li>
    * OperationsMetadata</li> <li>Filter_Capabilities</li> <br>
    */
   public enum Sections
   {
      // Metadata about the CSW implementation
      ServiceIdentification,
      // Metadata about the organisation that provides the CSW implementation
      ServiceProvider,
      // Metadata about the operations provided by this CSW implementation
      OperationsMetadata,
      // Metadata about the filter capabilities that are implemented at this
      // server
      Filter_Capabilities
   }

   /**
    *
    * Specifies the elements that should be returned in the response <br>
    * <li>brief</li> <li>summary</li> <li>full</li> <br>
    *
    * @author <a href="mailto:thomas@lat-lon.de">Steffen Thomas</a>
    * @author last edited by: $Author: thomas $
    *
    * @version $Revision: $, $Date: $
    */
   public enum ReturnableElement
   {

      /**
       * Brief representation of a record. This is the shortest view of a record
       * by a specific profile.
       */
      brief,

      /**
       * Summary representation of a record. This view responses all the
       * elements that should be queryable by a record-profile.
       */
      summary,
      /**
       * Full representation of a record. In that response there are all the
       * elements represented that a record holds. Thus, there are elements
       * presented that are not queryable regarding to the CSW specification.
       */
      full;

      private ReturnableElement()
      {}

      public static ReturnableElement determineReturnableElement(String returnableElement)
      {
         ReturnableElement elementSetName = null;
         returnableElement = returnableElement.toLowerCase();
         if(returnableElement.equalsIgnoreCase(ReturnableElement.brief.name()))
         {
            elementSetName = ReturnableElement.brief;
         }
         else if(returnableElement.equalsIgnoreCase(ReturnableElement.summary.name()))
         {
            elementSetName = ReturnableElement.summary;
         }
         else if(returnableElement.equalsIgnoreCase(ReturnableElement.full.name()))
         {
            elementSetName = ReturnableElement.full;
         }
         else
         {
            elementSetName = ReturnableElement.summary;
         }
         return elementSetName;
      }

   }

   /**
    *
    * Specifies in which filter mode the query has to be processed. Either there
    * is a OGC XML filter encoding after the filterspecification document <a
    * href="http://www.opengeospatial.org/standards/filter">OGC 04-095</a> or
    * there is a common query language string (CqlText) which can be seen as an
    * explicit typed statement like an SQL statement.
    *
    * @author <a href="mailto:thomas@lat-lon.de">Steffen Thomas</a>
    * @author last edited by: $Author: thomas $
    *
    * @version $Revision: $, $Date: $
    */
   public enum ConstraintLanguage
   {

      // Common Queryable Language Text filtering
      CQLTEXT,
      // Filterexpression specified in OGC Spec document 04-095
      FILTER
   }

   /**
    *
    * Defined in the CSW-publication.xsd. Specifies the data manipulation
    * operations <br>
    * <li>insert</li> <li>delete</li> <li>update</li> <br>
    */
   public enum TransactionType
   {
      /**
       * With the INSERT action of the transaction operation there can be
       * inserted one or more records into the backend.
       */
      INSERT,

      /**
       * With the DELETE action of the transaction operation there can be
       * deleted specific records defined by a filter expression.
       */
      DELETE,

      /**
       * With the UPDATE action of the transaction operation there can be
       * updated one complete record or just properties of specific records
       * defined by a filter expression.
       */
      UPDATE
   }

   public enum OutputSchema
   {
      DC,
      ISO_19115;

      private OutputSchema()
      {}

      public static URI determineOutputSchema(OutputSchema outputSchema)
         throws MetadataStoreException
      {
         URI schema = null;
         try
         {
            switch(outputSchema)
            {
               case DC:
                  schema = new URI(CSWConstants.CSW_202_NS);
                  break;
               case ISO_19115:
                  schema = new URI(CSWConstants.GMD_NS);
                  break;
            }
         }
         catch(URISyntaxException e)
         {
            throw new MetadataStoreException(e.getMessage());
         }
         return schema;
      }

      public static OutputSchema determineByTypeName(QName typeName)
      {
         String uri = typeName.getNamespaceURI();
         if(uri.equals(CSW_202_NS))
         {
            return DC;
         }
         else if(uri.equals(GMD_NS))
         {
            return ISO_19115;
         }
         return null;
      }
   }
}
