package eu.dnetlib.espas.data.harvest.csw;

import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLEncoder;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Formatter;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.Map;
import java.util.UUID;

import javax.xml.stream.XMLStreamReader;

import org.apache.log4j.Logger;

import eu.dnetlib.espas.data.harvest.csw.common.CSWConstants;
import eu.dnetlib.espas.data.harvest.csw.common.CSWKVPEncodingStrs;
import eu.dnetlib.espas.data.harvest.csw.common.CSWOutputFormatEnum;
import eu.dnetlib.espas.data.harvest.csw.common.CSWOutputSchemaStrs;
import eu.dnetlib.espas.data.harvest.csw.common.CSWRequestTypeEnum;
import eu.dnetlib.espas.data.harvest.csw.common.CSWRequestTypeStrs;
import eu.dnetlib.espas.data.harvest.csw.common.CSWResultSetTypeEnum;
import eu.dnetlib.espas.data.harvest.csw.common.CSWResutlSetTypeStrs;
import eu.dnetlib.espas.data.harvest.csw.common.Version;
import eu.dnetlib.miscutils.datetime.DateUtils;

/**
 * Represents a <Code>GetRecords</Code> request to a CSW.
 */
public class CSWGetRecordsRequest extends AbstractCSWRequest
{
   {  // Class initialization.
      logger = Logger.getLogger(CSWGetRecordsRequest.class);
   }

   // Constants
   private final static int DEFAULT_START_POSITION 	= 1;
   private final static int DEFAULT_MAX_RECORDS		= 10;

   private static final SimpleDateFormat ISO8601FORMAT = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ", Locale.US);

   private static Logger logger = null;
   protected CSWRequestTypeEnum request = null;
   protected CSWResultSetTypeEnum resultType = null;
   protected URI outputSchema = null;
   protected UUID requestID = null;
   protected Date fromDate = null;
   protected Date toDate = null;
   protected String type = null;
   protected int startPosition = DEFAULT_START_POSITION;
   protected int maxRecords = DEFAULT_MAX_RECORDS;

   protected Map<String, String> searchStatus = null;
   protected Map<String, String> searchResults = null;
   protected long numOfHarvestedRecords = 0;

   // The following attribute determines the validity of the request.
   // Any CSW records harvester should check whether the passed CSW GetRecords
   // request is valid. If the state of the passed CSW GetRecords request is invalid
   // no harvesting operation may be launched.
   protected boolean isValid = true;

   public CSWGetRecordsRequest()
   {
      super(null, CSWConstants.CSW_Service_NAME_STR, CSWConstants.VERSION_202, CSWOutputFormatEnum.CSW_OUTPUT_FORMAT_APPLICATION_XML);
      this.initialize(CSWResultSetTypeEnum.HITS, null, null, null, null, null, DEFAULT_START_POSITION, DEFAULT_MAX_RECORDS,
                      new HashMap<String, String>(), new HashMap<String, String>(), 0);
   }

   /**
    * Creates a new {@link CSWGetRecordsRequest} request.
    *
    * @param resultType mode of the response that is requested
    * @param outputSchema indicates the schema of the output (default: http://www.opengis.net/cat/csw/2.0.2)
    * @param startPosition used to specify at which position should be started
    * @param maxRecords defines the maximum number of records that should be returned
    */
   public CSWGetRecordsRequest(String baseURL, String service, Version version, CSWOutputFormatEnum outputFormat,
                                CSWResultSetTypeEnum resultType, URI outputSchema,
                                UUID uuidRequestID, String fromDate, String toDate, String type,
                                int startPosition, int maxRecords,
                                Map<String, String> searchStatus, Map<String, String> searchResults,
                                long numOfHarvestedRecords)
   {
       super(baseURL, service, version, outputFormat);
       this.initialize(resultType, outputSchema, uuidRequestID, fromDate, toDate, type, startPosition, maxRecords,
    		           searchStatus, searchResults, 0);
   }

   public void initialize(CSWResultSetTypeEnum resultType, URI outputSchema,
           UUID requestID, String fromDate, String toDate, String type,
           int startPosition, int maxRecords,
           Map<String, String> searchStatus, Map<String, String> searchResults,
           long numOfHarvestedRecords) {
	   this.setResultType(resultType);
	   this.setOutputSchema(outputSchema);
	   if(null == requestID) {
		   requestID = UUID.randomUUID();
		   logger.debug("Created UUID request id [" + requestID.toString() + "]");
		   this.setRequestID(requestID);
	   }
	   else {
		   logger.debug("Passed UUID request id [" + requestID.toString() + "]");
		   this.setRequestID(requestID);
	   }
	   this.setFromDate(fromDate);
	   this.setToDate(toDate);
	   this.setType(type);
	   this.setStartPosition(startPosition);
	   this.setMaxRecords(maxRecords);
	   this.setSearchStatus(searchStatus);
	   this.setSearchResults(searchResults);
	   this.setNumOfHarvestedRecords(numOfHarvestedRecords);
   }

   private String getGETMethodURLPartForCSWGetRecordsRequest()
   {
      Formatter formatter = null;
      Map<String, String> getParameters = new LinkedHashMap<String, String>();
      try
      {
    	 // Specify the request
    	 if(null == this.request) {
    		 getParameters.put(CSWKVPEncodingStrs.CSW_GET_RECORDS_KVP_KEY_REQUEST_STR, CSWRequestTypeStrs.CSW_REQUEST_TYPE_GET_RECORDS);
    	 }
    	 else {
    		 getParameters.put(CSWKVPEncodingStrs.CSW_GET_RECORDS_KVP_KEY_REQUEST_STR, this.request.getStrName());
    	 }
         // Specify the result set type
         if(null == this.resultType)
         {
            getParameters.put(CSWKVPEncodingStrs.CSW_GET_RECORDS_KVP_KEY_RESULT_SET_TYPE_STR, CSWResutlSetTypeStrs.CSW_RESULT_TYPE_HITS);
         }
         else
         {
            getParameters.put(CSWKVPEncodingStrs.CSW_GET_RECORDS_KVP_KEY_RESULT_SET_TYPE_STR, this.resultType.getStrName());
         }
         // Specify the output schema
         if(null == this.outputSchema || this.outputSchema.toString().equals(""))
         {
            getParameters.put(CSWKVPEncodingStrs.CSW_GET_RECORDS_KVP_KEY_OUTPUT_SCHEMA_STR, (new URI(CSWOutputSchemaStrs.CSW_OUTPUT_SCHEMA_CORE_DEFAULT)).toString());
         }
         else
         {
            getParameters.put(CSWKVPEncodingStrs.CSW_GET_RECORDS_KVP_KEY_OUTPUT_SCHEMA_STR, this.outputSchema.toString());
         }
         // Specify or generate the unique request ID (UUID).
         if(null == this.requestID) {
			// The client code did not pass or discarded the initializer's auto-generated UUID.
			// Thus, auto-generate a new UUID.
			this.requestID = UUID.randomUUID();
			logger.debug("Created UUID request id [" + this.requestID.toString() + "]");
			getParameters.put(CSWKVPEncodingStrs.CSW_GET_RECORDS_KVP_KEY_REQUEST_ID_STR, this.requestID.toString());
         }
         else {
			logger.debug("Passed UUID request id [" + this.requestID.toString() + "]");
			getParameters.put(CSWKVPEncodingStrs.CSW_GET_RECORDS_KVP_KEY_REQUEST_ID_STR, this.requestID.toString());
         }
         // For the time being the only supported constraint language is OGC/CQL.
    	 getParameters.put(CSWKVPEncodingStrs.CSW_GET_CONSTRAINT_LANGUAGE_STR, CSWConstants.CSW_CQL_CONSTRAINT_LANGUAGE);
         // Construct the CQL query.
         if(null == this.fromDate || null == this.type) {
        	 this.isValid = false;
        	 getParameters.put(CSWKVPEncodingStrs.CSW_GET_CQL_CONSTRAINT, "");
         }
         else {
        	 this.isValid = true;
        	 formatter = new Formatter();
	         formatter.format("\"Type like '%s' and Modified between '%s' and '%s'\"",
	        		 			this.type, (new DateUtils(fromDate)).getDateAsISO8601String(),
	        		 			null != this.toDate ? (new DateUtils(toDate)).getDateAsISO8601String() : DateUtils.now_ISO8601());
	         try {
	        	 getParameters.put(CSWKVPEncodingStrs.CSW_GET_CQL_CONSTRAINT, URLEncoder.encode(formatter.toString(), "UTF-8"));
	         }
	         catch(UnsupportedEncodingException e) {
	        	 logger.error("Unsupported encoding exception", e);
	         }
	         formatter.close();
         }
         // Specify the start position
         if(this.startPosition < 0)
         {
            getParameters.put(CSWKVPEncodingStrs.CSW_GET_RECORDS_KVP_KEY_START_POSITION_STR, Integer.toString(1));
         }
         else
         {
            getParameters.put(CSWKVPEncodingStrs.CSW_GET_RECORDS_KVP_KEY_START_POSITION_STR, Integer.toString(this.startPosition));
         }
         // Specify the max records
         if(this.maxRecords < 0)
         {  // The default value is 10.
            getParameters.put(CSWKVPEncodingStrs.CSW_GET_RECORDS_KVP_KEY_MAX_RECORDS_STR, Integer.toString(10));
         }
         else if(0 == this.maxRecords)
         {  // If its value is set to zero, then the behaviour is identical to setting "resultType=hits".
            getParameters.put(CSWKVPEncodingStrs.CSW_GET_RECORDS_KVP_KEY_MAX_RECORDS_STR, Integer.toString(this.maxRecords));
            getParameters.put(CSWKVPEncodingStrs.CSW_GET_RECORDS_KVP_KEY_RESULT_SET_TYPE_STR, CSWResutlSetTypeStrs.CSW_RESULT_TYPE_HITS);
         }
         else
         {
            getParameters.put(CSWKVPEncodingStrs.CSW_GET_RECORDS_KVP_KEY_MAX_RECORDS_STR, Integer.toString(this.maxRecords));
         }
         formatter = new Formatter();
         for(Iterator<Map.Entry<String, String>> it = getParameters.entrySet().iterator(); it.hasNext();)
         {
            Map.Entry<String, String> keyValuePair = it.next();
            formatter.format("%s=%s", keyValuePair.getKey(), keyValuePair.getValue());
            if(it.hasNext())
            {
               formatter.format("&");
            }
         }
      }
      catch (URISyntaxException e)
      {
         e.printStackTrace();
      }
      return formatter.toString();
   }

   @Override
   public String toString()
   {
      return this.getURLStr();
   }

   @Override
   public URL getURL()
   {
      try
      {
         if(null != this.getURLStr() && !this.getURLStr().equals(""))
         {
            return new URL(this.getURLStr());
         }
         else
         {
            return null;
         }
      }
      catch(MalformedURLException e) {
         logger.error("Malofrmed URL exception", e);
      }
      return null;
   }

   @Override
   public String getURLStr() {
      if(null != super.getURLStr() && !super.getURLStr().equals("")) {
    	  logger.debug("Passed URL from the AbstractCSWRequest " + super.getURLStr());
    	  logger.debug("URL component produced " + this.getGETMethodURLPartForCSWGetRecordsRequest());
    	  return super.getURLStr() + "&" + this.getGETMethodURLPartForCSWGetRecordsRequest();
      }
      return null;
   }

   public CSWRequestTypeEnum getRequest() {
	   return request;
   }

   public void setRequest(CSWRequestTypeEnum request) {
	   this.request = request;
   }

   public URI getOutputSchema() {
      return outputSchema;
   }

   public void setOutputSchema(URI outputSchema)
   {
      this.outputSchema = outputSchema;
   }

   public UUID getRequestID() {
	   return requestID;
   }

   public void setRequestID(UUID requestID) {
	   this.requestID = requestID;
   }

   public void setOutputSchema(String outputSchema)
   {
      try
      {
      this.outputSchema = new URI(outputSchema);
      }
      catch(URISyntaxException e)
      {
         logger.error("URI syntax exception", e);
      }
   }

   public synchronized Date checkISO8601Compliance(String dateStr) {
	   if(null == dateStr) {
		   return null;
	   }
	   try {
		   return ISO8601FORMAT.parse(dateStr);
	   }
	   catch (ParseException e) {
		   this.isValid = false;
		   logger.error("\"" + dateStr + "\"" + " is not compliant to ISO 8601.", e);
	   }
	   return null;
   }

   public Date getFromDate() {
	   return fromDate;
   }

   public void setFromDate(String fromDate) {

	   this.fromDate = this.checkISO8601Compliance(fromDate);
   }

   public void setFromDate(Date fromDate) {
	   this.fromDate = fromDate;
   }

   public Date getToDate() {
	   return toDate;
   }

   public void setToDate(String toDate) {
	   this.toDate = this.checkISO8601Compliance(toDate);
   }

   public void setToDate(Date toDate) {
	   this.toDate = toDate;
   }

   public String getType() {
      return this.type;
   }

   public void setType(String type) {
      this.type = type;
   }

   public int getStartPosition()
   {
      return startPosition;
   }

   public void setStartPosition(int startPosition)
   {
      this.startPosition = startPosition;
   }

   public int getMaxRecords()
   {
      return maxRecords;
   }

   public void setMaxRecords(int maxRecords)
   {
      this.maxRecords = maxRecords;
   }

   public CSWResultSetTypeEnum getResultType()
   {
      return resultType;
   }

   public void setResultType(CSWResultSetTypeEnum resultType)
   {
      this.resultType = resultType;
   }

   public Map<String, String> getSearchStatus() {
	   return searchStatus;
   }

   public void setSearchStatus(XMLStreamReader xmlStreamReader) {
	   int attributeListSize = xmlStreamReader.getAttributeCount();
	   for(int i = 0; i < attributeListSize; i++) {
		   this.searchStatus.put(xmlStreamReader.getAttributeLocalName(i), xmlStreamReader.getAttributeValue(i));
	   }
   }

   public void setSearchStatus(Map<String, String> searchStatus) {
	   this.searchStatus = searchStatus;
   }

   public Map<String, String> getSearchResults()
   {
      return this.searchResults;
   }

   public void setSearchResults(XMLStreamReader xmlStreamReader)
   {
      int attributeListSize = xmlStreamReader.getAttributeCount();
      for(int i = 0; i < attributeListSize; i++)
      {
         this.searchResults.put(xmlStreamReader.getAttributeLocalName(i), xmlStreamReader.getAttributeValue(i));
      }
   }

   public void setSearchResults(Map<String, String> searchResults)
   {
      this.searchResults = searchResults;
   }

	public long getNumOfHarvestedRecords() {
		return numOfHarvestedRecords;
	}

	public void setNumOfHarvestedRecords(long numOfHarvestedRecords) {
		this.numOfHarvestedRecords = numOfHarvestedRecords;
	}

	public void incrementNumOfHarvestedRecordsByOne() {
		this.numOfHarvestedRecords += 1;
	}

	public boolean isValid() {
		return isValid;
	}

   public void setValid(boolean isValid) {
	   this.isValid = isValid;
   }
}
