package eu.dnetlib.data.oai.store.conf;

import java.util.List;
import java.util.concurrent.Callable;

import javax.annotation.Resource;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import eu.dnetlib.data.information.oai.publisher.conf.OAIConfigurationReader;
import eu.dnetlib.data.information.oai.publisher.info.MDFInfo;
import eu.dnetlib.data.information.oai.publisher.info.SetInfo;
import eu.dnetlib.data.oai.store.mongo.MongoPublisherStore;
import eu.dnetlib.data.oai.store.mongo.MongoPublisherStoreDAO;
import eu.dnetlib.data.oai.store.sets.MongoSetCollection;

public class OAISetHelper {

	private static final Log log = LogFactory.getLog(OAISetHelper.class); // NOPMD by marko on 11/24/08 5:02 PM

	@Resource(name = "oaiConfigurationExistReader")
	private OAIConfigurationReader configurationReader;
	@Resource
	private MongoPublisherStoreDAO mongoPublisherStoreDAO;
	@Resource
	private MongoSetCollection mongoSetCollection;

	/*
	 * Helper method to count elements in OAI sets.
	 */
	public void updateAllCounts(final MDFInfo mdFormat) {
		updateTotalCount(mdFormat);
		List<SetInfo> sets = mongoSetCollection.getAllSets();
		this.updateCountForSets(sets, mdFormat);
	}

	public void updateConfiguredSetsCount(final MDFInfo mdFormat) {
		updateTotalCount(mdFormat);
		List<SetInfo> sets = mongoSetCollection.getConfiguredSets();
		this.updateCountForSets(sets, mdFormat);
	}

	protected void updateTotalCount(final MDFInfo mdFormat) {
		log.info("Ensuring indices on OAI sets mongo collection . . .");
		this.mongoSetCollection.ensureIndexes();
		String format = mdFormat.getSourceFormatName();
		String layout = mdFormat.getSourceFormatLayout();
		String interpretation = mdFormat.getSourceFormatInterpretation();
		String sourceKey = format + "-" + layout + "-" + interpretation;
		log.info("Updating count for OAI store " + sourceKey + " via metadata prefix " + mdFormat.getPrefix());
		log.debug(mdFormat);
		MongoPublisherStore store = this.mongoPublisherStoreDAO.getStoreFor(mdFormat.getPrefix());
		if (store == null) {
			log.info("Can't count elements for not yet created store (" + mdFormat + ").");
		} else {
			String baseQuery = mdFormat.getBaseQuery();
			log.info("Counting total for " + mdFormat.getPrefix() + " with query: " + baseQuery);
			int total = store.count(baseQuery);
			mongoSetCollection.updateCounts("ALL", mdFormat.getPrefix(), total);
			log.info("Counts updated for " + mdFormat.getPrefix());
		}
	}

	protected void updateCountForSets(final List<SetInfo> oaiSets, final MDFInfo mdFormat) {
		MongoPublisherStore store = this.mongoPublisherStoreDAO.getStoreFor(mdFormat.getPrefix());
		if (store == null) {
			log.info("Can't count elements for not yet created store (" + mdFormat + ").");
		} else {
			String baseQuery = mdFormat.getBaseQuery();
			boolean hasBaseQuery = !StringUtils.isBlank(baseQuery);
			for (SetInfo setInfo : oaiSets) {
				String setQuery = setInfo.getQuery();
				if (hasBaseQuery) {
					setQuery += " AND " + baseQuery;
				}
				log.info("Counting total for " + mdFormat.getPrefix() + " set " + setInfo + " with query: " + setQuery);
				int setCount = store.count(setQuery);
				mongoSetCollection.updateCounts(setInfo.getSetSpec(), mdFormat.getPrefix(), setCount);
			}
		}
	}

	// ////////////////////////////////////////////////////////////////////////////////////////////
	/*
	 * Helper methods to upsert OAI sets based on the configuration.
	 */
	/**
	 * Update OAI sets based on the configuration read with the given configurationReader.
	 * 
	 * Used by {@link OAISetNotificationHandler}
	 * 
	 * @param configReader
	 */
	public void updateSetsFromConfig(final OAIConfigurationReader configReader) {
		log.debug("*****Dropping and re-creating configuration sets******");
		this.mongoSetCollection.dropConfigurationSets();
		List<SetInfo> oaiConfigSets = configReader.getSets();
		for (SetInfo setInfo : oaiConfigSets) {
			this.mongoSetCollection.upsertSet(setInfo, true);
		}
		List<MDFInfo> metadataFormats = configReader.getMetadataFormatInfo();
		for (MDFInfo mdFormat : metadataFormats) {
			this.updateConfiguredSetsCount(mdFormat);
		}
		log.info("Sets updated succesfully");
	}

	public void updateSetsFromConfig() {
		this.updateSetsFromConfig(this.configurationReader);
	}

	public void updateSetsFromConfig(final Callable<?> callback, final Callable<?> failCallback) {
		try {
			this.updateSetsFromConfig(this.configurationReader);
			executeCallback(callback);
		} catch (Exception e) {
			log.error(e);
			executeCallback(failCallback);
		}
	}

	protected void executeCallback(final Callable<?> callback) {
		if (callback != null) {
			try {
				callback.call();
			} catch (Exception e) {
				log.error("Error executing callback", e);
			}
		}
	}

	public OAIConfigurationReader getConfigurationReader() {
		return configurationReader;
	}

	public void setConfigurationReader(final OAIConfigurationReader configurationReader) {
		this.configurationReader = configurationReader;
	}

	public MongoPublisherStoreDAO getMongoPublisherStoreDAO() {
		return mongoPublisherStoreDAO;
	}

	public void setMongoPublisherStoreDAO(final MongoPublisherStoreDAO mongoPublisherStoreDAO) {
		this.mongoPublisherStoreDAO = mongoPublisherStoreDAO;
	}

	public MongoSetCollection getMongoSetCollection() {
		return mongoSetCollection;
	}

	public void setMongoSetCollection(final MongoSetCollection mongoSetCollection) {
		this.mongoSetCollection = mongoSetCollection;
	}

}
