package eu.dnetlib.data.information.publisher;

import java.io.StringReader;
import java.io.StringWriter;
import java.util.List;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import javax.xml.ws.wsaddressing.W3CEndpointReference;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;

import com.google.common.base.Joiner;

import eu.dnetlib.data.information.publisher.rmi.PublisherService;
import eu.dnetlib.data.provision.index.rmi.IndexService;
import eu.dnetlib.data.provision.index.rmi.IndexServiceException;
import eu.dnetlib.enabling.resultset.rmi.ResultSetException;
import eu.dnetlib.enabling.resultset.rmi.ResultSetService;
import eu.dnetlib.enabling.tools.AbstractBaseService;
import eu.dnetlib.enabling.tools.ServiceLocator;
import eu.dnetlib.enabling.tools.ServiceResolver;
import eu.dnetlib.miscutils.collections.MappedCollection;
import eu.dnetlib.miscutils.functional.UnaryFunction;

/**
 * Dummy implementation of the PublisherService.
 * 
 * TODO: once the Harvesting Instances are working we can implement the rest of this class.
 * 
 * @author marko
 * 
 */
public class PublisherServiceImpl extends AbstractBaseService implements PublisherService {

	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(PublisherServiceImpl.class); // NOPMD by marko on 11/24/08 5:02 PM

	/**
	 * index service locator.
	 */
	private ServiceLocator<IndexService> indexLocator;

	/**
	 * service resolver.
	 */
	private ServiceResolver serviceResolver;

	/**
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.data.information.publisher.rmi.PublisherService#getResourceById(java.lang.String, java.lang.String)
	 */
	@Override
	public String getResourceById(final String id, final String format, final String layout, final String interpretation) {
		W3CEndpointReference epr;
		try {
			epr = indexLocator.getService().lookup("objIdentifier=\"" + id + "\"", format, layout, interpretation);

			final ResultSetService resultSet = serviceResolver.getService(ResultSetService.class, epr);
			final String rsId = serviceResolver.getResourceIdentifier(epr);

			if (resultSet.getNumberOfElements(rsId) > 0) {
				try {
					return unwrap(resultSet.getResult(rsId, 1, 1, "waiting").get(0));
				} catch (final TransformerFactoryConfigurationError e) {
					log.warn("transformer factory configuration", e);
					return null;
				} catch (final TransformerException e) {
					log.warn("transformer exception", e);
					return null;
				}
			}
		} catch (final IndexServiceException e) {
			log.warn("cannot find record for id " + id + " and format " + format + "and layout " + layout + " and interpretation " + interpretation, e);
			return null;
		} catch (final ResultSetException e) {
			log.warn("cannot fetch record for id " + id + " and format " + format + "and layout " + layout + " and interpretation " + interpretation, e);
			return null;
		}

		log.warn("no matching record for id " + id + " and format " + format + "and layout " + layout + " and interpretation " + interpretation);
		return null;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.data.information.publisher.rmi.PublisherService#getResourcesByIds(java.util.List, java.lang.String)
	 */
	@Override
	public W3CEndpointReference getResourcesByIds(final List<String> ids, final String format, final String layout, final String interpretation) {
		final String query = Joiner.on(" or ").join(new MappedCollection<String, String>(ids, new UnaryFunction<String, String>() {

			@Override
			public String evaluate(final String id) {
				return "objidentifier=\"" + id + "\"";
			}
		}));

		try {
			return unwrapResultSet(indexLocator.getService().lookup(query, format, layout, interpretation));
		} catch (final IndexServiceException e) {
			log.warn("cannot fetch record for ids " + ids + " and format " + format, e);
			return null;
		}
	}

	/**
	 * Unwrap a whole resultset.
	 * 
	 * @param indexLookup
	 *            index resultset
	 * @return publisher resultset
	 */
	public W3CEndpointReference unwrapResultSet(final W3CEndpointReference indexLookup) {
		return indexLookup;
	}

	/**
	 * Unwrap an index record extracting the contained record.
	 * 
	 * @param string
	 * @return
	 * @throws TransformerFactoryConfigurationError
	 *             could happen
	 * @throws TransformerException
	 *             could happen
	 */
	public String unwrap(final String source) throws TransformerFactoryConfigurationError, TransformerException {
		final StringWriter result = new StringWriter();
		final Transformer transformer = TransformerFactory.newInstance().newTransformer(new StreamSource(getClass().getResourceAsStream("unwrap.xsl")));
		transformer.transform(new StreamSource(new StringReader(source)), new StreamResult(result));
		return result.toString();
	}

	public ServiceLocator<IndexService> getIndexLocator() {
		return indexLocator;
	}

	@Required
	public void setIndexLocator(final ServiceLocator<IndexService> indexLocator) {
		this.indexLocator = indexLocator;
	}

	public ServiceResolver getServiceResolver() {
		return serviceResolver;
	}

	@Required
	public void setServiceResolver(final ServiceResolver serviceResolver) {
		this.serviceResolver = serviceResolver;
	}

}
