package eu.dnetlib.data.utility.cleaner;

import java.io.StringReader;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Required;

import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import eu.dnetlib.data.utility.cleaner.rmi.CleanerException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.tools.ServiceLocator;

public class CleaningRuleFactory {

	private ServiceLocator<ISLookUpService> lookupLocator;

	public CleaningRule obtainCleaningRule(final String ruleId) throws CleanerException {
		try {
			final String prof = lookupLocator.getService().getResourceProfileByQuery(
					"/RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value='" + ruleId + "' or .//CLEANER_NAME='" + ruleId + "']//CONFIGURATION");

			final SAXReader reader = new SAXReader();
			final Document doc = reader.read(new StringReader(prof));

			final CleaningRule rule = new CleaningRule();

			for (Object o : doc.selectNodes("//RULE")) {
				final Element node = (Element) o;

				final String xpath = node.valueOf("@xpath");
				final String vocabularies = node.valueOf("@vocabularies");
				final String groovyRule = node.valueOf("@groovy");
				final String strict = node.valueOf("@strict");

				final XPATHCleaningRule xpathRule;
				if ((vocabularies != null) && (vocabularies.length() > 0)) {
					final Set<String> list = Sets.newHashSet(Splitter.on(",").omitEmptyStrings().trimResults().split(vocabularies));
					xpathRule = new VocabularyRule(list, lookupLocator);
				} else {
					xpathRule = new GroovyRule(groovyRule);
				}
				xpathRule.setXpath(xpath);
				xpathRule.setStrict("true".equals(strict));
				rule.getXpathRules().add(xpathRule);
			}
			return rule;
		} catch (Exception e) {
			throw new CleanerException("Error obtaing cleaner rule " + ruleId, e);
		}
	}

	public List<String> getRuleIds() throws CleanerException {
		try {
			final HashSet<String> response = new HashSet<String>();

			final List<String> list = lookupLocator.getService().quickSearchProfile("//CLEANER_NAME");
			if (list != null) {
				response.addAll(list);
			}

			return Lists.newArrayList(response);
		} catch (ISLookUpException e) {
			throw new CleanerException("Error obtaining IDs of cleaner DSs", e);
		}
	}

	@Required
	public void setLookupLocator(final ServiceLocator<ISLookUpService> lookupLocator) {
		this.lookupLocator = lookupLocator;
	}

	public ServiceLocator<ISLookUpService> getLookupLocator() {
		return lookupLocator;
	}
}
