package eu.dnetlib.data.mdstore.modular;

import java.util.concurrent.Callable;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;

import eu.dnetlib.data.mdstore.modular.connector.MDStore;
import eu.dnetlib.data.mdstore.modular.connector.MDStoreDao;
import eu.dnetlib.enabling.is.registry.rmi.ISRegistryService;
import eu.dnetlib.enabling.resultset.client.ResultSetClientFactory;
import eu.dnetlib.enabling.tools.ServiceLocator;
import eu.dnetlib.miscutils.datetime.DateUtils;

public class MDStoreFeeder {
	private MDStoreDao dao;

	private ResultSetClientFactory resultSetClientFactory;

	private ServiceLocator<ISRegistryService> registryLocator;

	private boolean syncFeed = true;
	
	static final Log log = LogFactory.getLog(MDStoreFeeder.class);
	
	public void feed(final String mdId, final String rsEpr, final String storingType) {
		feed(mdId, rsEpr, storingType, syncFeed, null, null);
	}

	public void feed(final String mdId, final String rsEpr, final String storingType, final boolean sync, final Callable<?> callback, final Callable<?> failCallback) {
		log.info("Start feeding mdstore " + mdId + " with epr " + rsEpr);
		
		final Thread feederThread = new Thread(new Runnable() {
			@Override
			public void run() {
				try {
					final MDStore mdstore = dao.getMDStore(mdId);
	
					final Iterable<String> records = resultSetClientFactory.getClient(rsEpr);
	
					final boolean incremental = !"REFRESH".equals(storingType);
	
					if (!incremental)
						mdstore.truncate();
	
					final int size = mdstore.feed(records, incremental);
	
					touch(mdId, size);
	
					log.info("Stop feeding mdstore " + mdId + " - new size: " + size);
					
					if (callback != null) {
						try {
							callback.call();
						} catch (Exception e) {
							log.error("Error executing callback", e);
						}
					}
				} catch (Throwable e) {
					log.error("Error in feeding thread", e);
					if (failCallback != null) {
						try {
							failCallback.call();
						} catch (Exception e1) {
							log.error("Error executing failCallback", e);
						}
					}
				}
			}
		});
		
		feederThread.start();
		
		if(sync) {
			try {
				feederThread.join();
			} catch (InterruptedException e) {
				throw new IllegalStateException(e);
			}
		}
	}

	/**
	 * Sets the last modified date in the profile.
	 * 
	 * @param mdId
	 */
	public void touch(final String mdId, final int size) {
		try {
			final String now = DateUtils.now_ISO8601();

			final String mdstoreXUpdate = "for $x in //RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value = '" + mdId + "']"
					+ "return update value $x//LAST_STORAGE_DATE with '" + now + "'";

			registryLocator.getService().executeXUpdate(mdstoreXUpdate);

			touchSize(mdId, size);
		} catch (final Exception e) {
			throw new IllegalStateException(e);
		}
	}

	public void touchSize(final String mdId, final int size) {
		try {
			final String mdstoreNumberXUpdate = "for $x in //RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value = '" + mdId + "']"
					+ "return update value $x//NUMBER_OF_RECORDS with '" + size + "'";

			registryLocator.getService().executeXUpdate(mdstoreNumberXUpdate);
		} catch (final Exception e) {
			throw new IllegalStateException(e);
		}
	}

	public MDStoreDao getDao() {
		return dao;
	}

	@Required
	public void setDao(final MDStoreDao dao) {
		this.dao = dao;
	}

	public ResultSetClientFactory getResultSetClientFactory() {
		return resultSetClientFactory;
	}

	@Required
	public void setResultSetClientFactory(final ResultSetClientFactory resultSetClientFactory) {
		this.resultSetClientFactory = resultSetClientFactory;
	}

	public ServiceLocator<ISRegistryService> getRegistryLocator() {
		return registryLocator;
	}

	@Required
	public void setRegistryLocator(final ServiceLocator<ISRegistryService> registryLocator) {
		this.registryLocator = registryLocator;
	}

	public boolean isSyncFeed() {
		return syncFeed;
	}

	public void setSyncFeed(boolean syncFeed) {
		this.syncFeed = syncFeed;
	}
}
