package eu.dnetlib.data.collector.plugins.projects.wt;

import java.net.URL;
import java.util.Iterator;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.PriorityBlockingQueue;

import eu.dnetlib.data.collector.rmi.CollectorServiceException;
import eu.dnetlib.data.collector.rmi.CollectorServiceRuntimeException;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;

public class WTProjectsIterator implements Iterator<String>, Iterable<String> {

	private static final Log log = LogFactory.getLog(WTProjectsIterator.class); // NOPMD by marko on 11/24/08 5:02 PM

	private static String queryURL = "http://www.ebi.ac.uk/europepmc/GristAPI/rest/get/query=ga:%22Wellcome%20Trust%22&resultType=core";
	private int pageNumber = 0;
	private SAXReader reader = new SAXReader();
	private Queue<String> projects = new PriorityBlockingQueue<String>();
	private boolean morePages = true;
	//The following is for debug only
	private int nextCounter = 0;

	@Override
	public boolean hasNext() {
		try {
			fillProjectListIfNeeded();
		} catch (CollectorServiceException e) {
			throw new CollectorServiceRuntimeException(e);
		}
		return !projects.isEmpty();
	}

	@Override
	public String next() {
		nextCounter++;
		log.debug(String.format("Calling next %s times. projects queue has %s elements", nextCounter, projects.size()));
		try {
			fillProjectListIfNeeded();
			return projects.poll();
		} catch (CollectorServiceException e) {
			throw new CollectorServiceRuntimeException(e);
		}
	}

	private boolean fillProjectListIfNeeded() throws CollectorServiceException {
		if (morePages && projects.isEmpty()) {
			String resultPage = getNextPage();
			Document doc = null;
			try {
				doc = reader.read(IOUtils.toInputStream(resultPage));
				List<Element> records = doc.selectNodes("//RecordList/Record");
				if (records != null && !records.isEmpty()) {
					log.debug(String.format("Found %s projects in page", records.size()));
					for (Element p : records) {
						projects.add(p.asXML());
					}
					return true;
				} else {
					log.info("No more projects to read at page nr. " + pageNumber);
					morePages = false;
					return false;
				}
			} catch (DocumentException e) {
				throw new CollectorServiceException(e);
			}
		} else return false;
	}

	private String getNextPage() {
		pageNumber++;
		try {
			URL pageUrl = new URL(queryURL + "&page=" + pageNumber);
			log.debug("Getting page at: " + pageUrl.toString());
			return IOUtils.toString(pageUrl);
		} catch (Exception e) {
			throw new CollectorServiceRuntimeException("Error on page " + pageNumber, e);
		}
	}

	@Override
	public void remove() {
		throw new UnsupportedOperationException();
	}

	@Override
	public Iterator<String> iterator() {
		return this;
	}

	public int getPageNumber() {
		return pageNumber;
	}

	public void setPageNumber(final int pageNumber) {
		this.pageNumber = pageNumber;
	}
}
