package eu.dnetlib.data.download.worker;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.base.Function;
import com.google.common.base.Joiner;

import eu.dnetlib.data.download.DownloadReport;
import eu.dnetlib.data.download.DownloadReportMap;
import eu.dnetlib.data.download.DownloadServiceImpl;
import eu.dnetlib.data.download.rmi.DownloadItem;
import eu.dnetlib.data.objectstore.modular.ObjectStoreRecord;
import eu.dnetlib.data.objectstore.modular.connector.ObjectStore;
import eu.dnetlib.data.objectstore.rmi.ObjectStoreFile;
import eu.dnetlib.data.objectstore.rmi.ObjectStoreServiceException;
import eu.dnetlib.data.objectstore.rmi.Protocols;

/**
 * The Class DownloadWorker is a worker responsible to download the data into the object store.
 */
public class DownloadWorker implements Callable<DownloadReportMap> {

	/**
	 * The Constant log.
	 */
	private static final Log log = LogFactory.getLog(DownloadWorker.class);

	/**
	 * The queue.
	 */
	private BlockingQueue<String> queue = null;

	/**
	 * The object store.
	 */
	private ObjectStore objectStore = null;

	/**
	 * The protocol.
	 */
	private Protocols protocol;

	/**
	 * The mime type.
	 */
	private String mimeType;

	private Function<String, DownloadItem> converter;

	/**
	 * Instantiates a new download worker.
	 *
	 * @param queue
	 *            the queue
	 * @param objectStore
	 *            the object store
	 * @param protocol
	 *            the protocol
	 * @param mimeType
	 *            the mime type
	 */
	public DownloadWorker(final BlockingQueue<String> queue, final ObjectStore objectStore, final Protocols protocol, final String mimeType,
			final Function<String, DownloadItem> converter) {
		this.setConverter(converter);
		this.setQueue(queue);
		this.setObjectStore(objectStore);
		this.setMimeType(mimeType);
		this.setProtocol(protocol);
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see java.lang.Runnable#run()
	 */
	@Override
	public DownloadReportMap call() throws Exception {
		final DownloadReportMap report = new DownloadReportMap();
		try {

			String takedObject = queue.take();
			DownloadItem di = getConverter().apply(takedObject);
			if (log.isDebugEnabled()) {
				log.debug(takedObject);
			}

			while (DownloadServiceImpl.END_QUEUE != di) {

				log.debug(Thread.currentThread().getId()+": Reading  "+takedObject);

				if (di == null) {
					log.info("the current download item is Null, skipping");
				}
				if ((di != null) && (di.getUrl() != null) && (di.getUrl().length() != 0) && !checkIfExists(di)) {
					try {
						final URL toDownload = followURL(new URL(di.getUrl()), report, di);
						final ObjectStoreRecord record = new ObjectStoreRecord();
						final ObjectStoreFile metadata = new ObjectStoreFile();
						metadata.setObjectID(di.getFileName());
						metadata.setMetadataRelatedID(di.getIdItemMetadata());
						metadata.setAccessProtocol(this.protocol);
						metadata.setMimeType(this.mimeType);
						metadata.setDownloadedURL(di.getOriginalUrl());
						record.setFileMetadata(metadata);
						record.setInputStream(toDownload.openStream());
						objectStore.feedObjectRecord(record);
						report.addDowload();
						log.debug("Saved object " + metadata.toJSON());
					} catch (final Throwable e) {
						log.error("An error occur inside the loop: ",e);
						// reportException(report, di, e);
					}
				}
				takedObject = queue.take();
				log.debug(Thread.currentThread().getId()+": Next Object From Queue  "+takedObject);
				di = getConverter().apply(takedObject);
				if (log.isDebugEnabled()) {
					log.debug(takedObject);
				}
			}
			queue.put(DownloadServiceImpl.END_QUEUE_STRING);
		} catch (final Exception e) {
			log.error("An error occured : " + Joiner.on("\tat ").join(e.getStackTrace()));
			report.setStatus(false);
			return report;
		}

		log.info("CLOSED THREAD " + Thread.currentThread().getId());
		report.setStatus(true);
		return report;

	}

	private void reportException(final DownloadReportMap report, final DownloadItem di, final Exception e) {
		final String className = e.getClass().getName();
		if (!report.containsKey(className)) {
			final DownloadReport dr = new DownloadReport();
			dr.setStackTrace(Joiner.on("\tat ").join(e.getStackTrace()));
			dr.setRecordId(di.getIdItemMetadata());
			report.put(className, dr);
		} else {
			report.get(className).incrementError();
		}
	}

	private URL followURL(final URL inputURL, final DownloadReportMap report, final DownloadItem di) throws IOException {

		final String ptrcl = inputURL.getProtocol();
		if (ptrcl.startsWith("file")) {
			log.debug("the protocol is File, returning " + inputURL);
			return inputURL;
		}

		HttpURLConnection conn;

		conn = (HttpURLConnection) inputURL.openConnection();
		conn.setInstanceFollowRedirects(true);  // you still need to handle redirect manully.
		HttpURLConnection.setFollowRedirects(true);
		String location = inputURL.toString();
		if ((conn.getResponseCode() >= 300) && (conn.getResponseCode() < 400)) {
			location = conn.getHeaderFields().get("Location").get(0);
			conn.disconnect();
		}

		if (!location.equals(inputURL.toString())) return new URL(location);
		return inputURL;


	}

	private boolean checkIfExists(final DownloadItem di) {
		try {
			return objectStore.deliverObject(di.getFileName()) != null;
		} catch (final ObjectStoreServiceException e) {
			log.debug(e.getMessage());
			return false;
		}
	}

	/**
	 * Sets the object store.
	 *
	 * @param objectStore
	 *            the objectStore to set
	 */
	public void setObjectStore(final ObjectStore objectStore) {
		this.objectStore = objectStore;
	}

	/**
	 * Sets the queue.
	 *
	 * @param queue
	 *            the queue to set
	 */
	public void setQueue(final BlockingQueue<String> queue) {
		this.queue = queue;
	}

	/**
	 * Sets the protocol.
	 *
	 * @param protocol
	 *            the protocol to set
	 */
	public void setProtocol(final Protocols protocol) {
		this.protocol = protocol;
	}

	/**
	 * Sets the mime type.
	 *
	 * @param mimeType
	 *            the mimeType to set
	 */
	public void setMimeType(final String mimeType) {
		this.mimeType = mimeType;
	}

	public Function<String, DownloadItem> getConverter() {
		return converter;
	}

	public void setConverter(final Function<String, DownloadItem> converter) {
		this.converter = converter;
	}
}
