package eu.dnetlib.msro.eagle.workflows.nodes.transform.tmid;

import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamConstants;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;
import org.springframework.core.io.FileSystemResource;

import eu.dnetlib.miscutils.functional.UnaryFunction;
import eu.dnetlib.miscutils.functional.hash.Hashing;

public class TrismegistosInjectionUnaryFunction implements UnaryFunction<String, String> {

	private SAXReader reader;

	private static Map<String, List<String>> tmIdToLocalIdsMap;
	private static Map<String, String> localIdToTmIdMap;

	public TrismegistosInjectionUnaryFunction(final String tmIdListPath) throws XMLStreamException, IOException {
		this.reader = new SAXReader();
		tmIdToLocalIdsMap = new HashMap<String, List<String>>();
		localIdToTmIdMap = new HashMap<String, String>();

		FileSystemResource tmIds = new FileSystemResource(tmIdListPath);
		XMLInputFactory factory = XMLInputFactory.newInstance();
		XMLStreamReader reader = factory.createXMLStreamReader(tmIds.getInputStream());

		boolean isSafeToGetNextXmlElement = true;
		while (isSafeToGetNextXmlElement) {
			if (reader.getEventType() == XMLStreamConstants.START_ELEMENT) {
				if ("tmid".equals(reader.getLocalName())) {
					parseTmEvent(reader);
				}
			}
			if (reader.hasNext()) {
				reader.next();
			} else {
				isSafeToGetNextXmlElement = false;
				break;
			}
		}
	}

	private void parseTmEvent(final XMLStreamReader reader) throws XMLStreamException {
		String tmId = reader.getAttributeValue(null, "id");
		List<String> localIds = new ArrayList<String>();

		boolean isSafeToGetNextXmlElement = true;
		while (isSafeToGetNextXmlElement) {
			if (reader.getEventType() == XMLStreamConstants.END_ELEMENT) {
				break;
			} else {
				if (reader.getEventType() == XMLStreamConstants.START_ELEMENT) {
					if ("link".equals(reader.getLocalName())) {
						String localId = parseLinkEvent(reader);
						localIdToTmIdMap.put(localId, tmId);
						localIds.add(localId);
					}
				}
			}
			if (reader.hasNext()) {
				reader.next();
			} else {
				isSafeToGetNextXmlElement = false;
				break;
			}
		}
		tmIdToLocalIdsMap.put(tmId, localIds);
	}

	private String parseLinkEvent(final XMLStreamReader reader) throws XMLStreamException {
		String cp = reader.getAttributeValue(null, "cp");
		String val = "";
		boolean isSafeToGetNextXmlElement = true;
		while (isSafeToGetNextXmlElement) {
			if (reader.getEventType() == XMLStreamConstants.END_ELEMENT) {
				break;
			} else if (reader.getEventType() == XMLStreamConstants.CHARACTERS) {
				val = reader.getText();
			}
			if (reader.hasNext()) {
				reader.next();
			} else {
				isSafeToGetNextXmlElement = false;
				break;
			}
		}
		return cp + "::" + val;
	}

	@Override
	public String evaluate(final String input) {
		try {
			Document doc = reader.read(new StringReader(input));
			Node entityType = doc.selectSingleNode("//*[local-name()='entityType']");

			// TMid injection on Artifacts/Inscriptions
			if ("artifact".equals(entityType.getText())) {
				Node recordSourceInfo = doc.selectSingleNode("//*[local-name()='eagleObject']/*[local-name()='recordSourceInfo']");
				String localId =
						recordSourceInfo.selectSingleNode("@providerAcronym").getText() + "::" + recordSourceInfo.selectSingleNode("./text()").getText();
				String tmId = localIdToTmIdMap.get(localId);
				if (tmId != null) {
					// TMid found! Prepare for injection..
					List<String> alternateIds = tmIdToLocalIdsMap.get(tmId);
					Element inscription = (Element) doc.selectSingleNode("//*[local-name()='inscription']");
					Element injectedHasTmId = prepareInjectedElement(tmId, alternateIds, localId);
					replaceElement(inscription, (Element) inscription.selectSingleNode("hasTmId"), injectedHasTmId);
					return doc.asXML();
				} else {
					// There is no TM ID for this localID. Just return the input string here..
					return input;
				}
			}

			// TMid injection on other items with rel to Artifacts
			if ("visual".equals(entityType.getText()) || "documental".equals(entityType.getText())) {
				Node recordSourceInfo = doc.selectSingleNode("//*[local-name()='hasArtifact']/*[local-name()='recordSourceInfo']");
				String localId =
						recordSourceInfo.selectSingleNode("@providerAcronym").getText() + "::" + recordSourceInfo.selectSingleNode("./text()").getText();
				String tmId = localIdToTmIdMap.get(localId);
				if (tmId != null) {
					// TMid found! Prepare for injection..
					List<String> alternateIds = tmIdToLocalIdsMap.get(tmId);
					Element inscription = (Element) doc.selectSingleNode("//*[local-name()='hasArtifact']");
					Element injectedHasTmId = prepareInjectedElement(tmId, alternateIds, localId);
					replaceElement(inscription, (Element) inscription.selectSingleNode("hasTmId"), injectedHasTmId);
					return doc.asXML();
				} else {
					// There is no TM ID for this localID. Just return the input string here..
					return input;
				}
			}
		} catch (DocumentException e) {
			throw new IllegalStateException("Problems with Trismegistos injection", e);
		}

		return input;
	}

	private Element prepareInjectedElement(final String tmId, final List<String> alternateIds, final String localId) {
		Document fragment = DocumentHelper.createDocument();
		Element hasTmId = fragment.addElement("hasTmId");
		hasTmId.addElement("tmId").addText(tmId);
		for (String alternateId : alternateIds) {
			if (!localId.equals(alternateId)) {
				String[] tokens = alternateId.split("::");
				hasTmId.addElement("alternateId").addAttribute("providerAcronym", tokens[0]).addAttribute("localId", tokens[1])
						.setText(tokens[0] + "::" + Hashing.md5(tokens[1]) + "::artifact");
			}
		}
		return (Element) fragment.selectSingleNode("hasTmId");
	}

	private void replaceElement(final Element parent, final Element oldElement, final Element newElement) {
		List parentContent = parent.content();
		int index = parentContent.indexOf(oldElement);
		parentContent.set(index, newElement);
	}

}
