package eu.dnetlib.data.provision.pdf;

import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.Resource;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.pdfbox.exceptions.COSVisitorException;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPage;
import org.apache.pdfbox.pdmodel.edit.PDPageContentStream;
import org.apache.pdfbox.pdmodel.font.PDFont;
import org.apache.pdfbox.pdmodel.font.PDTrueTypeFont;
import org.dom4j.DocumentException;
import org.dom4j.io.SAXReader;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ClassPathResource;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import eu.dnetlib.functionality.index.client.IndexClient;
import eu.dnetlib.functionality.index.client.IndexClientException;
import eu.dnetlib.functionality.index.client.IndexClientFactory;

@Controller
public class PdfMaterializerController {

	private static final int MARGIN = 72;

	private static final float LEADING_SPACE = 1.5f;

	private static final int FONT_SIZE = 12;

	private static final Log log = LogFactory.getLog(PdfMaterializerController.class);

	@Resource
	private IndexClientFactory indexClientFactory;

	@Value("${eagle.provision.pdf.index.format}")
	private String indexFormat;

	@Value("${eagle.provision.pdf.index.layout}")
	private String indexLayout;

	@Value("${eagle.provision.pdf.index.interpretation}")
	private String indexInterpretation;

	@Value("${eagle.provision.pdf.index.id.field}")
	private String indexIdField;

	@Value("${eagle.provision.pdf.index.text.xpath}")
	private String textXpath;

	private ClassPathResource trueType = new ClassPathResource("/eu/dnetlib/data/provision/pdf/lmroman10-regular.ttf");

	@RequestMapping("/pdfs/{id}")
	public void generatePdfFromId(final HttpServletRequest request, final HttpServletResponse response, @PathVariable final String id) throws IOException,
	COSVisitorException, IndexClientException, DocumentException {
		log.info("Materializing pdf");

		List<String> arrayList = new ArrayList<String>();
		arrayList.add(getTextToPrint(id));

		writeText(arrayList, response.getOutputStream());

	}

	public void writeText(final List<String> arrayList, final ServletOutputStream outputStream) throws IOException,
	COSVisitorException {
		PDDocument document = new PDDocument();
		PDPage page = new PDPage();
		// PDFont font = PDType1Font.HELVETICA;
		PDFont font = PDTrueTypeFont.loadTTF(document, trueType.getInputStream());
		int positionX = 30;
		int positionY = 750;

		document.addPage(page);

		// Start a new content stream
		PDPageContentStream contentStream = new PDPageContentStream(document, page);

		// Define a text content stream using the selected font, moving the cursor and drawing the text in arrayList
		for (int i = 0; i < arrayList.size(); i++) {
			String text = arrayList.get(i);
			String[] tmpText = splitString(text);
			for (int k = 0; k < tmpText.length; k++) {
				contentStream.beginText();
				contentStream.setFont(font, 12);
				contentStream.moveTextPositionByAmount(positionX, positionY);
				contentStream.drawString(tmpText[k]);
				contentStream.endText();
				positionY = positionY - 20;
			}
			contentStream.setLineWidth((float) 0.25);
		}

		// Make sure that the content stream is closed:
		contentStream.close();
		document.save(outputStream);
		document.close();
	}

	public String[] splitString(final String text) {
		/*
		 * pdfBox doesnt support linebreaks. Therefore, following steps are requierd to automatically put linebreaks in the pdf 1) split
		 * each word in string that has to be linefeded and put them into an array of string, e.g. String [] parts 2) create an array of
		 * stringbuffer with (textlength/(number of characters in a line)), e.g. 280/70=5 >> we need 5 linebreaks! 3) put the parts into the
		 * stringbuffer[i], until the limit of maximum number of characters in a line is allowed, 4) loop until stringbuffer.length <
		 * linebreaks
		 */
		int linebreaks = text.length() / 40; // how many linebreaks do I need?
		String[] newText = new String[linebreaks + 1];
		String tmpText = text;
		String[] parts = tmpText.split(" "); // save each word into an array-element

		// split each word in String into a an array of String text.
		StringBuffer[] stringBuffer = new StringBuffer[linebreaks + 1]; // StringBuffer is necessary because of manipulating text
		int i = 0; // initialize counter
		int totalTextLength = 0;
		for (int k = 0; k < linebreaks + 1; k++) {
			stringBuffer[k] = new StringBuffer();
			while (true) {
				if (i >= parts.length)
				{
					break; // avoid NullPointerException
				}
				totalTextLength = totalTextLength + parts[i].length(); // count each word in String
				if (totalTextLength > 40)
				{
					break; // put each word in a stringbuffer until string length is >80
				}
				stringBuffer[k].append(parts[i]);
				stringBuffer[k].append(" ");
				i++;
			}
			// reset counter, save linebreaked text into the array, finally convert it to a string
			totalTextLength = 0;
			newText[k] = stringBuffer[k].toString();
		}
		return newText;
	}

	private String getTextToPrint(final String id) throws IndexClientException, DocumentException {
		IndexClient indexClient = indexClientFactory.getClient(indexFormat, indexLayout, indexInterpretation);
		List<String> results = indexClient.lookup(indexIdField + "=" + id, null, 0, 10).getRecords();
		if (results.size() != 1) {
			throw new IndexClientException("The query was supposed to return EXACTLY ONE record (not 0, nor 2 or more)");
		}
		else {
			String record = results.get(0);
			SAXReader reader = new SAXReader();
			return reader.read(new StringReader(record)).valueOf(textXpath);
		}
	}

	@ExceptionHandler(IndexClientException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public @ResponseBody String notFoundHandler(final Exception e) {
		e.printStackTrace();
		return e.getMessage();
	}

	@ExceptionHandler({ DocumentException.class, IOException.class, COSVisitorException.class })
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	public @ResponseBody String internaErrorHandler(final Exception e) {
		e.printStackTrace();
		return e.getMessage();
	}

	public String getIndexFormat() {
		return indexFormat;
	}

	public void setIndexFormat(final String indexFormat) {
		this.indexFormat = indexFormat;
	}

	public String getIndexLayout() {
		return indexLayout;
	}

	public void setIndexLayout(final String indexLayout) {
		this.indexLayout = indexLayout;
	}

	public String getIndexInterpretation() {
		return indexInterpretation;
	}

	public void setIndexInterpretation(final String indexInterpretation) {
		this.indexInterpretation = indexInterpretation;
	}

	public String getIndexIdField() {
		return indexIdField;
	}

	public void setIndexIdField(final String indexIdField) {
		this.indexIdField = indexIdField;
	}

	public String getTextXpath() {
		return textXpath;
	}

	public void setTextXpath(final String textXpath) {
		this.textXpath = textXpath;
	}
}
