package eu.dnetlib.msro.eagle.workflows.nodes.backlink.translations;

import java.io.StringReader;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import eu.dnetlib.miscutils.functional.UnaryFunction;

public class TranslationBacklinkUnaryFunction implements UnaryFunction<String, String> {

	private static final Log log = LogFactory.getLog(TranslationBacklinkUnaryFunction.class);

	final private SAXReader reader;
	final private Map<String, List<TranslationInfo>> eagleIdToTranslationsInfo;

	public TranslationBacklinkUnaryFunction(final Iterator<String> translationIterator) {
		reader = new SAXReader();
		eagleIdToTranslationsInfo = Maps.newHashMap();

		while (translationIterator.hasNext()) {
			String translationRecord = translationIterator.next();
			try {
				processTranslationRecord(translationRecord, eagleIdToTranslationsInfo);
			} catch (DocumentException e) {
				log.error("Error during the pre-processing of MediaWiki translations", e);
				throw new RuntimeException(e);
			}
		}
	}

	private void processTranslationRecord(final String translationRecord, final Map<String, List<TranslationInfo>> eagleIdToTranslations)
			throws DocumentException {
		Document doc = reader.read(new StringReader(translationRecord));
		List<Node> eagleIdList = doc.selectNodes("//*[local-name()='hasArtifact']/*[local-name()='dnetResourceIdentifier']");
		List<TranslationInfo> translationInfoList;
		for (Node e : eagleIdList) {
			// Fetch or create from scratch che List of EagleId
			if (eagleIdToTranslations.containsKey(e.getText())) {
				translationInfoList = eagleIdToTranslations.get(e.getText());
			} else {
				translationInfoList = Lists.newArrayList();
			}
			// Prepare TranslationInfo to append
			TranslationInfo info = new TranslationInfo();
			info.setDnetResourceIdentifier(doc.valueOf("//*[local-name()='eagleObject']/*[local-name()='dnetResourceIdentifier']"));
			info.setProviderAcronym(doc.valueOf("//*[local-name()='eagleObject']/*[local-name()='recordSourceInfo']/@providerAcronym"));
			info.setProviderName(doc.valueOf("//*[local-name()='eagleObject']/*[local-name()='recordSourceInfo']/@providerName"));
			info.setLandingPage(doc.valueOf("//*[local-name()='eagleObject']/*[local-name()='recordSourceInfo']/@landingPage"));
			info.setLocalId(doc.valueOf("//*[local-name()='eagleObject']/*[local-name()='recordSourceInfo']"));
			info.setText(doc.valueOf("//*[local-name()='eagleObject']/*[local-name()='text']"));
			info.setLang(doc.valueOf("//*[local-name()='eagleObject']/*[local-name()='text']/@lang"));
			translationInfoList.add(info);
			eagleIdToTranslations.put(e.getText(), translationInfoList);
		}
	}

	@Override
	public String evaluate(final String record) {
		try {
			Document doc = reader.read(new StringReader(record));
			String eagleObjId = doc.valueOf("//*[local-name()='eagleObject']/*[local-name()='dnetResourceIdentifier']");
			if (!eagleIdToTranslationsInfo.containsKey(eagleObjId)) return record;

			Element inscription = (Element) doc.selectSingleNode("//*[local-name()='inscription']");
			if (inscription == null) return record;

			List<TranslationInfo> translationsInfoList = eagleIdToTranslationsInfo.get(eagleObjId);
			for (TranslationInfo translationInfo : translationsInfoList) {
				String fragment = HasTranslationFragment.generateFragment(translationInfo);
				Document fragmentDoc = new SAXReader().read(new StringReader(fragment));
				inscription.add(fragmentDoc.getRootElement());
			}
			return doc.asXML();
		} catch (DocumentException e) {
			throw new RuntimeException(e);
		}
	}

}
