package eu.dnetlib.data.mapreduce.hbase.oai;

import java.io.IOException;
import java.net.UnknownHostException;
import java.text.ParseException;
import java.util.Collection;
import java.util.Date;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.io.NullWritable;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Mapper;
import org.apache.solr.common.util.DateUtil;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.mongodb.BasicDBObject;
import com.mongodb.DB;
import com.mongodb.DBCollection;
import com.mongodb.DBObject;
import com.mongodb.Mongo;

import eu.dnetlib.data.mapreduce.JobParams;
import eu.dnetlib.data.mapreduce.hbase.oai.config.OAIConfiguration;
import eu.dnetlib.data.mapreduce.hbase.oai.config.OAIConfigurationReader;
import eu.dnetlib.data.mapreduce.hbase.oai.config.OAIConfigurationStringReader;
import eu.dnetlib.data.mapreduce.hbase.oai.utils.MongoSetCollection;
import eu.dnetlib.data.mapreduce.hbase.oai.utils.PublisherField;
import eu.dnetlib.data.mapreduce.hbase.oai.utils.RecordFieldsExtractor;
import eu.dnetlib.data.mapreduce.hbase.oai.utils.SetInfo;
import eu.dnetlib.miscutils.functional.xml.IndentXmlString;

public class OaiFeedMapper extends Mapper<Text, Text, NullWritable, NullWritable> {

	enum RecordStatus {
		NEW, UPDATED, UNCHANGED;
	}

	private DBCollection collection;
	private DBCollection discardedCollection;
	private final boolean upsert = true;
	private final boolean multi = false;
	private OAIConfigurationStringReader oaiConfigurationReader;
	private OAIConfiguration oaiConfiguration;

	private Date feedDate;

	private MongoSetCollection mongoSetCollection;

	private RecordFieldsExtractor extractor;

	// these are set in the setup
	private String format;
	private String interpretation;
	private String layout;
	private Map<String, PublisherField> fieldsToIndex = Maps.newHashMap();

	@Override
	protected void setup(final Context context) throws UnknownHostException {

		String host = context.getConfiguration().get("services.publisher.oai.host");
		String port = context.getConfiguration().get("services.publisher.oai.port");
		String db = context.getConfiguration().get("services.publisher.oai.db");
		String collectionName = context.getConfiguration().get("services.publisher.oai.collection");

		System.out.println("Mongodb client params");
		System.out.println("host: " + host);
		System.out.println("port: " + port);
		System.out.println("db: " + db);
		System.out.println("collection: " + collectionName);

		String[] formatLayoutInterp = collectionName.split("-");
		format = formatLayoutInterp[0];
		layout = formatLayoutInterp[1];
		interpretation = formatLayoutInterp[2];

		String oaiConfigurationProfile = context.getConfiguration().get("oaiConfiguration");
		System.out.println("oaiConfiguration:\n" + IndentXmlString.apply(oaiConfigurationProfile));
		oaiConfigurationReader = new OAIConfigurationStringReader(oaiConfigurationProfile);
		oaiConfiguration = oaiConfigurationReader.getOaiConfiguration();

		System.out.println("parsed configuration:" + oaiConfiguration.toString());

		Mongo mongo = new Mongo(host, Integer.parseInt(port));
		DB mongoDB = mongo.getDB(db);
		collection = mongoDB.getCollection(collectionName);
		discardedCollection = mongoDB.getCollection("discarded-" + collectionName);
		mongoSetCollection = new MongoSetCollection(mongoDB);
		Collection<PublisherField> indexFields = oaiConfiguration.getFieldsFor(format, layout, interpretation);
		extractor = new RecordFieldsExtractor(Lists.newArrayList(indexFields));
		for (PublisherField field : indexFields) {
			fieldsToIndex.put(field.getFieldName(), field);
		}

		String feedDateString = context.getConfiguration().get(JobParams.OAI_FEED_DATE);
		try {
			feedDate = DateUtil.parseDate(feedDateString);
		} catch (ParseException e) {
			e.printStackTrace(System.err);
			throw new RuntimeException(e);
		}
	}

	@Override
	protected void map(final Text key, final Text value, final Context context) throws IOException, InterruptedException {

		String recordKey = key.toString();
		String recordBody = value.toString();
		if (StringUtils.isBlank(recordBody)) {
			discard(context, recordKey, recordBody, "blank body");
		} else {

			Multimap<String, String> recordFields = extractor.extractFields(recordBody);
			String id = "";
			String oaiID = "";
			if (recordFields == null) {
				context.getCounter("oai", "invalid").increment(1);
			} else if (recordFields.containsKey(OAIConfigurationReader.ID_FIELD)) {
				id = recordFields.get(OAIConfigurationReader.ID_FIELD).iterator().next();
				oaiID = getOAIIdentifier(id);
				handleRecord(getRecordStatus(oaiID, recordBody), context, oaiID, recordBody, recordFields);
			} else {
				discard(context, recordKey, recordBody, "missing " + OAIConfigurationReader.ID_FIELD);
			}
		}
	}

	private void handleRecord(final RecordStatus status,
			final Context context,
			final String oaiID,
			final String record,
			final Multimap<String, String> recordProperties) {
		DBObject oldObj = new BasicDBObject(OAIConfigurationReader.ID_FIELD, oaiID);
		DBObject obj = null;
		String counterReason = "";
		switch (status) {
		case NEW:
			obj = this.createBasicObject(oaiID, record, recordProperties, context);
			obj.put(OAIConfigurationReader.LAST_COLLECTION_DATE_FIELD, feedDate);
			obj.put(OAIConfigurationReader.DATESTAMP_FIELD, feedDate);
			obj.put(OAIConfigurationReader.UPDATED_FIELD, false);
			counterReason = "new record";
			// this.upsertSets(recordProperties.get(OAIConfigurationReader.SET_FIELD));
			break;
		case UPDATED:
			obj = this.createBasicObject(oaiID, record, recordProperties, context);
			obj.put(OAIConfigurationReader.LAST_COLLECTION_DATE_FIELD, feedDate);
			obj.put(OAIConfigurationReader.DATESTAMP_FIELD, feedDate);
			obj.put(OAIConfigurationReader.UPDATED_FIELD, true);
			counterReason = "updated record";
			// this.upsertSets(recordProperties.get(OAIConfigurationReader.SET_FIELD));
			break;
		case UNCHANGED:
			// I just want to set the last collection date, without changing/resetting any of the other dates
			obj = new BasicDBObject("$set", new BasicDBObject(OAIConfigurationReader.LAST_COLLECTION_DATE_FIELD, feedDate));
			counterReason = "unchanged record";
			break;
		}
		// collection.update(oldObj, obj, upsert, multi);
		collection.insert(obj);
		context.getCounter("oai", counterReason).increment(1);
		context.getCounter("oai", "total").increment(1);
	}

	private void upsertSets(final Iterable<String> setNames) {
		// feed the list of sets, if any
		if (setNames != null) {
			for (String setName : setNames) {
				if (StringUtils.isNotBlank(setName)) {
					final SetInfo set = new SetInfo();
					String setSpec = this.mongoSetCollection.normalizeSetSpec(setName);
					set.setSetSpec(setSpec);
					set.setSetName(setName);
					set.setSetDescription("This set contains metadata records whose provenance is " + setName);
					set.setEnabled(true);
					String query = OAIConfigurationReader.SET_FIELD + " = \"" + setSpec + "\"";;
					set.setQuery(query);
					this.mongoSetCollection.upsertSet(set, false);
				}
			}
		}
	}

	private void discard(final Context context, final String recordKey, final String recordBody, final String reason) {
		context.getCounter("oai", reason).increment(1);
		discardedCollection.insert(new BasicDBObject("id", recordKey).append(OAIConfigurationReader.BODY_FIELD, recordBody));
	}

	private String getOAIIdentifier(final String id) {
		return oaiConfiguration.getIdScheme() + ":" + oaiConfiguration.getIdNamespace() + ":" + id;
	}

	private RecordStatus getRecordStatus(final String oaiId, final String recordBody) {
		if (isNewRecord(oaiId)) return RecordStatus.NEW;
		if (isChanged(oaiId, recordBody)) return RecordStatus.UPDATED;
		else return RecordStatus.UNCHANGED;
	}

	private boolean isNewRecord(final String oaiIdentifier) {
		// return this.collection.findOne(new BasicDBObject(OAIConfigurationReader.ID_FIELD, oaiIdentifier)) == null;
		return true;
	}

	/**
	 * Tells if record has changed w.r.t. the version we currently have in the store.
	 * 
	 * <p>
	 * FIXME: see MongoPublisherStore and its implementation of the method. For now we just consider that a record has changed and return
	 * true.
	 * </p>
	 * 
	 * @param oaiID
	 *            oai identifier of the record
	 * @param recordBody
	 *            the record body
	 * @return true
	 */
	private boolean isChanged(final String oaiID, final String recordBody) {
		return true;
	}

	protected DBObject createBasicObject(final String oaiID, final String record, final Multimap<String, String> recordProperties, final Context context) {
		DBObject obj = new BasicDBObject();
		for (final String key : recordProperties.keySet()) {
			if (key.equals(OAIConfigurationReader.ID_FIELD)) {
				obj.put(key, oaiID);
			} else {
				Collection<String> values = recordProperties.get(key);
				if (key.equals(OAIConfigurationReader.SET_FIELD)) {

					Iterable<String> setSpecs = Iterables.transform(values, new Function<String, String>() {

						@Override
						public String apply(final String s) {
							return mongoSetCollection.normalizeSetSpec(s);
						}

					});
					obj.put(key, setSpecs);
				} else {
					PublisherField keyField = fieldsToIndex.get(key);
					if (keyField == null) {
						context.getCounter("oai", key + " found for record but not in configuration. Assuming it is repeatable.").increment(1);
					}
					// let's check if the key is the name of a repeatable field or not
					if ((keyField != null) && !keyField.isRepeatable()) {
						if ((values != null) && !values.isEmpty()) {
							obj.put(key, values.iterator().next());
						}
					} else {
						obj.put(key, values);
					}
				}
			}
		}
		obj.put(OAIConfigurationReader.BODY_FIELD, record);
		obj.put(OAIConfigurationReader.DELETED_FIELD, false);
		return obj;
	}

	@Override
	protected void cleanup(final Context context) throws IOException, InterruptedException {

		super.cleanup(context);
	}

	public DBCollection getCollection() {
		return collection;
	}

	public void setCollection(final DBCollection collection) {
		this.collection = collection;
	}

	public DBCollection getDiscardedCollection() {
		return discardedCollection;
	}

	public void setDiscardedCollection(final DBCollection discardedCollection) {
		this.discardedCollection = discardedCollection;
	}

	public OAIConfigurationStringReader getOaiConfigurationReader() {
		return oaiConfigurationReader;
	}

	public void setOaiConfigurationReader(final OAIConfigurationStringReader oaiConfigurationReader) {
		this.oaiConfigurationReader = oaiConfigurationReader;
	}

	public OAIConfiguration getOaiConfiguration() {
		return oaiConfiguration;
	}

	public void setOaiConfiguration(final OAIConfiguration oaiConfiguration) {
		this.oaiConfiguration = oaiConfiguration;
	}

	public Date getFeedDate() {
		return feedDate;
	}

	public void setFeedDate(final Date feedDate) {
		this.feedDate = feedDate;
	}

	public MongoSetCollection getMongoSetCollection() {
		return mongoSetCollection;
	}

	public void setMongoSetCollection(final MongoSetCollection mongoSetCollection) {
		this.mongoSetCollection = mongoSetCollection;
	}

}
